////////////////////////////////////////////////////////////////////////////////
//  Source File :  Room.cpp
//  Author      :  Daniel K. Vinther Wolf + Janus Bo Andersen
//  Date        :  02 may 2019
//  Last edit by:  Janus Bo Andersen
//  Last edited :  08 May 2019
//  Version     :  1.0
//
//  Description :  Rooms controller, Class Definition
///////////////////////////////////////80///////////////////////////////////////

/*
Changelog:
Date        By          Change
--------    --------    ------------------------------------------------------
20190503    Janus       Use CRLF line endings instead of endl
20190504    Janus       Add include guards
20190504    Janus       Add constructor with room name
20190504    Janus       Add destructor
20190504    Janus       Add private state to room and a getter (encapsulation)
20190504    Janus       Encapsulate the latch pins and include in constructor
20190504    Janus       Change <iostream> to "mbed.h" to avoid memory+lib issues
20190504    Janus       Change cout to printf for smaller memory footprint
20190505    Janus       Add name and state getters, and a state setter.
20190505    Janus       Exchange the array for a C++ vector (push / erase)
20190506    Janus       Implement check if person is first/last in/out of room
20190506    Janus       Include <algorithm> to use find
20190508    Janus       Include <map> to build hashmap of users and checkn/out times
20190508    Janus       Implemented ActivityTime map to "debounce" as user in the door

*/

#include "mbed.h"
#include <vector>
#include <algorithm>
#include <map>

#ifndef ROOM_H
#define ROOM_H

#define PULSEWIDTH_MS 100 //for the latches

class Room {
public:
    //constructor
    Room(char * roomName, PinName onSwitch, PinName offSwitch, time_t gate_wait) {
        _name = roomName;
        _state = false;
        _setPin = new DigitalOut(onSwitch);
        _rstPin = new DigitalOut(offSwitch);
        _GATE_TIME_S = gate_wait;
    }
    
    //destructor (clean up and free)
    ~Room(void) {
        delete this->_setPin;
        delete this->_rstPin;    
    }
    
    //force room off
    void reset(void) {
        this->setState(false);
    }


    void RoomCall(unsigned KeyID) {
        time_t time_now = time(NULL);
        vector<uint32_t>::iterator it;
        
        printf("RFID reading at room %s for user %8X at time %d.\r\n", this->getName(), KeyID, time_now);

        //check if KeyID is in vector already
        it = find(this->KeyVec.begin(), this->KeyVec.end(), KeyID);
        
        //Check whether person is coming or going
        if ( (it != this->KeyVec.end()) ) {  //Person is not new            
        
            //Check whether enough wait time has elapsed
            if  ( (time_now - this->ActivityTime[KeyID]) >= this->_GATE_TIME_S ) {
                //found in list -- person is leaving IF gate time has elapsed
                
                //Take person off the list for this room
                this->KeyVec.erase(it);
                //Update the activity time
                this->ActivityTime[KeyID] = time(NULL);
                //Display info
                printf("Checking out user %8X from %s at time %d...\r\n", KeyID, this->getName(), time_now);
                
                //if this is the last person leaving the room, turn lights off
                if (this->KeyVec.size() == 0) {
                    printf("Last out of the room: And there was darkness...\r\n");
                    this->setState(false);
                }
                
            } else if ( (time_now - this->ActivityTime[KeyID]) < this->_GATE_TIME_S ) {
                //update activity time for "debouncing"
                this->ActivityTime[KeyID] = time(NULL);
                printf("You are blocking the door-way. Updated your activity time to %d.\r\n", this->ActivityTime[KeyID]);
            }
            
        } else if ( it == this->KeyVec.end() ) { //person is new
            //person was not in the list already, so person is entering/new
            
            if ((time_now - this->ActivityTime[KeyID]) >= this->_GATE_TIME_S) {
                //put person on the list
                this->KeyVec.push_back(KeyID);
                this->ActivityTime[KeyID] = time(NULL);
                
                printf("Checking in user %8X in %s at time %d...\r\n", KeyID, this->getName(), time_now);
                
                //if this is the first person entering the room, turn lights on
                if (this->KeyVec.size() == 1) {
                    printf("First in the room: Let there be light!\r\n");
                    this->setState(true);
                }
            } else if ((time_now - this->ActivityTime[KeyID]) < this->_GATE_TIME_S) {
                //update activity time for "debouncing"
                this->ActivityTime[KeyID] = time(NULL);
                printf("You are blocking the door-way. Updated your activity time to %d.\r\n", this->ActivityTime[KeyID]);
            }
        }
    
    } //end RoomCall
    
    //state getter
    bool getState(void) {
        return this->_state;
    }
    
    //name getter
    char * getName(void) {
        return _name;
    }
    

private:
    vector <uint32_t> KeyVec; //vector of keys for people in the room
    map <uint32_t, time_t> ActivityTime; //map for keys and their last recorded activity
    
    bool _state;
    char * _name;
    DigitalOut * _setPin;
    DigitalOut * _rstPin;
    time_t _GATE_TIME_S;  //control the time a user must wait befire entering/leaving room (in seconds)
    
    //set/reset state and send external signal (100ms pulses to the latch)
    //the latches are active high
    void setState(bool s) {
          this->_state = s;
          *(this->_setPin) = this->_state;
          *(this->_rstPin) = !(this->_state);
          wait_ms(PULSEWIDTH_MS); //wait 100ms
          *(this->_setPin) = false; //set back to low
          *(this->_rstPin) = false;
    }

}; // End Room (Class)

#endif //end include guard