/*
 * File       : main.cpp
 * License    : 
 * Version    : 0.96
 * Author     : Janus Bo Andersen (JA67494)
 * Last edit  : Janus
 * Created on : 27 Apr 2019
 * Last change: 06 May 2019
 * Description: Dev. branch for E2PRO2 (smart house / RFID-house)
 *            : Communicate with RFID readers to sense movement in the house
 *            : Determine user location and turn rooms/house off/on accordingly
 * Structure  : Uses SPI protocol to communicate with the RFIDs.
 *            : Uses U(S)ART to communicate with connected computer/terminal.
 *            : Leverages a state machine-like Room class for control per room.
 * Adapted for: STM32 Nucleo L432KC and NXP KL25Z (macro settings).
 * Changelog  : 27 Apr 2019: Created demo program
                02 May 2019: Update pin mappings for FRDM-KL25Z Board-
                02 May 2019: Added set and reset for a latched circuit-
                03 May 2019: Upgrade to RTOS 5.
                03 May 2019: Run two RFIDs simultaneosuly (changed library setup
                             by decr. SPI freq. and adding reset managers).
                03 May 2019: Include Room class (room.hpp) and tests.
                05 May 2019: Downgrade to bare-metal.
                             Include testrunner header (homebrewed).
                             Fully utilize the room class.
                05 May 2019: Expanded the MRFC522 library to directly give UIDs
                             (4-byte little-endian delivered in uint32_t).
                06 May 2019: Move pin settings to separate header.
                06 May 2019: Passes test cases 1-6.
                08 May 2019: Introduced a GAIT_WAIT time for minimum time spent
                             inside/outside a room before new activity is registered
                             
 NOTICE LINE 42! Set/check precompiler constant below controlling pinmap (KL25Z vs 432KC)
 NOTICE LINE 43! Set/check whether tests should be run
 */

//mbed libraries
#include "mbed.h"
#include "MFRC522.h"

//our own libraries and classes
#define COMPILING_FOR_KL25Z 1 //Set this to 1 to compile with KL25Z pins
#define RUN_TEST 0
#include "pinsettings.hpp"
#include "room.hpp"

//speed setting
#define RFID_WAIT_MS 1000
#define GATE_WAIT_TIME_S 0

//Serial connection to terminal for output
Serial pc(SERIAL_TX, SERIAL_RX);

//Include test fixtures and testrunner
#include "tests.hpp"

int main(void) {
    
    set_time(0);
    
    printf("====== TEAM 2 HOUSE ======\r\n");
    
    #if RUN_TEST
    //Running diagnotic, unit tests and regression tests...
    runTests();
    #endif

    //On-board LED: Blinks when a card is read
    DigitalOut LedGreen(LED1);

    //Define the house and rooms
    Room r0("Team 2 Wooden House", FRONTDOOR_SET, FRONTDOOR_RST, GATE_WAIT_TIME_S);
    Room r1("The secret room where the Monster cans are stored", ROOM1DOOR_SET, ROOM1DOOR_RST, GATE_WAIT_TIME_S);

    //Connect two MFRC522 readers
    MFRC522 rfid0(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS1, MF_RESET1);
    MFRC522 rfid1(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS2, MF_RESET2);

    //Initialize the RC522 chips
    rfid0.PCD_Init();
    rfid1.PCD_Init();

    //Reset the latches connected to rooms (ensure off)
    printf("Resetting house...\r\n"); 
    r0.reset();
    r1.reset();
    printf("Done.\r\n");
    
    //Prepare other useful variables
    uint32_t read_key = 0;
    
    //Begin monitoring for RFIDs
    printf("Starting the Team 2 Pro 2 House RFID-monitor.\r\n");

    while (true) {
        LedGreen = 1;

        if (rfid0.PICC_IsNewCardPresent()) {
            if (rfid0.PICC_ReadCardSerial()) {

                LedGreen = 0; //signal that reading is going on            

                read_key = rfid0.UID_4_byte(); //get the UID that's read
                r0.RoomCall(read_key); //send the UID to the room controller

                wait_ms(RFID_WAIT_MS); //block for a while to prevent flicker
            } //end check for a readable card
        } //end rfid0 check for a card
       
        if ( rfid1.PICC_IsNewCardPresent()) {
            if (rfid1.PICC_ReadCardSerial()) {
    
                LedGreen = 0; //signal that reading is going on            

                read_key = rfid1.UID_4_byte();
                r1.RoomCall(read_key);
                
                wait_ms(RFID_WAIT_MS);                
            }
        }
        
    } //end infinite while
} //end main