/*
 * File       : main.cpp
 * License    : 
 * Version    : 0.92
 * Author     : Janus Bo Andersen (JA67494)
 * Last edit  : Janus
 * Created on : 27 Apr 2019
 * Last change: 03 May 2019
 * Description: Demo program for Mifare RC-522 RFID reader/writer chip. Will:
 *            : Communicate with RFID readers, transmit card numbers to terminal
 *            : and toggle a digital pin(s) upon reading a card.
 * Structure  : Uses SPI protocol to communicate with the RFIDs.
 *            : Uses U(S)ART to communicate with connected computer/terminal.
 * Adapted for: STM32 Nucleo L432KC and NXP KL25Z (macro settings).
 * Changelog  : 27 Apr 2019: Created demo program
              : 02 May 2019: Update pin mappings for FRDM-KL25Z Board
              : 02 May 2019: Added set and reset for a latched circuit
              : 03 May 2019: Upgrade to RTOS 5.
              : 03 May 2019: Run two RFIDs simultaneosuly (changed library setup
              :              by decr. SPI freq. and adding reset managers).
 
 Resources:
 https://github.com/armmbed/mbed-os
 https://os.mbed.com/docs/mbed-os/v5.9/tutorials/migrating.html
 
 */

/*Connect as follows:

Power supply MUST BE 3.3V and GND to the respective pins.

  RFID pins          ->   Board pinName   L432KC       FRDM-KL25Z
  ---------------------------------------------------------------------
  
  =======================  BUS   ======================================
  RFID MISO = pin4   ->   SPI_MISO        PA_6=D12     PTD3
  RFID MOSI = pin3   ->   SPI_MOSI        PA_7=D11     PTD2
  RFID SCK  = pin2   ->   SPI_SCK         PA_5=D13     PTD1
  ======================= RFID 1 ====================================== 
  RFID SDA  = pin1   ->   SPI_CS          PB_6=D10     PTD0
  RFID RST  = pin7   ->                   PA_9=D8      PTA13
  RFID IRQ  = pin5   ->                   -------- OPEN --------
  ======================= RFID 2 ======================================
  RFID SDA  = pin1   ->   SPI_CS          PA_12=D2     PTD5     
  RFID RST  = pin7   ->                   PB_1=D6      PTC9
  RFID IRQ  = pin5   ->                   -------- OPEN --------
  =====================================================================

Connecting the "HOUSE"
  Location      Circuit latch function    L432KC       FRDM-KL25Z
  ---------------------------------------------------------------------
  Front door    Set                       A1           A1
  Front door    Reset                     A2           A2
  Room 1 door   Set                       A3           A3
  Room 1 door   Reset                     A4           A4
  =====================================================================
  
  Mapping:
  Front door = rfid1
  Room 1 door = rfid2
  
*/     

#include "mbed.h"
#include "MFRC522.h"

#define COMPILING_FOR_KL25Z 1 //Set this to 1 to compile with KL25Z pins

#if COMPILING_FOR_KL25Z
  // FRDM KL25Z pins
  
  //USB communication
  #define SERIAL_TX PTE0  //UART0 USB
  #define SERIAL_RX PTE1  //UART0 USB
  
  //BUS
  #define SPI_MISO  PTD3
  #define SPI_MOSI  PTD2
  #define SPI_SCK   PTD1
  
  //RFID 1
  #define SPI_CS1   PTD0  //Chip select can be any digital pin
  #define MF_RESET1 PTA13
  
  //RFID 2
  #define SPI_CS2   PTD5
  #define MF_RESET2 PTC9

  //House circuit connections
  #define FRONTDOOR_SET A1
  #define FRONTDOOR_RST A2
  #define ROOM1DOOR_SET A3
  #define ROOM1DOOR_RST A4

#else
  // STM32 L432KC pins
  
  //USB communication
  #define SERIAL_TX USBTX  //UART0 USB
  #define SERIAL_RX USBRX  //UART0 USB
  
  //BUS
  #define SPI_MISO  D12
  #define SPI_MOSI  D11
  #define SPI_SCK   D13
  
  //RFID 1
  #define SPI_CS1   D10
  #define MF_RESET1  D8
  
  //RFID 2
  #define SPI_CS2    D2
  #define MF_RESET2  D6

  //House circuit connections
  #define FRONTDOOR_SET A1
  #define FRONTDOOR_RST A2
  #define ROOM1DOOR_SET A3
  #define ROOM1DOOR_RST A4

#endif

DigitalOut LedGreen(LED1); //Blinks when a card is read

//Define the latches
DigitalOut r0_set(FRONTDOOR_SET);
DigitalOut r0_rst(FRONTDOOR_RST);
DigitalOut r1_set(ROOM1DOOR_SET);
DigitalOut r1_rst(ROOM1DOOR_RST);

bool r0_state;
bool r1_state;

//Serial connection to terminal for output
Serial pc(SERIAL_TX, SERIAL_RX);

//MFRC522 readers
MFRC522 rfid1(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS1, MF_RESET1);
MFRC522 rfid2(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS2, MF_RESET2);

int main(void) {
    printf("Starting the Team 2 Pro 2 House...\n\r");

    // Init. RC522 chips
    rfid1.PCD_Init();
    rfid2.PCD_Init();

    printf("Resetting latches...\n\r");
    r0_set = 0;
    r0_rst = 1;
    r0_state = false;
    r1_set = 0;
    r1_rst = 1;
    r1_state = false;
    wait_ms(500);
    r0_rst = 0;
    r1_rst = 0;
    printf("Done.\n\r");
    
    printf("Starting RFID monitoring...\n\r");

    while (true) {
        LedGreen = 1;

        if (rfid1.PICC_IsNewCardPresent()) {
            if (rfid1.PICC_ReadCardSerial()) {

                LedGreen = 0;

                printf("Front door: ");
                for (uint8_t i = 0; i < rfid1.uid.size; i++) {
                    printf(" %02X", rfid1.uid.uidByte[i]);
                }
                printf("\n\r");
                
                //pulse room 0 (front door) latch
                r0_state = !r0_state;
                r0_set = r0_state;
                r0_rst = !r0_state;
                wait_ms(100);
                r0_set = 0;
                r0_rst = 0;
                
                wait_ms(500);
            }
        } //end rfid1
       

        if ( rfid2.PICC_IsNewCardPresent()) {
            if (rfid2.PICC_ReadCardSerial()) {
    
                LedGreen = 0;
    
                printf("Room 1: ");
                for (uint8_t i = 0; i < rfid2.uid.size; i++) {
                    printf(" %02X", rfid2.uid.uidByte[i]);
                }
                printf("\n\r");
                
                //pulse room 1 (front door) latch
                r1_state = !r1_state;
                r1_set = r1_state;
                r1_rst = !r1_state;
                wait_ms(100);
                r1_set = 0;
                r1_rst = 0;
                
                wait_ms(200);
            }
        } //end rfid2
    } //end while
} //end main