/*
Copyright (c) 2017-2018, E Ink Holdings Inc., All Rights Reserved
SPDX-License-Identifier: LicenseRef-PBL

This file and the related binary are licensed under the Permissive Binary 
License, Version 1.0 (the "License"); you may not use these files except in 
compliance with the License.

You may obtain a copy of the License here: 
LICENSE-permissive-binary-license-1.0.txt and at
https://www.mbed.com/licenses/PBL-1.0

See the License for the specific language governing permissions and limitations 
under the License.
*/
/**
 * \mainpage EinkShield usage sample code
 * \code
 * #include "mbed.h"
 * #include "EinkShield.h"
 * #include "image3.h"
 * int main() {        
 *     EinkShield epd(EL029TR1,
 *                  D7,
 *                  D6,
 *                  D5,
 *                  D10,
 *                  D2,
 *                  D13,
 *                  D11);        
 *     epd.EPD_Init();   
 *     epd.EPD_Display_Red();
 *     wait_ms(2000); 
 *     while(1) {  
 *        epd.EPD_Display_KWR(sale2_KW, sale2_R);
 *        wait_ms(2000); 
 *    } 
 * }
 * \endcode
 */
#ifndef EINK_SHIELD_H
#define EINK_SHIELD_H

typedef enum {
    EL029TR1,  
} EPD_driver;


/** @class EinkShield EinkShield.h */
/** class EinkShield for mbed-os */
class EinkShield {
private:
    DigitalOut bsi  ;//(D7);
    DigitalOut rstn ;//(D6);
    DigitalIn  busyn;//(D5);
    DigitalOut csb  ;//(D10);
    DigitalOut dc   ;//(D2);
    DigitalOut scl  ;//(D13);
    DigitalOut sda  ;//(D11);
       
    EPD_driver driver;
 
public:
   /**
    *    Constructor to set pin assignment and driver
    *    @param driver select different size display driver for EinkShield, for example: EL029TR1
    *    @param bsi_pin bus selection pin
    *    @param rstn_pin reset pin, L: driver will reset when low
    *    @param busyn_pin busy pin, L: driver is busy
    *    @param csb_pin chip-select pin
    *    @param dc_pin data/command pin
    *    @param scl_pin serial clock pin
    *    @param sda_pin serial data pin
    *    @return none
    */ 
    EinkShield(EPD_driver driver,
          PinName bsi_pin,
          PinName rstn_pin,
          PinName busyn_pin,
          PinName csb_pin,
          PinName dc_pin,
          PinName scl_pin,
          PinName sda_pin);
   /**
    *    Driver initial
    *    @param none
    *    @return none
    */          
    void EPD_Init(void);
   /**
    *    Display image with color: black, white and red.
    *    <pre>
    *    Resolution of EL029TR1 is 128x296.
    *    Pixel data alignment is from left to right and from top to bottom.
    *
    *    <STRONG>img_kw</STRONG> point to black and white raw pixel data of image, 
    *    1 bit per pixel, 0 = black, 1 = white,  
    *    Total size of img_kw is 128x296/8 = 4736 bytes for EL029TR1.
    *
    *    <STRONG>img_r</STRONG> point to red raw pixel data of image, 
    *    1 bit per pixel, 0 = red, 1 = reserved, 
    *    Total size of img_r is 128x296/8 = 4736 bytes for EL029TR1.
    *    </pre>
    *    
    *    @param img_kw <pre>point to black and white raw pixel data of image, 
    *    1 bit per pixel, 0 = black, 1 = white,</pre>
    *  
    *    @param img_r <pre>point to red raw pixel data of image, 
    *    1 bit per pixel, 0 = red, 1 = reserved,</pre>
    *    
    *    
    *    @return none
    */   
    void EPD_Display_KWR(unsigned char const *img_kw, unsigned char const *img_r);
   /**
    *    Display full screen red
    *    
    *    @param none
    *    @return none
    */   
    void EPD_Display_Red(void);          
};



#endif
