/**
 * Copyright (c) 2015 Digi International Inc.,
 * All rights not expressly granted are reserved.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Digi International Inc. 11001 Bren Road East, Minnetonka, MN 55343
 * =======================================================================
 */

#include "mbed.h"
#include "XBeeLib.h"
#if defined(ENABLE_LOGGING)
#include "DigiLoggerMbedSerial.h"
using namespace DigiLog;
#endif

#error "Replace next define with the remote module's Node Identifier (NI parameter)"
#define REMOTE_NODE_ID              "MyNodeID"

#define MAX_NODES                   5
#define NODE_DISCOVERY_TIMEOUT_MS   5000

using namespace XBeeLib;

Serial *log_serial;
XBeeZB * xbee = NULL;

RemoteXBeeZB remote_nodes_in_network[MAX_NODES];
unsigned int remote_nodes_count = 0;

void discovery_function(const RemoteXBeeZB& remote, char const * const node_id)
{
    MBED_ASSERT(xbee != NULL);
    const uint64_t remote_addr64 = remote.get_addr64();

    log_serial->printf("Found device '%s' [%08x:%08x][%04X]\r\n", node_id, UINT64_HI32(remote_addr64), UINT64_LO32(remote_addr64), remote.get_addr16());

    if (remote_nodes_count < MAX_NODES) {
        remote_nodes_in_network[remote_nodes_count] = remote;
        remote_nodes_count++;
    } else {
        log_serial->printf("Found more nodes than maximum configured for example (%d)\r\n", MAX_NODES);
    }
}

int main()
{
    log_serial = new Serial(DEBUG_TX, DEBUG_RX);
    log_serial->baud(9600);
    log_serial->printf("Sample application to demo how to discover other XBeeZB modules in the network and send a message to them\r\n\r\n");
    log_serial->printf(XB_LIB_BANNER);

#if defined(ENABLE_LOGGING)
    new DigiLoggerMbedSerial(log_serial, LogLevelInfo);
#endif

    xbee = new XBeeZB(RADIO_TX, RADIO_RX, RADIO_RESET, NC, NC, 9600);
    RadioStatus radioStatus = xbee->init();
    MBED_ASSERT(radioStatus == Success);

    /* Wait until the device has joined the network */
    log_serial->printf("Waiting for device to join the network: ");
    while (!xbee->is_joined()) {
        wait_ms(1000);
        log_serial->printf(".");
    }
    log_serial->printf("OK\r\n");

    log_serial->printf("Configuring Node Discovery timeout to %d ms\r\n", NODE_DISCOVERY_TIMEOUT_MS);
    radioStatus = xbee->config_node_discovery(NODE_DISCOVERY_TIMEOUT_MS);
    if (radioStatus != Success) {
        log_serial->printf("Error on config_node_discovery()\r\n");
    }

    log_serial->printf("Trying to discover '%s' by its NI\r\n", REMOTE_NODE_ID);
    RemoteXBeeZB remote_node = xbee->get_remote_node_by_id(REMOTE_NODE_ID);

    if (remote_node.is_valid()) {
        const uint64_t addr64 = remote_node.get_addr64();
        log_serial->printf("Found '%s'! [%08x:%08x|%04x]\r\n", REMOTE_NODE_ID, UINT64_HI32(addr64), UINT64_LO32(addr64), remote_node.get_addr16());

        const char message[] = "Hello " REMOTE_NODE_ID "!";
        log_serial->printf("Sending data to remote device\r\n");
        const TxStatus txStatus = xbee->send_data(remote_node, (const uint8_t *)message, sizeof message - 1);
        if (txStatus != TxStatusSuccess) {
            log_serial->printf("Found an error while sending data %d\r\n", txStatus);
        }
    } else {
        log_serial->printf("Couldn't find '%s' node\r\n", REMOTE_NODE_ID);
    }

    /* Register callbacks */
    xbee->register_node_discovery_cb(&discovery_function);

    log_serial->printf("\r\nStarting Node Discovery\r\n\r\n");
    xbee->start_node_discovery();

    do {
        xbee->process_rx_frames();
        wait_ms(10);
    } while(xbee->is_node_discovery_in_progress());
    log_serial->printf("\r\nNode Discovery Finished\r\n\r\n");

    for (unsigned int i = 0; i < remote_nodes_count; i++) {

        RemoteXBeeZB remote = remote_nodes_in_network[i];
        const uint64_t remote_addr64 = remote.get_addr64();

        const char message[] = "Hello neighbor!";
        log_serial->printf("Sending data to remote device [%08x:%08x][%04X]\r\n", UINT64_HI32(remote_addr64), UINT64_LO32(remote_addr64), remote.get_addr16());
        const TxStatus txStatus = xbee->send_data(remote, (const uint8_t *)message, sizeof message - 1);
        if (txStatus != TxStatusSuccess) {
            log_serial->printf("Found an error while sending data %d\r\n", txStatus);
        }
    }

    log_serial->printf("Example finished\r\n");

    delete(log_serial);
    delete(xbee);
    while (true) {
        wait_ms(10000);
    }
}
