/**
 * Copyright (c) 2015 Digi International Inc.,
 * All rights not expressly granted are reserved.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Digi International Inc. 11001 Bren Road East, Minnetonka, MN 55343
 * =======================================================================
 */

#include "mbed.h"
#include "XBeeLib.h"
#if defined(ENABLE_LOGGING)
#include "DigiLoggerMbedSerial.h"
using namespace DigiLog;
#endif

using namespace XBeeLib;

#define NEW_NODE_ID         "mbed XBee"
#define NEW_PANID           0xD161
#define NEW_CHANNEL_MASK    0x3FFF
#define NEW_POWER_LEVEL     2

Serial *log_serial;

/** Callback function, invoked at packet reception */
static void receive_cb(const RemoteXBeeZB& remote, bool broadcast, const uint8_t *const data, uint16_t len)
{
    log_serial->printf("Data received\r\n");
}

int main()
{
    log_serial = new Serial(DEBUG_TX, DEBUG_RX);
    log_serial->baud(9600);
    log_serial->printf("Sample application to demo how to configure a XBeeZB module through AT wrappers\r\n\r\n");
    log_serial->printf(XB_LIB_BANNER);

#if defined(ENABLE_LOGGING)
    new DigiLoggerMbedSerial(log_serial, LogLevelInfo);
#endif

    XBeeZB xbee = XBeeZB(RADIO_TX, RADIO_RX, RADIO_RESET, NC, NC, 9600);

    xbee.register_receive_cb(&receive_cb);
    RadioStatus radioStatus = xbee.init();
    MBED_ASSERT(radioStatus == Success);

    log_serial->printf("Always check for coordinator before connecting\r\n");
    xbee.check_for_coordinator_at_start(true);
    if (radioStatus != Success) {
        log_serial->printf("Failed to set check_for_coordinator_at_start()\r\n");
    }

    uint8_t encryption_key[] = {0x12, 0x34, 0x56, 0x78};
    log_serial->printf("Configuring Encryption Key\r\n");
    radioStatus = xbee.set_network_encryption_key(encryption_key, sizeof encryption_key);
    if (radioStatus != Success) {
        log_serial->printf("Error when setting encryption key\r\n");
    }

    log_serial->printf("Enabling Encryption\r\n");
    radioStatus = xbee.enable_network_encryption(true);
    if (radioStatus != Success) {
        log_serial->printf("Error when enabling encryption\r\n");
    }

    uint64_t current_panid;
    radioStatus = xbee.get_operating_panid(&current_panid);
    if (radioStatus != Success) {
        log_serial->printf("Error reading the PAN ID\r\n");
        current_panid = 0;
    }
    log_serial->printf("Current PAN ID is '%X%X', setting it to '%04X%04X'\r\n", UINT64_HI32(current_panid), UINT64_LO32(current_panid), UINT64_HI32(NEW_PANID), UINT64_LO32(NEW_PANID));
    radioStatus = xbee.set_panid(NEW_PANID);
    if (radioStatus != Success) {
        log_serial->printf("Error when setting PAN ID\r\n");
    }

    char current_node_identifier[21];
    radioStatus = xbee.get_node_identifier(current_node_identifier);
    if (radioStatus != Success) {
        log_serial->printf("Error reading the node identifier");
        current_node_identifier[0] = '\0'; /* Set to empty string. */
    }
    log_serial->printf("Current Node Identifier '%s', setting it to '%s'\r\n", current_node_identifier, NEW_NODE_ID);
    radioStatus = xbee.set_node_identifier(NEW_NODE_ID);
    if (radioStatus != Success) {
        log_serial->printf("Error setting the node identifier\r\n");
    }

    uint8_t current_power_level;
    radioStatus = xbee.get_power_level(&current_power_level);
    if (radioStatus != Success) {
        log_serial->printf("Error reading the Power Level\r\n");
        current_power_level = 0;
    }
    log_serial->printf("Current Power level is '%d', setting it to '%d'\r\n", current_power_level, NEW_POWER_LEVEL);
    radioStatus = xbee.set_power_level(NEW_POWER_LEVEL);
    if (radioStatus != Success) {
        log_serial->printf("Error when setting Power Level\r\n");
    }

    uint16_t current_channel_mask;
    radioStatus = xbee.get_channel_mask(&current_channel_mask);
    if (radioStatus != Success) {
        log_serial->printf("Error reading the Channel Mask\r\n");
        current_channel_mask = 0;
    }
    log_serial->printf("Current channel mask is '%04X', setting it to '%04X'\r\n", current_channel_mask, NEW_CHANNEL_MASK);
    radioStatus = xbee.set_channel_mask(NEW_CHANNEL_MASK);
    if (radioStatus != Success) {
        log_serial->printf("Error when setting channel\r\n");
    }

    do {
        wait_ms(100);
    } while (!xbee.is_joined());

    uint16_t network_addr;
    radioStatus = xbee.get_network_address(&network_addr);
    if (radioStatus != Success) {
        log_serial->printf("Error when reading network address\r\n");
        network_addr = 0xFFFF;
    }
    log_serial->printf("Network address is '%04x'\r\n", network_addr);


    //xbee.write_config(); /* Uncomment this line to save changes into Flash. */

    for(;;) {
        xbee.process_rx_frames();
        wait_ms(1000);
    }
    delete(log_serial);
}
