/**
 * Copyright (c) 2015 Digi International Inc.,
 * All rights not expressly granted are reserved.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Digi International Inc. 11001 Bren Road East, Minnetonka, MN 55343
 * =======================================================================
 */

#include "mbed.h"
#include "XBeeLib.h"
#if defined(ENABLE_LOGGING)
#include "DigiLoggerMbedSerial.h"
using namespace DigiLog;
#endif

#define REMOTE_NODE_ADDR64_MSB  ((uint32_t)0x0013A200)

#error "Replace next define with the LSB of the remote module's 64-bit address (SL parameter)"
#define REMOTE_NODE_ADDR64_LSB  ((uint32_t)0x01234567)

#define REMOTE_NODE_ADDR64      UINT64(REMOTE_NODE_ADDR64_MSB, REMOTE_NODE_ADDR64_LSB)

using namespace XBeeLib;

Serial *log_serial;

int main()
{
    log_serial = new Serial(DEBUG_TX, DEBUG_RX);
    log_serial->baud(9600);
    log_serial->printf("Sample application to demo how to handle remote XBeeZB devices DIOs and ADCs\r\n\r\n");
    log_serial->printf(XB_LIB_BANNER);

#if defined(ENABLE_LOGGING)
    new DigiLoggerMbedSerial(log_serial, LogLevelInfo);
#endif

    XBeeZB xbee = XBeeZB(RADIO_TX, RADIO_RX, RADIO_RESET, NC, NC, 9600);

    RadioStatus radioStatus = xbee.init();
    MBED_ASSERT(radioStatus == Success);

    /* Wait until the device has joined the network */
    log_serial->printf("Waiting for device to join the network: ");
    while (!xbee.is_joined()) {
        wait_ms(1000);
        log_serial->printf(".");
    }
    log_serial->printf("OK\r\n");

    const RemoteXBeeZB remoteDevice = RemoteXBeeZB(REMOTE_NODE_ADDR64);

    radioStatus = xbee.set_pin_config(remoteDevice, XBeeZB::DIO3_AD3, DigitalInput);
    MBED_ASSERT(radioStatus == Success);

    radioStatus = xbee.set_pin_pull_up(remoteDevice, XBeeZB::DIO3_AD3, true);
    MBED_ASSERT(radioStatus == Success);

    radioStatus = xbee.set_pin_config(remoteDevice, XBeeZB::DIO4, DigitalOutHigh);
    MBED_ASSERT(radioStatus == Success);

    radioStatus = xbee.set_pin_config(remoteDevice, XBeeZB::DIO2_AD2, Adc);
    MBED_ASSERT(radioStatus == Success);

//#define TEST_SUPPLY_VOLTAGE

#ifdef TEST_SUPPLY_VOLTAGE
    /* Set a high V+ value so radio sends Supply Voltage information */
    uint32_t v_value = 0x0C00;
    log_serial->printf("\r\nSetting remote device V+ parameter to 0x%04x:\r\n", v_value);
    AtCmdFrame::AtCmdResp cmdresp = xbee.set_param(remoteDevice, "V+", v_value);

    if (cmdresp == AtCmdFrame::AtCmdRespOk)
        log_serial->printf("OK\r\n");
    else
        log_serial->printf("FAILED with %d\r\n", (int) cmdresp);
#endif

    while(true) {
        IOSampleZB ioSample = xbee.get_iosample(remoteDevice);

        if (!ioSample.is_valid()) {
            log_serial->printf("get_iosample failed, ADC and Digital Input reads will be invalid\r\n");
        }
        /* Read DIO3_AD3 digital value */
        DioVal dio3_val;

        radioStatus = ioSample.get_dio(XBeeZB::DIO3_AD3, &dio3_val);
        MBED_ASSERT(radioStatus == Success);
        log_serial->printf("DIO3 value = %d\r\n", dio3_val);

        /* Toggle LED associated to DIO4 */
        static bool led_on = false;
        if (!led_on)
            radioStatus = xbee.set_dio(remoteDevice, XBeeZB::DIO4, Low);
        else
            radioStatus = xbee.set_dio(remoteDevice, XBeeZB::DIO4, High);
        MBED_ASSERT(radioStatus == Success);
        led_on = !led_on;

        /* Read DIO2_AD2 analog value */
        uint16_t adc2_val;
        radioStatus = ioSample.get_adc(XBeeZB::DIO2_AD2, &adc2_val);
        MBED_ASSERT(radioStatus == Success);
        log_serial->printf("ADC2 value = 0x%04x\r\n", adc2_val);

#ifdef TEST_SUPPLY_VOLTAGE
        uint16_t voltaje_val;
        radioStatus = ioSample.get_adc(XBeeZB::SUPPLY_VOLTAGE, &voltaje_val);
        if (radioStatus == Success)
            log_serial->printf("Remote module Supply Voltage = 0x%04x\r\n", voltaje_val);
#endif

        log_serial->printf("\r\n");

        wait(5);
    }

    delete(log_serial);
}
