/**
 * Copyright (c) 2015 Digi International Inc.,
 * All rights not expressly granted are reserved.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Digi International Inc. 11001 Bren Road East, Minnetonka, MN 55343
 * =======================================================================
 */

#include "mbed.h"
#include "XBeeLib.h"
#if defined(ENABLE_LOGGING)
#include "DigiLoggerMbedSerial.h"
using namespace DigiLog;
#endif

#define REMOTE_NODE_ADDR64_MSB  ((uint32_t)0x0013A200)

#error "Replace next define with the LSB of the remote module's 64-bit address (SL parameter)"
#define REMOTE_NODE_ADDR64_LSB  ((uint32_t)0x01234567)

#define REMOTE_NODE_ADDR64      UINT64(REMOTE_NODE_ADDR64_MSB, REMOTE_NODE_ADDR64_LSB)

using namespace XBeeLib;

Serial *log_serial;

static void io_data_cb(const RemoteXBeeZB& remote, const IOSampleZB& sample_data)
{
    const uint64_t remote_addr64 = remote.get_addr64();

    log_serial->printf("\r\nGot an IO SAMPLE packet [%08x:%08x|%04x]\r\n", UINT64_HI32(remote_addr64), UINT64_LO32(remote_addr64), remote.get_addr16());

    RadioStatus radioStatus;

    DioVal dio3;
    radioStatus = sample_data.get_dio(XBeeZB::DIO3_AD3, &dio3);
    if (radioStatus != Success)
        log_serial->printf("sample_data.get_dio(XBeeZB::DIO3_AD3, &dio3) FAILED\r\n");
    else
        log_serial->printf("DIO3 Digital value %d\r\n", dio3);

    uint16_t ad2;
    radioStatus = sample_data.get_adc(XBeeZB::DIO2_AD2, &ad2);
    if (radioStatus != Success)
        log_serial->printf("sample_data.get_adc(XBeeZB::DIO2_AD2, &ad2) FAILED\r\n");
    else
        log_serial->printf("AD2 Analog value %04x\r\n", ad2);
}

int main()
{
    log_serial = new Serial(DEBUG_TX, DEBUG_RX);
    log_serial->baud(9600);
    log_serial->printf("Sample application to demo how to receive IO Sample Data from a RemoteXBeeZB node\r\n\r\n");
    log_serial->printf(XB_LIB_BANNER);

#if defined(ENABLE_LOGGING)
    new DigiLoggerMbedSerial(log_serial, LogLevelInfo);
#endif

    XBeeZB xbee = XBeeZB(RADIO_TX, RADIO_RX, RADIO_RESET, NC, NC, 9600);

    /* Register callback */
    xbee.register_io_sample_cb(&io_data_cb);

    RadioStatus radioStatus;
    radioStatus = xbee.init();
    MBED_ASSERT(radioStatus == Success);

    /* Wait until the device has joined the network */
    log_serial->printf("Waiting for device to join the network: ");
    while (!xbee.is_joined()) {
        wait_ms(1000);
        log_serial->printf(".");
    }

    log_serial->printf("OK\r\n");

    const RemoteXBeeZB remoteDevice = RemoteXBeeZB(REMOTE_NODE_ADDR64);
    log_serial->printf("Configuring remote Device (%08X:%08X) DIO3_AD3 as Input and DIO2_AD2 as ADC\r\n", REMOTE_NODE_ADDR64_MSB, REMOTE_NODE_ADDR64_LSB);
    radioStatus = xbee.set_pin_config(remoteDevice, XBeeZB::DIO3_AD3, DigitalInput);
    MBED_ASSERT(radioStatus == Success);
    log_serial->printf("Enabling remote Device (%08X:%08X) internal Pull-up resistor on DIO3_AD3\r\n", REMOTE_NODE_ADDR64_MSB, REMOTE_NODE_ADDR64_LSB);
    radioStatus = xbee.set_pin_pull_up(remoteDevice, XBeeZB::DIO3_AD3, true);
    MBED_ASSERT(radioStatus == Success);
    radioStatus = xbee.set_pin_config(remoteDevice, XBeeZB::DIO2_AD2, Adc);
    MBED_ASSERT(radioStatus == Success);

    log_serial->printf("Configuring remote Device (%08X:%08X) to send samples every 5 seconds\r\n", REMOTE_NODE_ADDR64_MSB, REMOTE_NODE_ADDR64_LSB);
    xbee.set_io_sample_rate(remoteDevice, 5.0);

    log_serial->printf("Configuring remote Device (%08X:%08X) to send samples to this device\r\n", REMOTE_NODE_ADDR64_MSB, REMOTE_NODE_ADDR64_LSB);
    const uint64_t localAddr = xbee.get_addr64();
    xbee.config_io_sample_destination(remoteDevice, localAddr);

    while (true) {
        xbee.process_rx_frames();
        wait_ms(100);
        log_serial->printf(".");
    }

    delete(log_serial);
}
