/**
 * Copyright (c) 2015 Digi International Inc.,
 * All rights not expressly granted are reserved.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Digi International Inc. 11001 Bren Road East, Minnetonka, MN 55343
 * =======================================================================
 */

#include "mbed.h"
#include "XBeeLib.h"
#if defined(ENABLE_LOGGING)
#include "DigiLoggerMbedSerial.h"
using namespace DigiLog;
#endif

#define REMOTE_NODE_ADDR64_MSB  ((uint32_t)0x0013A200)

#error "Replace next define with the LSB of the remote module's 64-bit address (SL parameter)"
#define REMOTE_NODE_ADDR64_LSB  ((uint32_t)0x01234567)

#define REMOTE_NODE_ADDR64      UINT64(REMOTE_NODE_ADDR64_MSB, REMOTE_NODE_ADDR64_LSB)

using namespace XBeeLib;

Serial *log_serial;

int main()
{
    log_serial = new Serial(DEBUG_TX, DEBUG_RX);
    log_serial->baud(9600);
    log_serial->printf("Sample application to demo how to read and set local and remote AT parameters with XBeeDM\r\n\r\n");
    log_serial->printf(XB_LIB_BANNER);

#if defined(ENABLE_LOGGING)
    new DigiLoggerMbedSerial(log_serial, LogLevelInfo);
#endif

    XBeeDM xbee = XBeeDM(RADIO_TX, RADIO_RX, RADIO_RESET, NC, NC, 9600);

    RadioStatus radioStatus = xbee.init();
    MBED_ASSERT(radioStatus == Success);

    AtCmdFrame::AtCmdResp cmdresp;

    uint32_t value;

    /* Read local device SL parameter */
    log_serial->printf("\r\nReading local device SL parameter:\r\n");
    cmdresp = xbee.get_param("SL", &value);
    if (cmdresp == AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("OK. Local SL=%08x\r\n", value);

        /* Get the local device 64 bit address to compare */
        const uint64_t LocalDeviceAddr64 = xbee.get_addr64();
        MBED_ASSERT(value == (LocalDeviceAddr64 & 0xFFFFFFFF));
    } else {
        log_serial->printf("FAILED with %d\r\n", (int) cmdresp);
    }

    /* Set local device NI parameter */
    char ni_local[] = "ni_example_local";
    log_serial->printf("\r\nSetting local device NI parameter to '%s':\r\n", ni_local);
    cmdresp = xbee.set_param("NI", (uint8_t*)ni_local, strlen(ni_local));
    if (cmdresp == AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("OK\r\n");
    } else {
        log_serial->printf("FAILED with %d\r\n", (int) cmdresp);
    }

    const RemoteXBeeDM remoteDevice = RemoteXBeeDM(REMOTE_NODE_ADDR64);

    /* Read remote device SL parameter */
    log_serial->printf("\r\nReading remote device SL parameter:\r\n");
    cmdresp = xbee.get_param(remoteDevice, "SL", &value);
    if (cmdresp == AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("OK. Remote SL=%08x\r\n", value);
        MBED_ASSERT(value == REMOTE_NODE_ADDR64_LSB);
    } else {
        log_serial->printf("FAILED with %d\r\n", (int) cmdresp);
    }

    /* Set remote device NI parameter */
    char ni_remote[] = "ni_example_remote";
    log_serial->printf("\r\nSetting remote device NI parameter to '%s':\r\n", ni_remote);
    cmdresp = xbee.set_param(remoteDevice, "NI", (uint8_t*)ni_remote, strlen(ni_remote));
    if (cmdresp == AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("OK\r\n");
    } else {
        log_serial->printf("FAILED with %d\r\n", (int) cmdresp);
    }

    delete(log_serial);
}
