/**
 * Copyright (c) 2015 Digi International Inc.,
 * All rights not expressly granted are reserved.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Digi International Inc. 11001 Bren Road East, Minnetonka, MN 55343
 * =======================================================================
 */

#include "mbed.h"
#include "XBeeLib.h"
#if defined(ENABLE_LOGGING)
#include "DigiLoggerMbedSerial.h"
using namespace DigiLog;
#endif

using namespace XBeeLib;

#define SLEEP_SECONDS			(40)	/* 40 seconds, for demo */

#if !defined(RADIO_SLEEP_REQ)
    #error "Please define RADIO_SLEEP_REQ pin in config.h"
#endif
#if !defined(RADIO_ON_SLEEP)
    #error "Please define RADIO_ON_SLEEP pin in config.h"
#endif

Serial *log_serial;

DigitalOut* sleep_req = NULL;
DigitalIn* on_sleep = NULL;

bool is_radio_sleeping()
{
    assert(on_sleep != NULL);

    return on_sleep->read() == 0;
}

void sleep_radio()
{
    assert(sleep_req != NULL);

    sleep_req->write(1);
}

void awake_radio()
{
    assert(sleep_req != NULL);

    sleep_req->write(0);

    /* Wait until radio awakes. Typically 14 mS */
    while(is_radio_sleeping());
}

static void send_sample(XBee802& xbee)
{
    char data[60];
    static uint16_t sample = 0;
    const uint16_t data_len = sprintf(data, "\r\nSensor sample: %u, next sample in %lu seconds\r\n",
								 sample++, (uint32_t)SLEEP_SECONDS);

    const TxStatus txStatus = xbee.send_data_broadcast((const uint8_t *)data, data_len);
    if (txStatus == TxStatusSuccess)
        log_serial->printf("send_data OK\r\n");
    else
        log_serial->printf("send_data failed with %d\r\n", (int) txStatus);
}

int main()
{
    AtCmdFrame::AtCmdResp cmdresp;

    /* Configure cpu pins to control the radio power management pins:
     *     - RADIO_SLEEP_REQ: This pin connected to the radio SLEEP_RQ pin (pin 9 on THT modules, pin 10 on SMT modules) will
     *                   allow the cpu to request the radio to sleep or awake.
     *     - RADIO_ON_SLEEP: This pin connected to radio ON/SLEEP# pin (pin 13 on THT modules, pin 26 on SMT modules) will
     *                  allow the cpu to know if the radio is awaked or sleept.
     */
    sleep_req = new DigitalOut(RADIO_SLEEP_REQ);
    on_sleep = new DigitalIn(RADIO_ON_SLEEP);

    log_serial = new Serial(DEBUG_TX, DEBUG_RX);
    log_serial->baud(9600);
    log_serial->printf("Sample application to demo pin sleep power management with the XBee802\r\n\r\n");
    log_serial->printf(XB_LIB_BANNER);

#if defined(ENABLE_LOGGING)
    new DigiLoggerMbedSerial(log_serial, LogLevelInfo);
#endif

    XBee802 xbee = XBee802(RADIO_TX, RADIO_RX, RADIO_RESET, NC, NC, 9600);

    RadioStatus radioStatus = xbee.init();
    MBED_ASSERT(radioStatus == Success);

    /* Configure Sleep mode */
    cmdresp = xbee.set_param("SM", 1); /* Pin Hibernate */
    if (cmdresp != AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("SM Failed!!\r\n");
    }

    /* Start sending samples */
    while (1) {
        /* Awake the radio */
        awake_radio();

        send_sample(xbee);

        /* Sleep the radio again */
        sleep_radio();

        wait(SLEEP_SECONDS);
    }

    delete(log_serial);
}
