/**
 * Copyright (c) 2015 Digi International Inc.,
 * All rights not expressly granted are reserved.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Digi International Inc. 11001 Bren Road East, Minnetonka, MN 55343
 * =======================================================================
 */

#include "mbed.h"
#include "XBeeLib.h"
#if defined(ENABLE_LOGGING)
#include "DigiLoggerMbedSerial.h"
using namespace DigiLog;
#endif

using namespace XBeeLib;

/* Configure Sleep Options */
#define SLEEP_OPTIONS 0      /* Short Sleep */

/* Configure Sleep Period in mS */
#define SLEEP_PERIOD_MS (5000 / 10)   /* 5000 mS */

/* Configure Time before sleep */
#define TIME_BEFORE_SLEEP_MS 500    /* 500 mS */

Serial *log_serial;

/** Callback function, invoked at packet reception */
static void receive_cb(const RemoteXBee802& remote, bool broadcast, const uint8_t *const data, uint16_t len)
{
    const uint64_t remote_addr64 = remote.get_addr64();

    log_serial->printf("\r\nGot a %s RX packet [%08x:%08x|%04x], len %d\r\nData: ", broadcast ? "BROADCAST" : "UNICAST",
                                      UINT64_HI32(remote_addr64), UINT64_LO32(remote_addr64), remote.get_addr16(), len);

    log_serial->printf("%.*s\r\n\r\n", len, data);
}

int main()
{
    log_serial = new Serial(DEBUG_TX, DEBUG_RX);
    log_serial->baud(9600);
    log_serial->printf("Sample application to demo cyclic sleep power management with the XBee802\r\n\r\n");
    log_serial->printf(XB_LIB_BANNER);

#if defined(ENABLE_LOGGING)
    new DigiLoggerMbedSerial(log_serial, LogLevelInfo);
#endif

    XBee802 xbee = XBee802(RADIO_TX, RADIO_RX, RADIO_RESET, NC, NC, 9600);

    /* Register callbacks */
    xbee.register_receive_cb(&receive_cb);

    RadioStatus const radioStatus = xbee.init();
    MBED_ASSERT(radioStatus == Success);

    /* Configure sleep parameters */
    AtCmdFrame::AtCmdResp cmdresp;

    cmdresp = xbee.set_param("A1", 4);  /* AutoAssociate = Device attempts Association until success */
    if (cmdresp != AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("A1 Failed!!\r\n");
    }

    wait_ms(500);

    cmdresp = xbee.set_param("SO", (uint32_t)SLEEP_OPTIONS);
    if (cmdresp != AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("SO Failed!!\r\n");
    }

    cmdresp = xbee.set_param("SP", SLEEP_PERIOD_MS);
    if (cmdresp != AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("SP Failed!!\r\n");
    }

    cmdresp = xbee.set_param("ST", TIME_BEFORE_SLEEP_MS);
    if (cmdresp != AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("ST Failed!!\r\n");
    }

    /* Configure Sleep mode */
    cmdresp = xbee.set_param("SM", 4); /* Cyclic Sleep Remote */
    if (cmdresp != AtCmdFrame::AtCmdRespOk) {
        log_serial->printf("SM Failed!!\r\n");
    }

    /* Start processing frames */
    while (true) {
        xbee.process_rx_frames();
        wait_ms(100);
        log_serial->printf(".");
    }

    delete(log_serial);
}
