/**
read data temperature,time from DS3221 and write it to 25LC1024
author: Danyal Awan, Johannes Bienert, Max Östermann
*/

#include "mbed.h"
#define ADDR 0x68 << 1 // DS3231 adress
#define MESSPUNKTE 96 //Anzahl der Messungen, die durchgeführt werden sollen.
#define ADDRESSINCREASE 5// Anzahl der Adressen (25LC1024), die erhöht werden müssen.
                        //5 Werte pro Messung werden gespeichert.
#define PRESSED 0 // wenn Userbutton gedrückt
#define DELAY_IN_MS 900000 // Millisekunden, die zwischen den Messungen gewartet werden soll.

Serial pc(SERIAL_TX, SERIAL_RX); // Serielle Schnittstelle 
DigitalIn userButton(BUTTON1, PullUp); // Onboard Button 
I2C i2c(I2C_SDA , I2C_SCL); 
SPI spi(D11, D12, D13); // mosi, miso, sclk
DigitalOut cs(D10);
//DigitalOut out(D9);
AnalogIn input(A0); // Input for temperature data
int32_t counter;


// DS3231 must bet set in 24h mode!!!
// get time in hours from DS3231 via I2C
// @return hours 8_bit integer
int8_t getHours() {
  
    char reader[1]; 
    char cmd[1];    
    cmd[0] = 0x02;  // register for hours
    wait_ms(1);
    i2c.write(ADDR, cmd, 1);
    i2c.read(ADDR, reader, 1);
    int8_t stunden = reader[0];
    stunden = stunden & 0x3F;  
    return stunden;      
}

// get time in minutes from DS3231 via I2C
// @return minutes 8_bit integer
int8_t getMinutes(){
  
    char reader[1];
    char cmd[1]; 
    cmd[0] = 0x01; //register for minutes
    i2c.write(ADDR, cmd, 1); 
    i2c.read(ADDR, reader, 1);
    wait_ms(1);
    int8_t minuten = reader[0];
    return minuten;  
} 

// get temperature from DS3231 via I2C
// @return temperature 8_bit integer
int8_t getTemp(){
     
    int8_t temp = 0; 
    char reader[1];
    char cmd[1];   
    cmd[0] = 0x11; // register for temperature in front of comma
    i2c.write(ADDR, cmd, 1);
    i2c.read(ADDR, reader, 1);
    wait_ms(1);
    temp = reader[0];
    return temp;
}

// get analog input from phototransistor
// @return input voltage 16_bit integer
int16_t getInput(){  

    uint16_t inputWert = 0; 
    inputWert = input.read_u16();
    return inputWert;    
}

// write data to 25LC1024 via SPI
// @param Stunden
// @param Minuten
// @param temperature
// @param Voltage - 16_bit
// @param 24_bit register adress
void saveData(int8_t dataHour,int32_t adress,int8_t dataMinutes,
                                   int8_t dataTemp,int16_t light){
                                       
    char addr1, addr2, addr3, msb2, lsb2; // 1 Byte adress holder
    
    //  split 24_bit adress in 3 8_bit adresses
    addr1 = (adress & 0xFF0000)>>16;
    addr2 = (adress & 0x00FF00)>>8;
    addr3 = (adress & 0x0000FF);
    
    //split 16_bit integer in two 8_bit values
    msb2 = (light & 0xFF00)>>8;
    lsb2 = (light & 0x00FF);
        
    cs = 0; 
    spi.write(0x06); //WREN, Write on
    cs = 1;
    wait_ms(1);
    cs = 0;
    spi.write(0x02); //Write Instruction
    spi.write(addr1); //      -
    spi.write(addr2); // send 24_bit adress devided up by 3 8_bit integers
    spi.write(addr3); //      -
    spi.write(dataTemp);  // send data
    spi.write(dataHour);
    spi.write(dataMinutes);
    spi.write(msb2);
    spi.write(lsb2);
    cs = 1; 
    wait_ms(6);
    cs = 0; 
    spi.write(0x04); //WRDI, Write off
    cs = 1;
}

// read all data from 25LC1024 via SPI
// send all data in own arrays
// print all data on screen
//@param number of all data
void readOutData(int32_t leng) {
    char buffer[leng];
    char minuten[MESSPUNKTE];
    char stunden[MESSPUNKTE];
    char temp[MESSPUNKTE];
    char light1[MESSPUNKTE];
    char light2[MESSPUNKTE];
    int16_t light[MESSPUNKTE];
    // Chip must be deselected    
    cs = 0;
    spi.write(0x03); // read instruction
    spi.write(0x00); //     --
    spi.write(0x00); // 24_bit adress
    spi.write(0x01); //     --
    for (int i = 0; i<leng; i++) {
        buffer[i] = spi.write(0x00);  // send all data in buffer[]
    }
    cs = 1;  
    wait_ms(1); 
    int x = 0;   
    // now send all data in seperated array´s
    // send data to temp array
    for (int i = 0; i<leng; i=i+ADDRESSINCREASE) { 
        pc.printf("%d\n",buffer[i]);  // print temp for analytics
        temp[x] = buffer[i];
        x++;
    }
    x = 0;
    wait_ms(1);
    // send data to stunden array
    for (int i = 1; i<leng; i=i+ADDRESSINCREASE) {
        stunden[x] = buffer[i];
        x++;
    }
    x = 0;
    wait_ms(1);
    // send data minuten
    for (int i = 2; i<leng; i=i+ADDRESSINCREASE) {
        pc.printf("%x:%x\n",stunden[x],buffer[i]);  // print time for analytics
        minuten[x] = buffer[i];
        x++;
    }
    x = 0;
    wait_ms(1);
    for (int i = 3; i<leng; i=i+ADDRESSINCREASE) {
        light1[x] = buffer[i];
        x++;
    }          
    x = 0;
    wait_ms(1);
    for (int i = 4; i<leng; i=i+ADDRESSINCREASE) {
        light2[x] = buffer[i];
        x++;
    }
    x = 0;
    wait_ms(1);
    // output in percent (voltage/light level)
    for (int i = 0; i<MESSPUNKTE; i++){
        uint16_t ltemp = light1[i]<<8;
        uint16_t ltemp2 = light2[i];
        light[i] = (ltemp + ltemp2)/655;
        pc.printf("%d\n",light[i]);  // print light level for analytics
    }
    // print all messures. step by step
    for (int i = 0; i<counter; i++) {
        pc.printf("Messung: %d Temperatur: %d Uhrzeit: %x:%x Voltage: %d Prozent \n",i+1,temp[i],stunden[i],
        minuten[i],light[i]);
    }         
}

// if userbutton pressed programm ends and print all messures
// if userbutton is not pressed programm waits the delaytime till to the next messure.
// every 10 ms button query
//@param delay time in ms
//@return 0 or 1. 0 ends the programm, 1 continues
int8_t delayOrEnd(int32_t delay){
    int32_t x = 0;
    int8_t y = 1;
    while((x<delay/10) && y){
        if(userButton == PRESSED){
            y = 0;
        }
        else{
            wait_ms(10);
        }
        x++;
    }
    return y;
} 
   
      
int main(){

    pc.printf("Press any key on the keyboard to start the program\n\r");
    pc.putc(pc.getc()); //echo keyboard input, blocking wait
    pc.printf(" was pressed\n\r");
    for (int32_t i=0; i<=5; i++){
        pc.printf("***\n\r"); //Generate some new lines
    }
    pc.printf("Messung wird gestartet.\n");
    pc.printf("Anzahl der vorrausichtlichen Messungen: %d.\n",MESSPUNKTE);
    pc.printf("Messabstaende in Millisekunden: %d.\n",DELAY_IN_MS);
    pc.printf("Userbutton druecken, um Messung vorlaeufig zu beenden.\n");

    cs = 1;
    // Setup the spi for 8 bit data, high steady state clock,
    // second edge capture, with a 1MHz clock rate
    spi.format(8,0);
    spi.frequency(100000);
    int32_t adress = 0x01; // starting register adress
 
    //int x =0;
    int k = 1;
    counter = 0;
    while(counter<MESSPUNKTE && k){
          
        saveData(getHours(),adress,getMinutes(),getTemp(),getInput());
        wait_ms(1);
       
        adress = adress + ADDRESSINCREASE;   // erhöhe Speicheradresse um 5 
        counter++;
        pc.printf("Messung ausgefuehrt: %d\n",counter);
        k = delayOrEnd(DELAY_IN_MS);
    }
    pc.printf("\n");
    pc.printf("Messung beendet \n");
    pc.printf("*************************************************\n");
    readOutData(MESSPUNKTE*ADDRESSINCREASE); // MESSPUNKTE*ADDRESSINCREASE = Anzahl aller Daten.
  
}