#ifndef WIFIDEVICE_H_
#define WIFIDEVICE_H_

 /** Interface using WI-FI to connect to Ethernet
 *
 */
 
//#include "mbed.h"
#include <stdio.h>
#include "nrf_delay.h"

#include "drivers/Ticker.h"
#include "drivers/InterruptIn.h"
using namespace mbed;

#include "EthernetInterface.h"
 
class WIFIDevice {
public:

  /** Set WIFI in sleep mode.
  * Disable WIFI and set into sleep mode to conserve energy (no WIFI function is available at this point).
  * \return 0 on success, a negative number on failure
  */
  int sleep(void);

  /** Check if the device is connected to Access Point
  * \return 1 if connected, 0 otherwise.
  */
  int is_AP_connected(void);
    
  /** Perform WIFI scan.
  * Scan for available access point on all channels.
  * \return number of scanned WIFI access point
  */
  int apScan(void (*eventCallback)(tstrM2mWifiscanResult result));
    
  /** Set network connection in priority.
  * Set SSID, password and priority to connect.
  * \param secType type of securith support in STA mode, support
    M2M_WIFI_SEC_OPEN       < Open security >
    M2M_WIFI_SEC_WPA_PSK    < Wi-Fi network is secured with WPA/WPA2 personal(PSK) >
    M2M_WIFI_SEC_WEP        < Security type WEP (40 or 104) OPEN OR SHARED >
    M2M_WIFI_SEC_802_1X     < Wi-Fi network is secured with WPA/WPA2 Enterprise.IEEE802.1x user-name/password authentication >
  * \param SSID name of access point to connect
  * \param PW password of the given SSID   
  * \param wepIndex WEP Index of PW when using M2M_WIFI_SEC_WEP (set between 1 to 4)    
  */
  void setNetwork(uint8_t secType, char* SSID, char* PW, uint8_t wepIndex = 1);

  /** Set WPA/WPA2 Enterprise.IEEE802.1x user-name/password authentication.
  * \param USER user-name of IEEE802.1x authentication
  * \param PW password of IEEE802.1x authentication 
  */  
  void setLogin(char* USER, char* PW);

  /** Trigger Wi-Fi Protected Setup (WPS).
  * WPS activation and scan for fixed 120 seconds (support WPA/WPA2 key only).
  * \return 0 on success (AP connected), a negative number on failure (timeout)
  */
  int triggerWPS();
    
  /** Enable Access Point in given configuration.
  * Set SSID, password, security type and channel in AP mode.
  * \param SSID name of access point in AP mode
  * \param PW password of the given SSID
  * \param AP_ip the IP address of AP
  * \param security tyoe of security in AP mode (default: (M2M_WIFI_SEC_WEP)), support
    M2M_WIFI_SEC_OPEN       < Open security >
    M2M_WIFI_SEC_WEP        < Security type WEP with 10 hexadecimal digits >
  * \param channel range from 1 to 14 (default: (1))
  * \return 0 on success, a negative number on failure  
  */
  int enableAccessPoint(char* SSID, char* PW, char* AP_ip = "192.168.1.1", uint8_t secType=M2M_WIFI_SEC_WEP, uint8_t channel=1);

  /** Set TX power level, must be called after the EthernetInterface initialization and before any connection request and can't be changed in runtime.
  * \param power_level can be configurable as following
    TX_PWR_HIGH             < High power level >
    TX_PWR_MED              < Med power level >
    TX_PWR_LOW              < Low power level >  
  * \return 0 on success, a negative number on failure   
  */  
  int setTXPower(uint8_t power_level);

  /** Set the PPA (Pre-Power Amplifier) gain, must be called after the EthernetInterface initialization
  * \param gain_dB is only configable among 3, 6, 9 dB, default is 9 dB
  * \return 0 on success, a negative number on failure   
  */  
  int setPPAGain(uint8_t gain_dB);

  /** get the current RSSI from the connected AP
  * \return a negative RSSI value in dBm   
  */
  int getRSSI(void);
  
  /** Erase a 4KB sector of embeeded flash.
  * Flash memory will be erased in groups of 4KB sector.
  * \param address range from 0x00 to 0x3F000 (must be a multiple of 0x1000).
  * \return 0 on success, a negative number on failure
  */  
  //int storage_erase4KB(uint32_t address);
  
  /** Write data into embeeded flash.
  * Write one (multiple) byte(s) of data into flash, must perform erase before the flash memory area can be overwrite.
  * \param address flash memory address to be written, range from 0x00 to 0x3FFFF.
  * \param data pointer to the buffer containing data to be written.
  * \param len length of the data to be written to flash.
  * \return 0 on success, a negative number on failure
  */    
  //int storage_write(uint32_t address, uint8_t *data, uint16_t len);
  
  /** Read data from embeeded flash.
  * Read one (multiple) byte(s) of data from flash.
  * \param address flash memory address to be read, range from 0x00 to 0x3FFFF.
  * \param data pointer to the buffer containing data to be read.
  * \param len length of the data to be read from flash.
  * \return 0 on success, a negative number on failure
  */     
  //int storage_read(uint32_t address, uint8_t *data, uint16_t len);
  
};

int enableCoexistance();
void RF_Switch_WIFI_lock(void);
void RF_Switch_WIFI_unlock(void);
#define WIFI_WINDOW_US      600     //min is 300
#define BLE_WINDOW_US      600      //min is 300

#endif /* WIFIDEVICE_H_ */

