#include <mbed.h>

#include "Gap.h"
#include "command-interpreter.h"  
#include "ble/BLE.h"
#include "ble/services/BatteryService.h"
#include "ble/services/DeviceInformationService.h"
#include "ble/services/UARTService.h"

#define uart_buffer_size    64      //Do not increase uart_buffer_size to prevent out of memory, 
#define uart_baudrate       38400   //supported baudrate range between 1200 to 230400 (38400) for NQ620 (NNN50)
unsigned char uart_buf[uart_buffer_size];
unsigned int i = 0;
bool isGetCommand = false;

Serial console(USBTX, USBRX);

DeviceInformationService *deviceInfo;
BatteryService *batteryService;
static UARTService *uartServicePtr;
const static char     DEVICE_NAME[]        = "DELTA_CLI_UART";
static const uint16_t uuid16_list[]        = {GattService::UUID_HUMAN_INTERFACE_DEVICE_SERVICE,
                                              GattService::UUID_BATTERY_SERVICE,
                                              GattService::UUID_DEVICE_INFORMATION_SERVICE};

void CLI_execute() {
        if (uart_buf[i-2] != '\r' || uart_buf[i-1] != '\n') return; //detecting \r\n
        
        isGetCommand = true;
            
}
void uart_interrupt() {
    uart_buf[i++] = console.getc();
    CLI_execute();
}

void onConnectionCallback(const Gap::ConnectionCallbackParams_t *params)
{
    console.printf("Connected\r\n");
}

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    console.printf("Disconnected\r\n");
    BLE::Instance(BLE::DEFAULT_INSTANCE).gap().startAdvertising(); // restart advertising
}

void onTimeoutCallback(Gap::TimeoutSource_t source)
{
    switch (source) {
        case Gap::TIMEOUT_SRC_ADVERTISING:
            console.printf("Advertising timeout\r\n");
            break;  
        case Gap::TIMEOUT_SRC_SECURITY_REQUEST:
            console.printf("Security request timeout\r\n");
            break;
        case Gap::TIMEOUT_SRC_SCAN:
            console.printf("Scanning timeout\r\n");
            break;
        case Gap::TIMEOUT_SRC_CONN:
            console.printf("Connection timeout\r\n");
            break;
    }
}

void serverDataWrittenCallback(const GattWriteCallbackParams *response) {
    console.printf("serverDataWrittenCallback\r\n");
    if (response->handle == uartServicePtr->getTXCharacteristicHandle()) {
        for(int j=0;j<response->len;j++) {
            console.printf("data: %02X\r\n", response->data[j]);
        }
    }
}

void dataSentCallback(const unsigned callback) {
    //uart.printf("dataSentCallback\r\n");
}

void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE &ble          = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        return;
    }
    
    ble.gap().onDisconnection(disconnectionCallback);
    ble.gap().onConnection(onConnectionCallback);
    ble.gap().onTimeout(onTimeoutCallback);
    ble.gattServer().onDataWritten(serverDataWrittenCallback);
    ble.gattServer().onDataSent(dataSentCallback);

    /* Setup primary service. */
    uartServicePtr = new UARTService(ble);

    /* Setup auxiliary service. */
    batteryService = new BatteryService(ble, 100);
    deviceInfo = new DeviceInformationService(ble, "DELTA", "NQ620", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");

    /* Setup advertising. */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(50); /* 50ms */
    ble.gap().startAdvertising();
    console.printf("Start advertising\r\n");
}

int main(void)
{
    
    console.attach(&uart_interrupt);
    console.baud(uart_baudrate);
    console.printf("Application Start\r\n");
    
    BLE& ble = BLE::Instance(BLE::DEFAULT_INSTANCE);
    ble.init(bleInitComplete);

    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) { /* spin loop */ }
    
    while(1) {
        if(isGetCommand) {
            isGetCommand = false;
            
            if(uart_buf[0] == 'b') {
                if (ble.getGapState().connected) {
                    
                    //Write data via RX characteristic
                    ble.gattServer().write(uartServicePtr->getRXCharacteristicHandle(), static_cast<const uint8_t *>(uart_buf)+2, i-4);
                }
            }
            else{
                for (int j=0; j<i; j++)
                    cyntecProcessCommandInput(uart_buf[j]);
                    //console.putc(uart_buf[j]);//used for debug only
            }
            
            i=0;
        }
        
        ble.waitForEvent(); // low power wait for event
    }
}
