/* Copyright (c) Cyntec Inc. All Rights Reserved.*/

#ifndef XMPPCLIENT_H
#define XMPPCLIENT_H

#include <string>
#include "EthernetInterface.h"

/** XMPPClient class.
   used for linking XMPP server(above Openfire3.9.3).

  Example:
  @code

    WIFIDevice wifi;
    XMPPClient xmppClient("delta_nnn40_mod1","85.93.10.5","lester",5222);
    void on_get_mesage(const char*);
    DigitalOut led1(p0);

    int main(void)
    {
        wifi.setNetwork("delta_ap", "password", 0); // set given SSID and PW as the highest priority
        xmppClient.login();
        while(true) {
            xmppClient.set_message_tracker(on_get_mesage);
        }
    }

    void on_get_mesage(const char *msg)
    {
        if(strcmp(msg,"turn on")==0) {
            xmppClient.send_message_to_client("delta_nnn40_android1","the led1 is already turned on");
            led1=1;
        } else if(strcmp(msg,"turn off")==0) {
            xmppClient.send_message_to_client("delta_nnn40_android1","the led1 is already turned off");
            led1=0;
        }
    }

  @endcode
*/

class MxmppClient
{

private:

    EthernetInterface eth;
    TCPSocketConnection socket;

    char *userName;
    char *domain;
    char *resource;
    char *password;
    int port;
    char serverName[128];

    bool bLogin;

public:

    /** Create XMPPClient instance*/

    MxmppClient();

    /** Create XMPPClient instance
    * @param user_name The username is for login XMPP server.
    * @param domain The domain is for server domain.
    * @param pwd The password is for login XMPP server.
    * @param port The port is XMPP server listen port.
    */

    MxmppClient(const char *userName,const char *domain,const char *pwd,const int port=5222);

    /** Set user name which is for login XMPP server.
    * @param name The name is XMPP server account name.
    */

    void setUserName(const char *name);

    /** Set domain which is server domain.
    * @param domain The domain is XMPP server domain.
    */

    void setDomain(const char *domain);

    /** Set password which is for XMPP server login.
    * @param pwd The password is for XMPP server account.
    */

    void setPassword(const char *pwd);

    /** Set port which is for server connection.
    * @param port The port is XMPP server listen port.
    */

    void setPort(const int num);

    /** Get user name.
    * @returns
    * user name
    */

    const char *getUserName();

    /** Get domain.
    * @returns
    * domain
    */

    const char *getDomain();

    /** Get password.
    * @returns
    * password
    */

    const char *getPassword();

    /** Get port.
    * @returns
    * connection port
    */

    const int getPort();

    /** have fill user name?
    * @returns
    * true, have use name,
    * false, no use name
    */

    bool haveUserName();

    /** have fill domain?
    * @returns
    * true, have domain;
    * false, no domain
    */

    bool haveDomain();

    /** have fill password?
    * @returns
    * true, have password;
    * false, no password
    */

    bool havePassword();

    /** have fill port?
    * @returns
    * true, have connection port;
    * false, no connection port
    */

    bool havePort();

    /** have connect data which include username, domain, password and connect port?
    * @returns
    * true, have connection data;
    * false, no connection data
    */

    bool haveConnectData();

    /** Is login to XMPP server?
    * @returns
    * true, is already logged XMPP server;
    * false, is not yet logged XMPP server.
    */

    bool isLogin();

    /** login XMPP server, module will connect to XMPP server and execute login processes.*/

    void login();

    /** logout XMPP server, module will connect to XMPP server and execute logout processes.*/

    void logout();

    /** send message to other client
    * @param client client name,
    * @param chat_msg the sent message.
    */

    void sendMessageToClient(char *client,char* chatMsg);

    /** set chat message listener
    * @param on_get_mesage callback function.
    */

    void setMessageTracker(void (*onGetMessage)(const char *msg));

};

#endif // XMPPCLIENT_H
