#include "mbed.h"
#include "BLE.h"
#include "BatteryService.h"
#include "DeviceInformationService.h"
#include "HealthThermometerService.h"
#include "hts221.h"

#define UPDATE_PARAMS_FOR_LONGER_CONNECTION_INTERVAL 0

BLEDevice  ble;
DigitalOut led1(p1);
DigitalOut led2(p2);

const static char     DEVICE_NAME[]        = "BLEThermo";
static const uint16_t uuid16_list[]        = {GattService::UUID_BATTERY_SERVICE,
                                              GattService::UUID_DEVICE_INFORMATION_SERVICE,
                                              GattService::UUID_HEALTH_THERMOMETER_SERVICE,};
                                              
static volatile bool  triggerSensorPolling = false;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    ble.startAdvertising(); // restart advertising
}

void onConnectionCallback(Gap::Handle_t handle, Gap::addr_type_t peerAddrType, const Gap::address_t peerAddr, const Gap::ConnectionParams_t *params)
{
    #if UPDATE_PARAMS_FOR_LONGER_CONNECTION_INTERVAL
    
    #define MIN_CONN_INTERVAL 250  /**< Minimum connection interval (250 ms) */
    #define MAX_CONN_INTERVAL 350  /**< Maximum connection interval (350 ms). */
    #define CONN_SUP_TIMEOUT  6000 /**< Connection supervisory timeout (6 seconds). */
    #define SLAVE_LATENCY     4

        Gap::ConnectionParams_t gap_conn_params;
        gap_conn_params.minConnectionInterval        = Gap::MSEC_TO_GAP_DURATION_UNITS(MIN_CONN_INTERVAL);
        gap_conn_params.maxConnectionInterval        = Gap::MSEC_TO_GAP_DURATION_UNITS(MAX_CONN_INTERVAL);
        gap_conn_params.connectionSupervisionTimeout = Gap::MSEC_TO_GAP_DURATION_UNITS(CONN_SUP_TIMEOUT);
        gap_conn_params.slaveLatency                 = SLAVE_LATENCY;
        ble.updateConnectionParams(handle, &gap_conn_params);
    #endif /* #if UPDATE_PARAMS_FOR_LONGER_CONNECTION_INTERVAL */
}

void periodicCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */
    triggerSensorPolling = true;
}

float tempCelsius = 25.50;
float humi = 55;
int humiMax = 100; 

int main(void)
{     
    led2 = 0;
    led1 = 0;   
    if (hts221_init())
    {
        HTS221_Calib();
    
        Ticker ticker;
        ticker.attach(periodicCallback, 1);
    
        ble.init();
        ble.onDisconnection(disconnectionCallback);
#if UPDATE_PARAMS_FOR_LONGER_CONNECTION_INTERVAL
        ble.onConnection(onConnectionCallback);
#endif /* #if UPDATE_PARAMS_FOR_LONGER_CONNECTION_INTERVAL */

        
        HealthThermometerService    ThermoService(ble, tempCelsius , HealthThermometerService::LOCATION_EAR);
        BatteryService              battery(ble);
        DeviceInformationService    deviceInfo(ble, "Cyntec", "Combo module", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");
    
        /* Setup advertising. */
        /* Setting advertising string*/
        ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
        ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
        ble.accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_TAG);
        ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
        /* Setting advertising parameters*/
        ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
        ble.setAdvertisingInterval(1000); 
        ble.setAdvertisingTimeout(0x1e);  //Timeout, stop advertising after 30sec
        ble.startAdvertising();

        while (true) {
            if (triggerSensorPolling && ble.getGapState().connected) {
                triggerSensorPolling = false;
            
                HTS221_ReadTempHumi(&tempCelsius, &humi);
                ThermoService.updateTemperature(tempCelsius);
            
            } else {
                ble.waitForEvent();
            }
        }
    } else {
        while(true)
        {
            led2 = !led2;
            wait(1);
        }
    } 
}