/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/services/BatteryService.h"
#include "ble/services/DeviceInformationService.h"
#include "ble/BLE.h"

#include "hts221.h"
#include "uvis25.h"

DigitalOut  led1(LED1);

static const uint8_t UUID_HUMI_AND_UVI[] = {0xf5, 0x59, 0xa2, 0x49, 0xbe, 0xb1, 0x4c, 0x54, 0xa1, 0x0a, 0xc7, 0x95, 0x7e, 0x17, 0xf8, 0x67};
uint8_t wrs_HumiUVI_payload[11] = {0, };
uint8_t htsTempPayload[7] = {0,};
/* Health Thermometer Service */
/* Service:  https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.heart_rate.xml */
/* Temperature Measurement: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.temperature_measurement.xml */
GattCharacteristic htsTemp       ( GattCharacteristic::UUID_TEMPERATURE_MEASUREMENT_CHAR, htsTempPayload, 6, 13, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_INDICATE);
GattCharacteristic wrs_HumiUVI   ( UUID_HUMI_AND_UVI, wrs_HumiUVI_payload, 1, 7, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_INDICATE);
GattCharacteristic *htsChars[] = {&htsTemp, &wrs_HumiUVI};
GattService        htsService    ( GattService::UUID_HEALTH_THERMOMETER_SERVICE, htsChars, sizeof(htsChars) / sizeof(GattCharacteristic *));

float tempCelsius = 25.50;
int32_t tempCelsius_ix100;
int8_t exponent = -2;
uint8_t UTC[7] = { 222, 7, 9, 11, 13, 42, 59 }; 
float humi = 55;
int32_t humi_ix100;
uint8_t uvi = 8; 
uint32_t counter = 0; 

const static char     DEVICE_NAME[]        = "WRS_d7";
static const uint16_t uuid16_list[]        = {GattService::UUID_BATTERY_SERVICE,
                                              GattService::UUID_DEVICE_INFORMATION_SERVICE};
static volatile bool  triggerSensorPolling = false;

/**************************************************************************/
/*!
    @brief  This custom class can be used to override any GattServerEvents
            that you are interested in handling on an application level.
*/
/**************************************************************************/
//
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance(BLE::DEFAULT_INSTANCE).gap().startAdvertising(); // restart advertising
}

void periodicCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */

    /* Note that the periodicCallback() executes in interrupt context, so it is safer to do
     * heavy-weight sensor polling from the main thread. */
    triggerSensorPolling = true;
}

void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE &ble          = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        return;
    }
    
    ble.gap().onDisconnection(disconnectionCallback);
    
    /* Setup auxiliary services. */
    BatteryService           battery(ble);
    DeviceInformationService deviceInfo(ble, "Delta", "NQ620", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");
  
    /* Setup advertising. */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_HEART_RATE_SENSOR);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms */
    ble.gap().startAdvertising();
    
    ble.addService(htsService);
}

/**************************************************************************/
/*!
    @brief  Program entry point
*/
/**************************************************************************/

int main(void)
{
    
    hts221_init();
    uvis25_init(); 

    HTS221_Calib();
    
    Ticker ticker;
    ticker.attach(periodicCallback, 2);
    
    BLE& ble = BLE::Instance(BLE::DEFAULT_INSTANCE);
    ble.init(bleInitComplete);
    
    //ble.init();
//    ble.onDisconnection(disconnectionCallback);

    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) { /* spin loop */ }   
    
    while (1) {

            if(hts221_verify_product_id())
                HTS221_ReadTempHumi(&tempCelsius, &humi);
            else {
                //show dummy value
                tempCelsius = 100;
                humi = 100;
            }
                
            if(uvis25_verify_product_id())
                uvi = UVIS25_ReadUVI();
            else
                uvi = 100;  //Environment Sensor APP will show NO UV DEVICE.
        
            /* Update the Temperature measurement */
            /* See --> https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.temperature_measurement.xml */
            tempCelsius_ix100 = tempCelsius * 100;      
            htsTempPayload[0] = 0x02;         // flag
            htsTempPayload[1] = tempCelsius_ix100%0xff;
            htsTempPayload[2] = tempCelsius_ix100/0xff;
            htsTempPayload[3] = tempCelsius_ix100/0xffff;
            htsTempPayload[4] = exponent;
            for (int i = 0; i < 7; i++) htsTempPayload[i+5] = UTC[i];
            ble.updateCharacteristicValue(htsTemp.getValueAttribute().getHandle(), htsTempPayload, 12);  

            ble.waitForEvent(); 
            wait(1); 
            wrs_HumiUVI_payload[0] = 0x03;    // flag
            memcpy(&wrs_HumiUVI_payload[1],&humi,4);
            wrs_HumiUVI_payload[5] = uvi;
      
            ble.updateCharacteristicValue(wrs_HumiUVI.getValueAttribute().getHandle(), wrs_HumiUVI_payload, 6);
            ble.waitForEvent();                    
            wait(1);
            
    }
        
}