#include "mbed.h"
#include "BLE.h"
#include "DeviceInformationService.h"
#include "BloodPressureService.h"
 
BLEDevice  ble;
DigitalOut led01(LED1);
 
const static char     DEVICE_NAME[]        = "Blood Pressure";
static const uint16_t uuid16_list[]        = {GattService::UUID_BLOOD_PRESSURE_SERVICE,
                                              GattService::UUID_DEVICE_INFORMATION_SERVICE};
static volatile bool  triggerSensorPolling = false;
 
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    ble.startAdvertising(); // restart advertising
}
 
void periodicCallback(void)
{
    led01 = !led01; /* Do blinky on LED1 while we're waiting for BLE events */
 
    /* Note that the periodicCallback() executes in interrupt context, so it is safer to do
     * heavy-weight sensor polling from the main thread. */
    triggerSensorPolling = true;
}
 
int main(void)
{
    led01 = 1;
    Ticker ticker;
    ticker.attach(periodicCallback, 1); // blink LED every second
 
    ble.init();
    ble.gap().onDisconnection(disconnectionCallback);
 
    /* Setup primary service. */
    float SystolicValue = 100.0f; //  mmHg
    float DiastolicValue = 70.0f; //  
    float MeanArterialValue = 55.0f; //
    
    BloodPressureService BloPressureService(ble, SystolicValue, DiastolicValue, MeanArterialValue, 0);
 
    /* Setup auxiliary service. */
    DeviceInformationService deviceInfo(ble, "ARM", "Model1", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");
 
    /* Setup advertising. */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_BLOOD_PRESSURE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.setAdvertisingInterval(1000);
    ble.startAdvertising();
 
    while (1) {
        // check for trigger from periodicCallback()
        if (triggerSensorPolling && ble.getGapState().connected) {
            triggerSensorPolling = false;
 
            // Do blocking calls or whatever is necessary for sensor polling.
            // In our case, we simply update the HRM measurement. 
            SystolicValue++;
            DiastolicValue++;
            MeanArterialValue++;
            
            if (SystolicValue == 140.0f) {
                SystolicValue = 90.0f;
            }
            if (DiastolicValue == 90.0f) {
                DiastolicValue = 50.0f;
            }
            if (MeanArterialValue == 105.0f) {
                MeanArterialValue = 70.0f;
            }
            // update Blood Pressure 
            BloPressureService.updateBloodPressureMeasurement(SystolicValue, DiastolicValue, MeanArterialValue);
        } else {
            ble.waitForEvent(); // low power wait for event
        }
    }
}