#include "CmdSendStringOnInterval.h"
#include "CommandTerminal.h"

CmdSendStringOnInterval::CmdSendStringOnInterval(mDot* dot, mts::MTSSerial& serial)
:
  Command(dot, "Send Continuous", "AT+SENDI", "Sends supplied packet data on interval between sends, output any recevied packets (escape sequence: +++)"),
  _serial(serial) {
    _help = std::string(text()) + ": " + std::string(desc());
    _usage = "(100-2147483647) ms,(string:242)";
}

uint32_t CmdSendStringOnInterval::action(std::vector<std::string> args) {
    int32_t code;
    uint32_t interval;
    std::string text;

    sscanf(args[1].c_str(), "%lu", &interval);

    for (size_t i = 2; i < args.size(); i++) {
        text.append(args[i]);
        if (i != args.size() - 1)
            text.append(",");
    }
    std::vector<uint8_t> data(text.begin(), text.end());

    while (true) {
        if (_dot->getJoinMode() == mDot::AUTO_OTA) {
            Timer timer;
            timer.start();
            uint32_t backoff = 2000;
            while (!_dot->getNetworkJoinStatus()) {
                _serial.writef("Joining network... ");
                if (_dot->joinNetworkOnce() == mDot::MDOT_OK) {
                    _serial.writef("Network Joined\r\n");
                } else {
                    _serial.writef("Join failed\r\n");
                }

                if (CommandTerminal::waitForEscape(backoff)) {
                    return 0;
                }

                if (backoff < 60 * 60 * 1000) {
                    backoff *= 2;
                }
            }
        } else if (!_dot->getNetworkJoinStatus()) {
            setErrorMessage(mDot::getReturnCodeString(mDot::MDOT_NOT_JOINED));
            return 1;
        }

        if ((code = _dot->send(data, _dot->getTxWait())) != mDot::MDOT_OK) {
            std::string error = mDot::getReturnCodeString(code);

            if (code != mDot::MDOT_NOT_JOINED)
                error +=  + " - " + _dot->getLastError();

            if (code == mDot::MDOT_INVALID_PARAM) {
                setErrorMessage(error);
                return 1;
            } else {
                _serial.writef("%s\r\n", error.c_str());
            }
        }

        if (code == mDot::MDOT_OK) {
            if (CommandTerminal::waitForEscape(3000, _dot, CommandTerminal::WAIT_SEND)) {
                return 0;
            }

            if (_dot->getTxWait() || _dot->getAck() > 0) {
                std::vector<uint8_t> rx_data;

                if (_dot->recv(rx_data) == mDot::MDOT_OK) {
                    if (!rx_data.empty()) {
                        if (_dot->getVerbose())
                            _serial.writef("Packet data:\r\n");
                        _serial.writef("%s\r\n", CommandTerminal::formatPacketData(rx_data, _dot->getRxOutput()).c_str());
                    }
                }
            }
        }

        if (CommandTerminal::waitForEscape(interval)) {
            return 0;
        }
    }

    return 0;
}

bool CmdSendStringOnInterval::verify(std::vector<std::string> args) {
    if (args.size() >= 3) {
        uint32_t interval;
        size_t size = 0;

        if (sscanf(args[1].c_str(), "%lu", &interval) != 1) {
            setErrorMessage("Invalid argument");
            return false;
        }

        if (interval < 100 || interval > INT_MAX) {
            setErrorMessage("Invalid interval, expects (100-2147483647) ms");
            return false;
        }

        for (int i = 2; i < args.size(); i++)
            size += args[i].size();
        if (size > 242) {
            setErrorMessage("Invalid packet, expects (string:242)");
            return false;
        }

        return true;
    }

    setErrorMessage("Invalid arguments");
    return false;
}
