/* 
* @ File main.cpp
* @ Author - David Leaming - 25574043
* @ Date - December 2021
*
* Acknowledgements 
* Craig A. Evans, University of Leeds, TMP102 Library ,Feb 2016
* Dr Edmond Nurellari, University of Lincoln, Joystick & N5110 Libraries
*
*/ 

#include "mbed.h"                                                               // include the library header, ensure the library has been imported into the project
#include "TMP102.h"
#include "N5110.h"
#include "Joystick.h"
#include "Bitmap.h"

TMP102 tmp102(I2C_SDA,I2C_SCL);                                                 // Create TMP102 object  

//        VCC,SCE,RST,D/C,MOSI,SCLK,LED 
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);                                      // Create Screen Object - K64F - pwr from 3V3, GND Pin also needs connecting

//                  y     x     button
Joystick joystick(PTB10,PTB11,PTC16);                                           // Define Joystick Object 

Serial pc(USBTX,USBRX);                                                         // UART connection for PC

Ticker ticker_menu;                                                             // Create Menu ticker object

DigitalOut r_led(LED_RED);                                                      // K64F on-board LEDs 
DigitalOut g_led(LED_GREEN);                                                    // K64F on-board LEDs 
DigitalOut b_led(LED_BLUE);                                                     // K64F on-board LEDs 

DigitalOut LED01 (PTA1);                                                        // PCB Surface Mounted LED's - LED1
DigitalOut LED02 (PTA2);                                                        // PCB Surface Mounted LED's - LED2
DigitalOut LED03 (PTC2);                                                        // PCB Surface Mounted LED's - LED3
DigitalOut LED04 (PTC3);                                                        // PCB Surface Mounted LED's - LED4
DigitalOut LED05 (PTC4);                                                        // PCB Surface Mounted LED's - LED5
DigitalOut LED06 (PTD3);                                                        // PCB Surface Mounted LED's - LED6

InterruptIn sw2(SW2);                                                           // K64F on-board switches
InterruptIn sw3(SW3);                                                           // K64F on-board switches
InterruptIn ButtonA (PTB9);                                                     // PCB Button - A
InterruptIn ButtonB (PTD0);                                                     // PCB Button - B

volatile int g_ButtonA_flag = 0;                                                // flag - must be volatile as changes within ISR - g_ prefix makes it easier to distinguish it as global
volatile int g_ButtonB_flag = 0;                                                // flag - must be volatile as changes within ISR - g_ prefix makes it easier to distinguish it as global
volatile int g_sw2_flag = 0;                                                    // flag - must be volatile as changes within ISR - g_ prefix makes it easier to distinguish it as global
volatile int g_menu_timer_flag = 0;                                             // flag - must be volatile as changes within ISR - g_ prefix makes it easier to distinguish it as global
volatile int option = 0;                                                        // Menu option selection based on joystick direction

void error();                                                                   // error function hangs flashing an LED
void init_serial();                                                             // setup serial port
void init_K64F();                                                               // set-up the on-board LEDs and switches
void init_PCB();                                                                // set-up the PCB LEDs and buttons
void ButtonA_isr();                                                             //
void ButtonB_isr();                                                             //
void sw2_isr();                                                                 //
void menu_timer_isr();                                                          //

int main()
{
    init_K64F();                                                                // initialise the board
    init_serial();                                                              // initialise the serial port
    init_PCB();                                                                 // initialise the PCB
        
    tmp102.init();                                                              // call the sensor init method using dot syntax
    lcd.init();                                                                 // initialise display
    joystick.init();                                                            // initialise joystick
    
    ticker_menu.attach(&menu_timer_isr,0.2);                                    // Attach ticker for the Joystick
    
    sw2.fall(&sw2_isr);                                                         // SW2 has a pull-up resistor, so the pin will be at 3.3 V by default and fall to 0 V when pressed. We therefore need to look for a falling edge on the pin to fire the interrupt
    ButtonA.rise(&ButtonA_isr);                                                 // External push button, pin set to 0V by pull down command, means a rising edge is looked for
    ButtonB.rise(&ButtonB_isr);                                                 // External push button, pin set to 0V by pull down command, means a rising edge is looked for
    
    lcd.setContrast(0.5);                                                       // change set contrast in range 0.0 to 1.0
    
    char buffer[14];                                                            // each character is 6 pixels wide, screen is 84 pixels (84/6 = 14)
    
        lcd.clear();                                                            // clear buffer at start of every loop
        lcd.printString("Smart Cold",0,0);                                      // can directly print strings at specified co-ordinates (must be less than 84 pixels to fit on display)
        lcd.printString("Storage",0,1);                                         // Just a welcome message before auto moving to main menu
        lcd.printString("Monitoring",0,2);
        lcd.printString("",0,2);                                                // Blank Line
        lcd.printString("V1.0 - 2021",0,4);
        lcd.printString("David Leaming",0,5);
        lcd.refresh();                                                          // need to refresh display after setting pixels or writing strings 
        wait(5.0);                                                              // leave welcome screen on for designated amount of time
        lcd.clear();                                                            // clear buffer at start of every loop
        lcd.refresh();                                                          // need to refresh display after setting pixels or writing strings 
        lcd.printString("Use Joystick",0,0);                                    // Instruction for use of menu
        lcd.printString("To Navigate",0,1);
        lcd.printString("",0,2);                                                // Blank Line
        lcd.printString("A = Select",0,3);  
        lcd.refresh();                                                          // need to refresh display after setting pixels or writing strings
        wait(5.0);  

    MainMenu();                                                                 //
     
}      

void init_serial() {
    pc.baud(115200);                                                            // set to highest baud - ensure terminal software matches
}

void init_K64F() 
{
    r_led = 1;                                                                  // on-board LEDs are active-low, so set pin high to turn them off.
    g_led = 1;                                                                  // on-board LEDs are active-low, so set pin high to turn them off.
    b_led = 1;                                                                  // on-board LEDs are active-low, so set pin high to turn them off.
    
    sw2.mode(PullNone);                                                         // since the on-board switches have external pull-ups, we should disable the internal pull-down
    sw3.mode(PullNone);                                                         // resistors that are enabled by default using InterruptIn

}

void init_PCB ()
{
    LED01 = 1;                                                                  // PCB surface mounted LED's are active low - write a 1 to turn them off initiallly
    LED02 = 1;                                     
    LED03 = 1;
    LED04 = 1;
    LED05 = 1;
    LED06 = 1;
    
    ButtonA.mode(PullDown);                                                     // Set pin to Pull Down to OV, meaning that a rising edge is looked for when button is pressed
    ButtonB.mode(PullDown);                                                     // Set pin to Pull Down to OV, meaning that a rising edge is looked for when button is pressed
    
}

void ButtonA_isr()                                                              // ButtonA event-triggered interrupt
{
    g_ButtonA_flag = 1;                                                         // set flag in ISR
}

void ButtonB_isr()                                                              // ButtonB event-triggered interrupt
{
    g_ButtonB_flag = 1;                                                         // set flag in ISR
}

void sw2_isr()                                                                  // SW2 event-triggered interrupt
{
    g_sw2_flag = 1;                                                             // set flag in ISR
}

void menu_timer_isr()
{
    g_menu_timer_flag = 1;                                                           // set flag in ISR
}

void MainMenu()
{
    while(1) {
        if (g_menu_timer_flag){
            g_menu_timer_flag = 0;
            
            lcd.clear();                                                        // clear buffer at start of every loop
            joystick.init();                                                    // initialise joystick
            
            lcd.printString("---- MENU ----",0,0);                              // can directly print strings at specified co-ordinates (must be less than 84 pixels to fit on display)
            lcd.printString("M1 - Monitor",0,1);                                // 
            lcd.printString("M2 - One-off",0,2);                                //
            lcd.printString("About",0,3);                                       //
            lcd.refresh();                                                      //
            
            switch (joystick.get_direction())  {                                // Call to check direction joystick is pointing
                    printf("Direction = %i\n");                                 //
                case N:                                                         //
                option--;                                                       //
                break;                                                          //
                case S:                                                         //
                option++;                                                       //
                break;                                                          //
            }
            if (option < 0)  {                                                  // If last/bottom menu item is selcted and user presses joystick down
                option = 2;                                                     // Selection circle to move to first/top menu option
            }
            if (option < 2)  {                                                  // If first/top menu item is selcted and user presses joystick up
                option = 0;                                                     // Selection circle to move to last/bottom menu option
            }
            
            if (option == 1) {                                                  //
                lcd.drawCircle (55,20,2,0);                                     //
            }    
            if (option == 2) {                                                  //
                lcd.drawCircle (35,35,2,0);                                     //
            }
            if (option == 3) {                                                  //
                lcd.drawCircle (20,50,2,0);                                     //
            }                                             
                     
}  

void ConstantMonitoring()              
{        
    while (1) {
        
        lcd.normalMode();                                                       // normal colour mode
        lcd.setBrightness(0.5);                                                 // put LED backlight on 50%
        lcd.clear();                                                            // clear buffer at start of every loop
        lcd.refresh();                                                          // need to refresh display after setting pixels or writing strings 
        
        float T = tmp102.get_temperature();                                     // read temperature and print to lcd
        int length = sprintf(buffer,"T = %.2f C",T);                            // print formatted data to buffer - it is important the format specifier ensures the length will fit in the buffer
        if (length <= 14)                                                       // if string will fit on display (assuming printing at x=0)
        lcd.printString(buffer,0,2);                                            // display on screen
        lcd.refresh();                                                          // need to refresh display after setting pixels or writing strings 
        wait(2.0);
                    
            if (g_ButtonA_flag) {                                               // check if flag i.e. interrupt has occured
            g_ButtonA_flag = 0;                                                 // if it has, clear the flag

            printf("Button A Press \n");                                        // send message over serial port - can observe in CoolTerm etc.
            LED01=0;
            wait(5.0);
            LED01=1;     
            }
      
            if (g_ButtonB_flag) {                                               // check if flag i.e. interrupt has occured
            g_ButtonB_flag = 0;                                                 // if it has, clear the flag

            printf("Button B Press \n");                                        // send message over serial port - can observe in CoolTerm etc.
            LED02=0;
            wait(5.0);
            LED02=1;  
            }
    }
}      