/*******************************************************************************
Edward Hindley
Assistive Health device - measure temperature in an energy efficient and easy to use way 
Board is initialised and awaits edge triggered interrupts to enable 
    a menu, measurement type selection and an average temperature reading.


* Craig A. Evans, University of Leeds, TMP102 Library, Feb 2016
* Dr Edmond Nurellari, University of Lincoln, Joystick, N5110 Libraries, Feb 2022

*******************************************************************************/

#include "mbed.h"                                                               //Include all header files to be used throughout 
#include "Timer.h"
#include "TMP102.h"
#include "Bitmap.h"
#include "N5110.h"
#include "Joystick.h"

/*******************************************************************************
Set-up of all vital peripherals
Note that the following values correspond to these notes
 NOTE_C4  262, NOTE_CS4 277, NOTE_D4  294, NOTE_DS4 311
 NOTE_E4  330, NOTE_F4  349, NOTE_FS4 370, NOTE_G4  392
 NOTE_GS4 415, NOTE_A4  440, NOTE_AS4 466, NOTE_B4  494
 NOTE_C5  523, NOTE_CS5 554, NOTE_D5  587, NOTE_DS5 622
 NOTE_E5  659, NOTE_F5  698, NOTE_FS5 740, NOTE_G5  784
 NOTE_GS5 831, NOTE_A5  880, NOTE_AS5 932, NOTE_B5  988
 NOTE_C6  1047
 *Author Jorge Rancé, PUBLISHED December 4, 2017
*******************************************************************************/
TMP102 tmp102(I2C_SDA,I2C_SCL); 
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Joystick joystick(PTB10,PTB11,PTC16); 
InterruptIn StartButton(PTC5);
InterruptIn BackButton(PTB19);
InterruptIn MenuOnButton(PTB3);                                                 //Open Menu
InterruptIn MenuOffButton(PTB18);                                               //Close Menu
InterruptIn YButton(PTC12);                                                     //Set contrast 
DigitalIn AButton(PTB9);                                                        //Submit Joystick Position 
DigitalIn BButton(PTD0);                                                        //As Yet Unused
BusOut LedStatus(PTA1, PTA2, PTC2, PTC3, PTC4, PTD3);                           
PwmOut Buzz(PTC10); 
AnalogIn Pot(PTB2);                                                            
Serial pc(USBTX,USBRX);
/*******************************************************************************
Initiate Variables and Voids 
*******************************************************************************/
volatile int G_RepeatNum;                                                       //Global variables denoted G_ 
volatile float G_TempR;                                                         // TempR = Temp read from sensor
volatile float G_TempAve;
volatile int G_MLayerFlag;                                                       // MenuLayer flags  = Maintains while loops whilst navigating menus 
volatile bool G_APressed;
volatile bool G_BPressed;
volatile int G_PageNum;
volatile int G_PagePos;
volatile int G_SpeciesSelect;

Timer MLayer1Time;                                                              //Create Flag reset timer to initiate sleep in inactivity 
Timer MLayer2Time;                                                              //      used extensively to ensure device is not on during periods of inactivity 
Timer ReviewTime; 
Timer PromptTime;                                                               

void SetContrast();
void initDevice();
void FillAnimal();
void OpenLayer1();
void OpenLayer2();
void PageOne();
void PageTwo();
void PageThree();
void PG2_0();
void PG2_1();
void PG2_2();
void PG2_3();
void PG2_4();
void PG3_0();
void PG3_1();
void PG3_2();
void PG3_3();
void PG3_4();
void TakeRdng();
void Display();
void HealthCheck();
void CloseMenu();
void RemPrompt();

/*struct Animal{                                                                Unsuccessful use of structs to store temperature ranges
    float UTemp,LTemp;                                                                  considered healthy for different species 
};

struct Dog{                                                                                                                             
    float UTemp = 39.2;
    float LTemp = 37.5;
};
struct Horse{
    float UTemp = 38.6;
    float LTemp = 37.5;
};
struct Rabbit{
    float UTemp = 40.0;
    float LTemp = 38.5;
};
struct Snake{
    float UTemp = 31.1;
    float LTemp = 23.8;
};*/

//******************************************************************************
void SetContrast()
{                                                                               //Move Potentiometer to required position and Press Y to Set  
    float PotValue = Pot.read();
    printf("Brightness %f\n",PotValue);
    lcd.setBrightness(PotValue);
}

void initDevice()
{
        pc.baud(9600);                                                          //Set compatible to Terminal software 
        lcd.init();                                                             //Use library initialisation function for onboard devices
        tmp102.init();
        joystick.init();                                                          
        StartButton.mode(PullDown);                                             //Set so rise is press and fall is release 
        BackButton.mode(PullDown);
        MenuOnButton.mode(PullDown);
        MenuOffButton.mode(PullDown);
        AButton.mode(PullDown);
        BButton.mode(PullDown);
        YButton.mode(PullDown);
        lcd.setBrightness(0.00);
        Buzz.period(1.0/587.0);                                                 //Play tones users can recognise as initialising the device 
        Buzz = 0.15;                                                            //      ready for use but screen not on
        wait(0.1);
        Buzz.period(1.0/392.0);                                                  
        Buzz = 0.5;                                                                                   
        wait(0.2);  
        Buzz.period(1.0/659.0);                                                  
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                             
        Buzz = 0;
        printf("Hardware Initialisation Complete \n");
}

void FillAnimal()
{
//    struct Animal Dog = {39.2,37.5};                                          Unsuccessful use of structs to store temperature ranges
    //float UTemp,LTemp;                                                                  considered healthy for different species 
/*     Animal Dog[2] = 
    {
        Dog.UTemp = 39.2;
        Dog.LTemp = 37.5;
    }
    Animal Horse[2] =
    {
        Horse.UTemp = 38.6;
        Horse.LTemp = 37.5;
    }
    Animal Rabbit[2] =
    {
        Rabbit.UTemp = 40.0;
        Rabbit.LTemp = 38.5;
    }
    Animal Snake[2] =
    {
        Snake.UTemp = 31.1;
        Snake.LTemp = 23.8;
    }*/
}

/******************************************************************************
Reading values fromthe joystick - direction converted to integer in joystick cpp file
1 corresponds to North in the Enumerator (from joystick header)
3 corresponds to East in the Enumerator (from joystick header)
5 correspsonds to South in the Enumerator (from joystick header)
7 corresponds to West in the Enumerator (from joystick header)


*******************************************************************************/
void OpenLayer1()
{
    G_MLayerFlag = 1;
    MLayer1Time.start();                                                          //MLayer1Time timer started in case this is first time
    MLayer1Time.reset();                                                          //    running and reset incase this is not first time running
    PageOne();
    wait(2.0);
    if (G_PageNum==1)
    {
        while(G_MLayerFlag==1)
        {
            //printf("G_MLayerFlag is True \n");
            Direction d = joystick.get_direction();                             //Upon each cycle of while, check joystick direction and buttons statuses
            //printf("Direction %i \n", d);                                             improvement could be made to up the reaction time to user input
            G_APressed = AButton.read();
            G_BPressed = BButton.read();
            printf("A Button Status : %d \n", G_APressed);
            wait(0.1);
            float TimeGone = MLayer1Time.read();
            
            if(TimeGone >= 10)                                                  //Set value for timer comparitor called TimeGone (Seconds)
            {
                CloseMenu();
            }
            else if(d==1)                                                       
            {
                printf("North Time is %.1f \n", TimeGone);
                G_PagePos = 1;                                                  
                G_SpeciesSelect = 1;
                PageTwo();                              
                MLayer1Time.reset();
                wait(0.05);
            }
            else if(d==3)                                                       
            {
                printf("East Time is %.1f \n", TimeGone);
                G_PagePos = 2;
                G_SpeciesSelect = 2;
                PageTwo();
                MLayer1Time.reset();
                wait(0.2);
            }
            else if(d==5)                                                       
            {
                printf("South Time is %.1f \n", TimeGone);
                G_PagePos = 3;
                G_SpeciesSelect = 3;
                PageTwo();
                MLayer1Time.reset();
                wait(0.05);            
            }
            else if(d==7)                                                       
            {
                printf("West Time is %.1f \n", TimeGone);
                G_PagePos = 4;
                G_SpeciesSelect = 4;
                PageTwo();
                MLayer1Time.reset();
                wait(0.05);
            }
            else if(TimeGone >=2)                                               //removes arrow from last previous joystick input after 2 seconds
            {
                PG2_0();
            }
            if(G_BPressed==1)                                                   //B button is used to reverse through menus 
            {
                if(G_PageNum==1)
                {
                    printf("Go to CloseMenu");
                    CloseMenu();
                }
                else if(G_PageNum==2)
                {
                    printf("Go to PageOne");
                    PageOne();   
                }
            }
        }
    }
    else if(G_PageNum==2)                                                       
    {
        OpenLayer2();                                                           //if MLayer2Flag == 1 should return to OpenLayer2 
    }
    else if(G_PageNum==0)
    {
        printf("Menue Closed");
    }
}
/*******************************************************************************
Page One used as a welcome screen to user
Plays welcome sounds when screen turns on and displays information about device
******************************************************************************/
void PageOne()
{
    float PotValue = Pot.read();
    lcd.setBrightness(PotValue);
    printf("Start Page");
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("  Veterinary  ",0,1);                                         
    lcd.printString("Thermometer",0,2);                                          
    lcd.printString("            ",0,3);                                         
    lcd.printString("EJH - Feb '22",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                 
    lcd.refresh();
        Buzz.period(1.0/523.0);                                                  
        Buzz = 0.15;
        wait(0.1);
        Buzz.period(1.0/659.0);                                                  
        Buzz = 0.5;                                                                                   
        wait(0.1);  
        Buzz.period(1.0/784.0);                                                  
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                            
        Buzz = 0;
    G_PageNum = 1;
    return; 
}

void PageTwo()                                                                  //Direct to different menu statements based on user selection
{
    G_PageNum = 2;
    if(G_PagePos==1)
    {
        PG2_1();
    }          
    else if(G_PagePos==2)
    {
        PG2_2();
    } 
    else if(G_PagePos==3)
    {
        PG2_3();
    } 
    else if(G_PagePos==4)
    {
        PG2_4();
    } 
    return;
}
void PG2_0()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("   Dog",0,1);                                         
    lcd.printString("   Horse",0,2);                                          
    lcd.printString("   Rabbit",0,3);                                         
    lcd.printString("   Snake",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    return;
}
void PG2_1()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("-> Dog",0,1);                                         
    lcd.printString("   Horse",0,2);                                          
    lcd.printString("   Rabbit",0,3);                                         
    lcd.printString("   Snake",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    if (G_APressed==1)
    {
        G_PageNum=2;
        G_MLayerFlag=2;
        Buzz.period(1.0/784.0);                                                 //Affirmation sound on user input 
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                              
        Buzz = 0;
        OpenLayer2();
    }
    else
    {
        return;
    }
}
void PG2_2()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("   Dog",0,1);                                         
    lcd.printString("-> Horse",0,2);                                          
    lcd.printString("   Rabbit",0,3);                                         
    lcd.printString("   Snake",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    if (G_APressed==1)
    {
        G_PageNum=2;
        G_MLayerFlag=2;
        Buzz.period(1.0/784.0);                                                 //Affirmation sound on user input 
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                             
        Buzz = 0;
        OpenLayer2();
    }
    else
    {
        return;
    } 
}
void PG2_3()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("   Dog",0,1);                                         
    lcd.printString("   Horse",0,2);                                          
    lcd.printString("-> Rabbit",0,3);                                         
    lcd.printString("   Snake",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    if (G_APressed==1)
    {
        G_PageNum=2;
        G_MLayerFlag=2;
        Buzz.period(1.0/784.0);                                                 //Affirmation sound on user input 
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                             
        Buzz = 0;
        OpenLayer2();
    }
    else
    {
        return;
    }   
}
void PG2_4()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("   Dog",0,1);                                         
    lcd.printString("   Horse",0,2);                                          
    lcd.printString("   Rabbit",0,3);                                         
    lcd.printString("-> Snake",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    if (G_APressed==1)
    {
        G_PageNum=2;
        G_MLayerFlag=2;
        Buzz.period(1.0/784.0);                                                 //Affirmation sound on user input
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                             
        Buzz = 0;
        OpenLayer2();
    }
    else
    {
        return;
    }    
}
/*******************************************************************************
Reading values fromthe joystick - direction converted to integer in joystick cpp file
1 corresponds to North in the Enumerator (from joystick header)
3 corresponds to East in the Enumerator (from joystick header)
5 correspsonds to South in the Enumerator (from joystick header)
7 corresponds to West in the Enumerator (from joystick header)
*******************************************************************************/
void OpenLayer2()
{
    wait(1.0);
    MLayer2Time.start();                                                        //MLayer2Time timer started in case this is first time
    MLayer2Time.reset();                                                        //    running and reset incase this is not first time running    
                                                            
    if(G_PageNum==2)
    {
        while(G_MLayerFlag==2)
        {
            //printf("G_MLayerFlag is True \n");
            Direction d = joystick.get_direction();                                 //Upon each cycle of while, check joystick direction and buttons statuses
            printf("Direction %i \n", d);
            bool G_APressed = AButton.read();
            bool G_BPressed = BButton.read();
            //printf("A Button Status : %d \n", G_APressed);
            wait(0.05);
            float TimeGone2 = MLayer2Time.read();
            
            if(TimeGone2 >= 10)                                                 //Set value for timer comparitor called TimeGone2 (Seconds)
            {
                OpenLayer1();
            }
            else if(d==1)                                                           
            {
                printf("North Time is %.1f \n", TimeGone2);
                G_PagePos = 5;
                PageThree();
                MLayer2Time.reset();
                wait(0.05);
            }
            else if(d==3)                                                           
            {
                printf("East Time is %.1f \n", TimeGone2);
                G_PagePos = 6;
                PageThree();
                MLayer2Time.reset();
                wait(0.05);
            }
            else if(d==5)                                                           
            {
                printf("South Time is %.1f \n", TimeGone2);
                G_PagePos = 7;
                PageThree();
                MLayer2Time.reset();
                wait(0.05);            
            }
            else if(d==7)                                                           
            {
                printf("West Time is %.1f \n", TimeGone2);
                G_PagePos = 8;
                PageThree();
                MLayer2Time.reset();
                wait(0.05);
            }
            else if(TimeGone2 >=2)
            {
                PG3_0();
            }
            else if(G_BPressed==1)                                              //Direct to previous menu
            {
                G_PageNum=1;
                OpenLayer1();
            }
        }
        if (G_MLayerFlag==3)
        {
            TakeRdng();
        }
    }
    else if(G_PageNum==1)
    {
        OpenLayer1();                                                           //if MLayer2Flag == 1 should return to OpenLayer2 
    }
}
//******************************************************************************

void PageThree()
{
    //G_PageNum = 3;
    if(G_PagePos==5)
    {
        PG3_1();
    }     
    else if(G_PagePos==6)
    {
        PG3_2();
    } 
    else if(G_PagePos==7)
    {
        PG3_3();
    } 
    else if(G_PagePos==8)
    {
        PG3_4();
    } 
    return;
}
void PG3_0()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("  1   Reading",0,1);                                         
    lcd.printString("  5   Readings",0,2);                                          
    lcd.printString("  20  Readings",0,3);                                         
    lcd.printString("  100 Readings",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    return;
}
void PG3_1()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("->1   Reading",0,1);                                         
    lcd.printString("  5   Readings",0,2);                                          
    lcd.printString("  20  Readings",0,3);                                         
    lcd.printString("  100 Readings",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    bool G_APressed = AButton.read();
    if (G_APressed==1)
    {
        G_RepeatNum = 1;
        G_MLayerFlag=3;
        Buzz.period(1.0/587.0);                                                 //Affirmation sound on user input
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                            
        Buzz = 0;
        //TakeRdng();
    }
    else
    {
        return;
    }   
}
void PG3_2()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("  1   Reading",0,1);                                         
    lcd.printString("->5   Readings",0,2);                                          
    lcd.printString("  20  Readings",0,3);                                         
    lcd.printString("  100 Readings",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    bool G_APressed = AButton.read();
    if (G_APressed==1)
    {
        G_RepeatNum = 5;
        G_MLayerFlag=3;
        Buzz.period(1.0/587.0);                                                 //Affirmation sound on user input
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                             
        Buzz = 0;
        //TakeRdng();
    }
    else
    {
        return;
    }
}
void PG3_3()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("  1   Reading",0,1);                                         
    lcd.printString("  5   Readings",0,2);                                          
    lcd.printString("->20  Readings",0,3);                                         
    lcd.printString("  100 Readings",0,4);                                      
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    bool G_APressed = AButton.read();
    if (G_APressed==1)
    {
        G_RepeatNum = 20;
        G_MLayerFlag=3;
        Buzz.period(1.0/587.0);                                                 //Affirmation sound on user input
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                             
        Buzz = 0;
        //TakeRdng();
    }
    else
    {
        return;
    }
}
void PG3_4()
{
    lcd.clear();                                                                
    lcd.drawLine(1,1,84,1,1);                                     
    lcd.printString("  1   Reading",0,1);                                         
    lcd.printString("  5   Readings",0,2);                                          
    lcd.printString("  20  Readings",0,3);                                         
    lcd.printString("->100 Readings",0,4);                                    
    lcd.drawLine(1,47,84,47,1);                                     
    wait(0.05);                                    
    lcd.refresh();
    bool G_APressed = AButton.read();
    if (G_APressed==1)
    {
        G_RepeatNum = 100;
        G_MLayerFlag=3;
        Buzz.period(1.0/587.0);                                                 //Affirmation sound on user input 
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                            
        Buzz = 0;
        //TakeRdng();
    }
    else
    {
        return;
    }   
}

/*******************************************************************************
Close screen as a result of inactivity or user input exiting menus 
Play close down tones for user affirmation
*******************************************************************************/
void CloseMenu()
{
    lcd.setBrightness(0.00);
    G_PageNum = 0;
    G_MLayerFlag = 0;
    lcd.clear();
    lcd.refresh();
        Buzz.period(1.0/784.0);                                                  
        Buzz = 0.5;                                                                                   
        wait(0.1); 
        Buzz.period(1.0/659.0);                                                  
        Buzz = 0.5;                                                                                   
        wait(0.1); 
        Buzz.period(1.0/523.0);                                                  
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                            
        Buzz = 0;
    printf("Menu Closed and Flag Reset \n");
}


void TakeRdng()
{
        float TempSum = 0;
        G_TempR = tmp102.get_temperature();                                     // read temperature save to variable TempR                                      
        for(int i=0; i<G_RepeatNum; i++) {                                      //change G_RepeatNum set by user in menu
        printf("Temp was %.2f \n", G_TempR);
        TempSum = TempSum + G_TempR;                                
        printf("TempSum = %.2f \n",TempSum);
        Buzz.period(1.0/523.0);                                                  
        Buzz = 0.15;
        wait(0.1);
        Buzz.period(1.0/659.0);                                                  
        Buzz = 0.5;                                                                                   
        wait(0.1);                                                              //Time between tones utilisaed as break between readings 
        Buzz = 0;
            if(G_BPressed==1)
            {
                G_RepeatNum = 0;
                OpenLayer2();
            }
        }
        G_TempAve = TempSum / G_RepeatNum;                                      //Average Temperature Calculation using the RepeatNum 
        HealthCheck();                                                          //      variable as number of elements in average 
}
enum SpeciesPage
{                                                                               //Enum used in case switching below defines states to switch between 
    SpeciesDog,
    SpeciesHorse,
    SpeciesRabbit,
    SpeciesSnake,
    SpeciesState,
};
void Display()
{
    printf("Display \n");
    int DisplayType = SpeciesState;
    int NextDisplayType = SpeciesState;
    printf("SpeciesType %i", G_SpeciesSelect);
    switch(DisplayType)
    {
        case SpeciesState:                                                      //uses global SpreciesSelect variable to determine range of healthy temperatures
        {
            if(G_SpeciesSelect == 1)
            {
                NextDisplayType = SpeciesDog;
            }
            else if(G_SpeciesSelect == 2)
            {
                NextDisplayType = SpeciesHorse;
            }
            else if(G_SpeciesSelect == 3)
            {
                NextDisplayType = SpeciesRabbit;
            }
            else if(G_SpeciesSelect == 4)
            {
                NextDisplayType = SpeciesSnake;
            }
        }
        case SpeciesDog:
        {
            lcd.clear();                                                        //Display relevant information for species previously selected by user 
            lcd.drawLine(1,1,84,1,1);
            lcd.printString("Dog Health",0,1);      
            char buffer[14];
            int length = sprintf(buffer,"T=%.3f 'C", G_TempAve);
            if (length <= 14)                                                   //check string fits on screen (14 lots of 6 wide characters)
                lcd.printString(buffer,0,2); 
            lcd.drawLine(1,47,84,47,1);
            lcd.refresh();
                /*if (G_TempAve >= Dog.LTemp and G_TempAve <= Dog.UTemp)
                {
                    lcd.printString("Temperature OK",0,3);      
                }*/
        }
        case SpeciesHorse:
        {                                                                       //Display relevant information for species previously selected by user 
            lcd.clear();
            lcd.drawLine(1,1,84,1,1);
            lcd.printString("Horse Health",0,1);      
            char buffer[14];
            int length = sprintf(buffer,"T=%.3f 'C", G_TempAve);
            if (length <= 14)                                                   //check string fits on screen (14 lots of 6 wide characters)
                lcd.printString(buffer,0,2); 
            lcd.drawLine(1,47,84,47,1);
            lcd.refresh();
                /*if (G_TempAve >= Horse.LTemp and G_TempAve <= Horse.UTemp)
                {
                    lcd.printString("Temperature OK",0,3);      
                }*/
        }
        case SpeciesRabbit:
        {                                                                       //Display relevant information for species previously selected by user 
            lcd.clear();
            lcd.drawLine(1,1,84,1,1);
            lcd.printString("Rabbit Health",0,1);      
            char buffer[14];
            int length = sprintf(buffer,"T=%.3f 'C", G_TempAve);
            if (length <= 14)                                                   //check string fits on screen (14 lots of 6 wide characters)
                lcd.printString(buffer,0,2); 
            lcd.drawLine(1,47,84,47,1);
            lcd.refresh();
                /*if (G_TempAve >= Rabbit.LTemp and G_TempAve <= Rabbit.UTemp)
                {
                    lcd.printString("Temperature OK",0,3);      
                }*/
        }
        case SpeciesSnake:
        {                                                                       //Display relevant information for species previously selected by user 
            lcd.clear();
            lcd.drawLine(1,1,84,1,1);
            lcd.printString("Snake Health",0,1);      
            char buffer[14];
            int length = sprintf(buffer,"T=%.3f 'C", G_TempAve);
            if (length <= 14)                                                   //check string fits on screen (14 lots of 6 wide characters)
                lcd.printString(buffer,0,2); 
            lcd.drawLine(1,47,84,47,1);
            lcd.refresh();
                /*if (G_TempAve >= Snake.LTemp and G_TempAve <= Snake.UTemp)
                {
                    lcd.printString("Temperature OK",0,3);      
                }*/
        }
    }
        /*printf("Average Reading = %.2f \n",G_TempAve);
        lcd.clear();
        char buffer[14];
        int length = sprintf(buffer,"T=%.3f 'C", G_TempAve);
        if (length <= 14)                                                       //check string fits on screen (14 lots of 6 wide characters)
            lcd.printString(buffer,0,1);           
        lcd.refresh();      */
}

/*******************************************************************************
waits for user inactivity then promts the user to keep device on else device 
will shut down 
*******************************************************************************/
void RemPrompt()
{
        printf("RP ");
        lcd.clear();
        lcd.drawLine(1,1,84,1,1);                                                                            
        lcd.printString("Device may",0,1);                              
        lcd.printString("sleep soon!",0,2);                                      
        lcd.printString("A to keep Temp",0,3);
        lcd.printString("B to main menu",0,4);                                    
        lcd.drawLine(1,47,84,47,1);        
        lcd.refresh(); 
        PromptTime.start();
        while (PromptTime<=10)
        {
            Display();
            if (G_APressed==1)
            {
                ReviewTime.reset();                                             //Time out function is reset when the user interacts with device 
                HealthCheck();
            }
            else if(G_BPressed==1) 
            {
                G_MLayerFlag =1;
                G_PageNum =1;
                OpenLayer1();
            }
        }   
        return; 
}                                                                               //Prompts user to allow screen to sleep
void HealthCheck()
{
    printf("HealthCheck \n");
    ReviewTime.start();                                                         //ReviewTime timer started in case this is first time
    Display();
    while(ReviewTime<=30)
    {
        bool G_APressed = AButton.read();
        bool G_BPressed = BButton.read();
        if(ReviewTime>=10)                                                      //Added Time out function after 10 seconds
        {
            RemPrompt();
        }
    }
    CloseMenu();                                                  
}

int main()
{ 
        initDevice();
        FillAnimal();                                                           //Initiate Hardware
        YButton.rise(&SetContrast);                                             //Interrupts connected to voids 
        StartButton.rise(&TakeRdng);
        MenuOnButton.rise(&OpenLayer1);
        MenuOffButton.rise(&CloseMenu);                                         //Obselete
}