/**********************************************************************************
* @file    main.cpp
* @author  Petr Dousa
* @version V0.1
* @date    09-March-2015
* @brief   Read value from pins and write it to serial.
*          Some settings can be set through serial.
*          Serial speed is set to 115200.
***********************************************************************************/

/**************************************************************************************************************************************************/
/*                                                                    Table how to find 0xE000                                                    */
/**************************************************************************************************************************************************/
/* Num.        |   15  |   14  |   13  |   12  ||   11  |   10  |   9   |   8   ||   7   |   6   |   5   |   4   ||   3   |   2   |   1   |   0   */
/* Num. of pin | PC_15 | PC_14 | PC_13 | PC_12 || PC_11 | PC_10 | PC_9  | PC_8  || PC_7  | PC_6  | PC_5  | PC_4  || PC_3  | PC_2  | PC_1  | PC_0  */
/* Num. in BIN |   1   |   1   |   1   |   0   ||   0   |   0   |   0   |   0   ||   0   |   0   |   0   |   0   ||   0   |   0   |   0   |   0   */
/* Num. in HEX |               E               ||               0               ||               0               ||               0               */
/**************************************************************************************************************************************************/

/* Includes ----------------------------------------------------------------------*/
#include "mbed.h"

/* Defines -----------------------------------------------------------------------*/

/* Function prototypes -----------------------------------------------------------*/

/* Variables ---------------------------------------------------------------------*/
int pin =0;             
int printout =0;           
bool start = true;      

//mbed - initialization of peripherals
PortIn myIOs(PortC, 0xFFFF);        // PC_all
Serial pc(SERIAL_TX, SERIAL_RX);    // initialize Serial to connect to PC
Ticker toggle_ticker;               // initialize ticker
DigitalOut led(LED1);               // initialize LED

/* Functions----------------------------------------------------------------------*/

/***********************************************************************************
* Function Name  : toggle.
* Description    : Read value from pins, blink with LED and send value to serial.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
void toggle()
{
    int meas,z;
    meas=myIOs.read();       // read pins

    led= !led;               //blink led
    if (meas == pin ) {      // If the value is as we want 
        start = true;        //start send data to Serial
    }
    if(start) {
        if(printout==1) {                      // write value in bin
            while(!pc.writeable());
            for (z =32768; z > 0; z >>= 1) {
                while(!pc.writeable());
                pc.printf(((meas & z) == z) ? "1" : "0");
            }
            while(!pc.writeable());
            pc.printf("\n");
        } else if(printout==2) {            // write value in decimal
            while(!pc.writeable());
            pc.printf("%d\n", meas);
        } else if(printout==3) {            // write value in oct
            while(!pc.writeable());
            pc.printf("%o\n", meas);
        } else if(printout==4) {            // write value in hex
            while(!pc.writeable());
            pc.printf("%x\n", meas);
        } else {                            // write value in all options
            while(!pc.writeable());
            pc.printf("DEC: %d, HEX: %x, OCT: %o, BIN: ", meas,meas,meas);
            for (z =32768; z > 0; z >>= 1) {
                while(!pc.writeable());
                pc.printf(((meas & z) == z) ? "1" : "0");
            }
            while(!pc.writeable());
            pc.printf("\n");
        }
    }
}

/***********************************************************************************
* Function Name  : flushSerialPort.
* Description    : Serial flush routine.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
void flushSerialPort()
{
    while(pc.readable())
        pc.getc();
    return;
}

/***********************************************************************************
* Function Name  : menu.
* Description    : Print menu to serial.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
void menu()
{
    while(!pc.writeable()); // wait for the serial to be available for sending data
    pc.printf("HELP - MENU\n");// send text to serial
    while(!pc.writeable());
    pc.printf("Data to PC are in dec format.\n");
    while(!pc.writeable());
    pc.printf("Input format: \"xx yy\", where xx is the setting and yy is the value.\n");
    while(!pc.writeable());
    pc.printf("01 y.yyy - set period [seconds] to send data to PC and start reading value, example: 01 0.01\n");
    while(!pc.writeable());
    pc.printf("02 0 - stop reading value, example: 02 0\n");
    while(!pc.writeable());
    pc.printf("03 yyyy - Wait for pin to get to high level. yyyy write in decimal, example: 03 2000\n");
    while(!pc.writeable());
    pc.printf("04 yy - print format yy: 00-default all, 01-bin, 02-dec, 03-oct, 04-hex, example: 04 00\n");
    while(!pc.writeable());
    pc.printf("end HELP\n");
}

/***********************************************************************************
* Function Name  : main.
* Description    : Main routine.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
int main()
{
    myIOs.mode(PullNone); // Modes: PullDown PullUp PullNone OpenDrain

    // variables to read from serials
    int prijData=0;
    int Data1=0;
    float Data2=0;

    //set serial
    pc.baud(115200);
    pc.printf("\nLogic sond.\n");
    menu(); //print menu

    toggle_ticker.detach();             // detach all functions from ticker
    toggle_ticker.attach(&toggle, 1);   // call function toggle every single second


    while(1) {
        //accepted data from serial
        prijData=pc.scanf("%d",&Data1);             // read number from serial
        if(prijData==1 && (Data1>=1 && Data1<=3)) { // test if number was read and if it's between 1 and 3
            prijData=pc.scanf("%f",&Data2);         // read float number from serial
            if(prijData==1) {
                prijData=2;         // set variable to set measure of values
            } else {
                flushSerialPort();  // clear serial port, discard all data in the buffer
                prijData=0;         // set variable to print menu
            }
        } else {
            flushSerialPort();      // clear serial port, discard all data in the buffer
            prijData=0;             // set variable to print menu
        }

        if(prijData==2) {
            if(Data1==1) {
                toggle_ticker.detach();                 // detach all functions from ticker
                toggle_ticker.attach(&toggle, Data2);   // Data2 seconds was passed, call function toggle
            } else if(Data1==2) {
                toggle_ticker.detach();                 // detach all functions from ticker
            } else if(Data1==3) {
                pin=(int)Data2;                         //set variables, which are used in toggle
                start = false;
            } else if(Data1==4) {
                printout=(int)Data2;                    //set variables to set printf, which are used in toggle
            }
        } else {
            toggle_ticker.detach();                     // detach all functions from ticker
            menu();                                     // print menu
            flushSerialPort();                          // clear serial port, discard all data in the buffer
        }
    }
}
