/**********************************************************************************
* @file    main.cpp
* @author  Petr Dousa
* @version V1.00
* @date    22-March-2015
* @brief   Read analog value and write it to serial.
*          Some settings can be set through serial.
*          Serial speed is set to 115200.
***********************************************************************************/

/**********************************************************************************/
/*                 Table of A/D pins on Nucleo F303 (LQFP64)                      */
/**********************************************************************************/
/*  LQFP64 pin   |   Nucleo pin   |   ST Pin   |   AD Number      |    Channel    */
/*       8       |      A5        |    PC_0    |      ADC12       |       6       */
/*       9       |      A4        |    PC_1    |      ADC12       |       7       */
/*      14       |      A0        |    PA_0    |      ADC1        |       1       */
/*      15       |      A1        |    PA_1    |      ADC1        |       2       */
/*      16       |      D1        |    PA_2    |      ADC1        |       3       */
/*      17       |      D0        |    PA_3    |      ADC1        |       4       */
/*      20       |      A2        |    PA_4    |      ADC2        |       1       */
/*      21       |      D13       |    PA_5    |      ADC2        |       2       */
/*      22       |      D12       |    PA_6    |      ADC2        |       3       */
/*      23       |      D11       |    PA_7    |      ADC2        |       4       */
/*      26       |      A3        |    PB_0    |      ADC3        |       12      */
/**********************************************************************************/


/* Includes ----------------------------------------------------------------------*/
#include "mbed.h"

/* Defines -----------------------------------------------------------------------*/
#define voltage 3300                        //voltage in mV, used in Nucleo kits
#define mv(x)       ((x*voltage)/0xFFFF)    // raw value to millvolts

/* Function prototypes -----------------------------------------------------------*/

/* Variables ---------------------------------------------------------------------*/
int voltages =0;            
bool start = true;          

//mbed - initialization of peripherals
AnalogIn analog_value(A0);      // initialize analog input A0
Serial pc(SERIAL_TX, SERIAL_RX);// initialize Serial to connect to PC
DigitalOut led(LED1);           // initialize LED
Ticker toggle_ticker;           // initialize ticker

/* Functions----------------------------------------------------------------------*/

/***********************************************************************************
* Function Name  : toggle.
* Description    : Read analog value, compare, blink with LED and send value to serial.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
void toggle()
{
    int meas;

    meas = analog_value.read_u16(); // Converts and reads the analog input value (value from 0.0 to 0xFFFF)
    meas = mv(meas);                // Changes the value to be in the 0 to 3300 range (millivolts)


    led= !led;
    if (meas > voltages ) {         // If the value is greater than set voltage
        start = true;               //start send data to Serial
    }
    if(start) {
        while(!pc.writeable());     // wait to be serial available for sending data
        pc.printf("%d\n", meas);    // send data to Serial
    }
}

/***********************************************************************************
* Function Name  : flushSerialPort.
* Description    : Serial flush routine.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
void flushSerialPort()
{
    while(pc.readable())
        pc.getc();
    return;
}

/***********************************************************************************
* Function Name  : menu.
* Description    : Print menu to serial.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
void menu()
{
    while(!pc.writeable()); // wait to be serial available for sending data
    pc.printf("HELP - MENU\n"); // send text to Serial
    while(!pc.writeable());
    pc.printf("Data sent to PC are in millivolts(mV).\n");
    while(!pc.writeable());
    pc.printf("Input format: \"xx yy\", where xx is the setting and yy is the value.\n");
    while(!pc.writeable());
    pc.printf("01 y.yyy - set period [seconds] to send data to PC and start reading value, example: 01 0.01\n");
    while(!pc.writeable());
    pc.printf("02 0 - stop reading value, example: 02 0\n");
    while(!pc.writeable());
    pc.printf("03 yyyy - Wait for minimum voltage [mV] (to 3299) and send data, example:03 2000\n");
    while(!pc.writeable());
    pc.printf("end HELP\n");
}

/***********************************************************************************
* Function Name  : main.
* Description    : Main routine.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
int main()
{
    // variables to read from serials
    int prijData=0;     // check if data are set correctly  
    int Data1=0;        // first data from serial
    float Data2=0;      // second data from serial

    pc.baud(115200);    //set serial speed to 115200

    toggle_ticker.detach();             //detach all functions from ticker
    toggle_ticker.attach(&toggle, 1);   // 1 second passed, call function toggle

    pc.printf("\nAnalogIn example, data (numbers) are in mV.\n");
    menu(); //print menu

    while(1) {

        //accepted data from serial
        prijData=pc.scanf("%d",&Data1);             // read number from serial
        if(prijData==1 && (Data1>=1 && Data1<=3)) { // test if number was read and if it is between 1 and 3
            prijData=pc.scanf("%f",&Data2);         // read float number from serial
            if(prijData==1) {
                prijData=2;         // set variable to set measure of values
            } else {
                flushSerialPort();  // clear serial port, discard all data in the buffer
                prijData=0;         // set variable to print menu
            }
        } else {
            flushSerialPort();      // clear serial port, discard all data in the buffer
            prijData=0;             // set variable to print menu
        }

        if(prijData==2) {
            if(Data1==1) {                              // change time to call function toggle
                toggle_ticker.detach();                 //detach all functions from ticker
                toggle_ticker.attach(&toggle, Data2);   // Data2 seconds was passed, call function toggle
            } else if(Data1==2) {                       // stop call function toggle
                toggle_ticker.detach();                 //detach all functions from ticker
            } else if(Data1==3) {                       //set variables, which are use in toggle
                voltages=(int)Data2;
                start = false;
            }
        } else {
            menu();             // print menu
            flushSerialPort();  // clear serial port, discard all data in the buffer

        }
    }
}
