/* Demo BLE service
 * Copyright (c) 2006-2013 ARM Limited
 * Copyright (c) 2015 Adam Berlinger
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __BLE_DEMOAPP_SERVICE_H__
#define __BLE_DEMOAPP_SERVICE_H__

#include "Stream.h"

#include "UUID.h"
#include "BLEDevice.h"
#include "Utils.h"

typedef void (*DemoAppCallback)(uint8_t event);

class DemoAppService {
    public:
        static const uint8_t EVENT_SLIDER1_CHANGED = 0x01;
        static const uint8_t EVENT_SLIDER2_CHANGED = 0x02;
        static const uint8_t EVENT_BUTTON_CHANGED = 0x04;
    
        static const uint8_t ServiceUUID[LENGTH_OF_LONG_UUID];
        static const uint8_t slider1CharacteristicUUID[LENGTH_OF_LONG_UUID];
        static const uint8_t slider2CharacteristicUUID[LENGTH_OF_LONG_UUID];
        static const uint8_t buttonCharacteristicUUID[LENGTH_OF_LONG_UUID];
    protected:
        BLEDevice &ble;
        
        DemoAppCallback eventCallback;
        GattCharacteristic slider1Characteristic;
        GattCharacteristic slider2Characteristic;
        GattCharacteristic buttonCharacteristic;
        
        uint16_t lastButtonPressed;
        uint16_t slider1Value;
        uint16_t slider2Value;
    public:
        DemoAppService(BLEDevice &_ble) :
            ble(_ble),
            eventCallback(NULL),
            slider1Characteristic(slider1CharacteristicUUID, (uint8_t*)&slider1Value, 2, 2,
                         GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE),
            slider2Characteristic(slider2CharacteristicUUID, (uint8_t*)&slider2Value, 2, 2,
                         GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE),
            buttonCharacteristic(buttonCharacteristicUUID, (uint8_t*)&lastButtonPressed, 2, 2,
                         GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE){
                             
            GattCharacteristic *charTable[] = {&slider1Characteristic, &slider2Characteristic, &buttonCharacteristic};
            GattService         demoService(ServiceUUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));
            ble.addService(demoService);
            ble.onDataWritten(this, &DemoAppService::onDataWritten);
        }
        
        void setCallback(DemoAppCallback callback){eventCallback = callback;}
        
        void waitForEvent(void){ble.waitForEvent(); }
        
        virtual void onDataWritten(const GattCharacteristicWriteCBParams *params) {
            uint8_t event = 0;
            DEBUG("Demo service onWrite\n\r");
            if (params->charHandle == slider1Characteristic.getValueAttribute().getHandle()) {
                uint16_t bytesRead = params->len;
                if (bytesRead == 2) {
                    memcpy(&slider1Value, params->data, 2);
                    DEBUG("Slider1: %d\n\r", slider1Value);
                    event |= EVENT_SLIDER1_CHANGED;
                }
            }
            else if (params->charHandle == slider2Characteristic.getValueAttribute().getHandle()) {
                uint16_t bytesRead = params->len;
                if (bytesRead == 2) {
                    memcpy(&slider2Value, params->data, 2);
                    DEBUG("Slider2: %d\n\r", slider2Value);
                    event |= EVENT_SLIDER2_CHANGED;
                }
            }
            else if (params->charHandle == buttonCharacteristic.getValueAttribute().getHandle()) {
                uint16_t bytesRead = params->len;
                if (bytesRead == 2) {
                    memcpy(&lastButtonPressed, params->data, 2);
                    event |= EVENT_BUTTON_CHANGED;
                }
            }
            if(event && eventCallback){
                eventCallback(event);
            }
        }
        
        uint16_t getSlider1Value()const{return slider1Value;}
        uint16_t getSlider2Value()const{return slider2Value;}
};

DemoAppService *startDemoBLE(const char* name);

#endif