/**********************************************************************************
* @file    main.cpp
* @author  Petr Douša
* @version V1.0
* @date    20-April-2015
* @brief   This program uses digital output (software) and hardware PWM output
*          to generate two separate PWMs
***********************************************************************************/

/**********************************************************************************/
/*                    Table of PWM pins on Nucleo F303 (LQFP64)                   */
/**********************************************************************************/
/*  LQFP64 pin   |   Nucleo pin   |   ST Pin   |    PWM number    |    Channel    */
/*      15       |      A1        |    PA_1    |      PWM15       |       1       */
/*      26       |      A3        |    PB_0    |      PWM1        |       2       */
/*       9       |      A4        |    PC_1    |      PWM1        |       2       */
/*       8       |      A5        |    PC_0    |      PWM1        |       1       */
/*      17       |      D0        |    PA_3    |      PWM15       |       2       */
/*      16       |      D1        |    PA_2    |      PWM15       |       1       */
/*      43       |      D2        |    PA_10   |      PWM1        |       3       */
/*      55       |      D3        |    PB_3    |      PWM8        |       1       */
/*      57       |      D4        |    PB_5    |      PWM17       |       1       */
/*      56       |      D5        |    PB_4    |      PWM16       |       1       */
/*      41       |      D7        |    PA_8    |      PWM1        |       1       */
/*      42       |      D8        |    PA_9    |      PWM1        |       2       */
/*      38       |      D9        |    PC_7    |      PWM3        |       2       */
/*      58       |      D10       |    PB_6    |      PWM16       |       1       */
/*      23       |      D11       |    PA_7    |      PWM15       |       2       */
/*      22       |      D12       |    PA_6    |      PWM15       |       1       */
/*      62       |      D14       |    PB_9    |      PWM17       |       1       */
/*      61       |      D15       |    PB_8    |      PWM16       |       1       */
/*      46       |                |    PA_13   |      PWM16       |       1       */
/*      49       |                |    PA_14   |      PWM8        |       2       */
/*      59       |                |    PB_7    |      PWM17       |       1       */
/*       2       |                |    PC_13   |      PWM1        |       1       */
/*      10       |                |    PC_2    |      PWM1        |       3       */
/*      11       |                |    PC_3    |      PWM1        |       4       */
/*      40       |                |    PC_9    |      PWM3        |       4       */
/*      39       |                |    PC_8    |      PWM3        |       3       */
/*      37       |                |    PC_6    |      PWM3        |       1       */
/*      45       |                |    PA_12   |      PWM16       |       1       */
/*      44       |                |    PA_11   |      PWM1        |       4       */
/*      27       |                |    PB_1    |      PWM1        |       3       */
/*      36       |                |    PB_15   |      PWM17       |       1       */
/*      35       |                |    PB_14   |      PWM16       |       1       */
/**********************************************************************************/      

/* Includes ----------------------------------------------------------------------*/
#include "mbed.h"

/* Defines -----------------------------------------------------------------------*/

/* Function prototypes -----------------------------------------------------------*/
void toggleOff(void);

/* Variables ---------------------------------------------------------------------*/
int on_delay = 0;
int off_delay = 0;

//mbed - initialization of peripherals
Timeout timer;              // timer used with digital IO 
DigitalOut my_pwm(LED1);    // digital IO used by pwm_io function
PwmOut      real_pwm(PA_8); // regular PWM output

/* Functions----------------------------------------------------------------------*/

/*******************************************************************************
* Function Name  : toggleOn.
* Description    : Set high level on digital pin and timer startup.
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void toggleOn(void)
{
    my_pwm = 1;
    timer.attach_us(toggleOff, on_delay);
}

/*******************************************************************************
* Function Name  : toggleOff.
* Description    : Set low level on digital pin and set and start timer.
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void toggleOff(void)
{
    my_pwm = 0;
    timer.attach_us(toggleOn, off_delay);
}

/*******************************************************************************
* Function Name  : pwm_io.
* Description    : Set variables and start software PWM on digital output.
* Input          : p_us: signal period in micro_seconds, dc: signal duty-cycle (0.0 to 1.0) .
* Output         : on_delay: output ON time, off_delay: output OFF time
* Return         : None.
*******************************************************************************/
void pwm_io(int p_us, float dc)
{
    timer.detach();                 // shutdown timer
    if ((p_us == 0) || (dc == 0)) { // if duty-cycle is 0
        my_pwm = 0;                 // set high IO to low
        return;
    }
    if (dc >= 1) {                  // if duty-cycle is 1 and more
        my_pwm = 1;                 // set high IO to low
        return;
    }
    on_delay = (int)(p_us * dc);    // calculate the time
    off_delay = p_us - on_delay;
    toggleOn();                     // call function toggleOn to start PWM
}

/***********************************************************************************
* Function Name  : main.
* Description    : Main routine.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
int main()
{
    double a=0;                 // variable to set duty-cycle

    real_pwm.period_us(2000);   // set period of hardware PWM to 2 milliseconds
    real_pwm.write(a);          // set duty-cycle from a (0 at this point)


    pwm_io(2000, a);            // set period of software PWM to 2 ms and duty-cycle as in variable a (0 at this point)


    while(1) {                  // main cycle of the program

        if(a>=1) {              // if variable is grater then 1 set it 0 else increment 0.01
            a=0;
        } else {
            a=a+0.01;
        }

        real_pwm.write(a);      // set duty-cycle of hardware PWM as as it is in variable a

        pwm_io(2000, a);        // set duty-cycle of software PWM as as it is in variable a and period to 2 ms

        wait(0.05);             // wait some time in milliseconds
    }
}
