/**********************************************************************************
* @file    main.cpp
* @author  Daniel Toms
* @version V0.1
* @date    09-March-2015
* @brief   Simple ADC convertor example for Nucleo without DMA.
*          The value is written to the serial port every second at 115200baud.
*          Compatible with all the NUCLEOs.
***********************************************************************************/

/**********************************************************************************/
/*                 Table of A/D - D/A pins on Nucleo F303 (LQFP64)                */
/**********************************************************************************/
/*  LQFP64 pin   |   Nucleo pin   |   ST Pin   |   AD/DA Number   |    Channel    */
/*       8       |      A5        |    PC_0    |      ADC12       |       6       */
/*       9       |      A4        |    PC_1    |      ADC12       |       7       */
/*      14       |      A0        |    PA_0    |      ADC1        |       1       */
/*      15       |      A1        |    PA_1    |      ADC1        |       2       */
/*      16       |      D1        |    PA_2    |      ADC1        |       3       */
/*      17       |      D0        |    PA_3    |      ADC1        |       4       */
/*      20       |      A2        |    PA_4    |      ADC2        |       1       */
/*      21       |      D13       |    PA_5    |      ADC2        |       2       */
/*      22       |      D12       |    PA_6    |      ADC2        |       3       */
/*      23       |      D11       |    PA_7    |      ADC2        |       4       */
/*      26       |      A3        |    PB_0    |      ADC3        |       12      */
/**********************************************************************************/      
/*      20       |      A2        |    PA_4    |      DAC1        |       1       */
/*      21       |      D13       |    PA_5    |      DAC1        |       2       */
/**********************************************************************************/

/* Includes ----------------------------------------------------------------------*/
#include "mbed.h"

/* Defines -----------------------------------------------------------------------*/
// Calculate the corresponding acquisition measure for a given value in mV
#define rawmeas(x) ((0xFFFF*x)/3300)
// Calculate the corresponding mV value from measured voltage
#define mv(x)       ((x*3300)/0xFFFF)

/* Function prototypes -----------------------------------------------------------*/
 
/* Variables ---------------------------------------------------------------------*/

//mbed - initialization of peripherals
AnalogIn analog_value(A0);
Serial pc(SERIAL_TX, SERIAL_RX);
DigitalOut myled(LED1);
 
/* Functions----------------------------------------------------------------------*/


/***********************************************************************************
* Function Name  : main.
* Description    : Main routine.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/ 
int main() {
    
    pc.baud(115200);        //Sets the serial port baud rate to 115200baud.
    
    while(1) {                                      // infinite loop          
        uint16_t meas = analog_value.read_u16();    // Converts and read the analog input value
        if (meas > rawmeas(1000)) {                 // If the value is greater than 1000 mV toggle the LED
          myled = !myled;                           // toggles the LED
        }
        
        pc.printf("Measured value: %d\n", meas);    // Prints the measured value in millivolts
        pc.printf("%d mV\n", mv(meas));
        
        wait(1);                                    // Waits for one second between measures.
                                                    // You can use wait_ms(x) and wait_us(x) to wait for shorter periods.
        
    }
}
