/**********************************************************************************
* @file    main.cpp
* @author  Marta Krepelkova
* @version V0.1
* @date    22-March-2015
* @brief   Simply UART communication for STM32F0 Discovery kit.
*          Received character is sent back to your computer.
***********************************************************************************/

/**********************************************************************************/
/*   Table of used pins on STM32F0 Discovery kit with STM32F051R8 MCU (LQFP64)    */
/**********************************************************************************/
/*  LQFP64 pin   | Discovery pin  | ST Nucleo F030R8 pin  |      peripheral       */
/*      42       |     PA_9       |         PA_9          |      SERIAL_TX        */
/*      43       |     PA_10      |         PA_10         |      SERIAL_RX        */
/*      39       |     PC_8       |         PA_5          |      LED              */
/*      40       |     PC_9       |                       |      LED              */
/**********************************************************************************/

/* Includes ----------------------------------------------------------------------*/
#include "mbed.h"

/* Defines -----------------------------------------------------------------------*/

/* Function prototypes -----------------------------------------------------------*/

/* Variables ---------------------------------------------------------------------*/
char buffer[255];               // for receiving more characters from the computer
int received=0;                 // how many characters were received from computer
int sent=0;                     // how many characters were sent to computer

// mbed - initialization of peripherals
Serial pc(PA_9, PA_10);         // initialize SERIAL_TX=PA_9, SERIAL_RX=PA_10
DigitalOut blue(PC_8);          // initialize blue LED on STM32F0 discovery
DigitalOut green(PC_9);         // initialize green LED on STM32F0 discovery

/* Functions----------------------------------------------------------------------*/

/*******************************************************************************
* Function Name  : serialRx.
* Description    : Saves all received characters to the buffer.
* Input          : None
* Output         : None.
* Return         : None
*******************************************************************************/

void serialRx()
{
    while(pc.readable()) {              // while there is a character to read from the serial port.
        char c=pc.getc();               // receive the charracter
        buffer[received++]=c;           // save the charracter to the next place in buffer, increments number of received charactbers
    }
}

/***********************************************************************************
* Function Name  : main.
* Description    : Main routine.
* Input          : None.
* Output         : None.
* Return         : None.
***********************************************************************************/
int main()
{   
//--------------------------------------
// Hyperterminal configuration is default
// 9600 bauds, 8-bit data, no parity
//--------------------------------------
// Communication settings:
// pc.format(bits,parity,stop_bits)
//      bits: 5-8
//      parity: SerialBase::None, SerialBase::Odd, SerialBase::Even, SerialBase::Forced1, SerialBase::Forced0
//      stop_bits: 1 or 2
// pc.baudrate(baud)
//      baud: The desired baudrate (speed)
//--------------------------------------
// Example for default settings:
// pc.format(8,SerialBase::None,1);   
// pc.baud(9600);                
//--------------------------------------

    int i = 1;                                                                          // increments every second
    pc.printf("Program started !\r\n");                                                 // text displayed on a computer
    pc.attach(&serialRx,Serial::RxIrq);                                                 // Attach a function serialRx to be called whenever a serial interrupt is generated
    while(1) {
        while(sent<received) {                                                          // while the number of received characters is greater than the number of sent characters
            pc.printf("Received char: %c (%d).\r\n", buffer[sent],(int)buffer[sent]);   // send the character and the character number
            blue = !blue;                                                               // indicate this by LED - inverse value of blue LED
            sent++;                                                                     // increment number of sent charracters
            if(sent>received) {                                                         // if it sent all characters
                received=0;                                                             // number of received charracters is 0
                sent=0;                                                                 // number of sent charracters is 0
            }
        }
        wait(1);                                                                        // wait 1 second
        pc.printf("This program runs since %d seconds.\r\n", i++);                      // sends how long is the program running
        green = !green;                                                                 // inverse value of green LED
    }
}