#include "mbed.h"
#include <string.h>
#include "UART_Radar.h"
#include "buzzer.h"

#define TRIG_1  PA_8
#define ECHO_1  PC_7

Serial pc(USBTX, USBRX); //reception --- tx, rx

//Instanciation des GPIOs.
DigitalOut trig(TRIG_1);            //Sortie digital pour le trigger.
InterruptIn echo(ECHO_1);           //Entrée digital sur interuption pour l'écho.
InterruptIn position(USER_BUTTON);    //Entrée digital sur interuption pour le levier de vitesse.

//Instanciation du Timer.
Timer distance_time;            //Timer pour la mesure de distance.
Ticker mesure_time;             //Timer (fixe) avec interruption pour l'acquisition de la distance.
Ticker send_time;               //Timer (fixe) avec interruption pour l'envoie de la distance.
Ticker bip;                     //Timer (fixe) avec interruption pour l'actualisation du buzzer.

//Instanciation des variables.
Buzzer buzzo;
int distance;


/********************Fonction qui génére le burst pour la mesure de distance.****************************/
//But : Générer un burst et reset le timer de sitance.
//
//Donnée :  Le timer distance_time est un timer global.  
//          trig est un sortie digital. 
//                   _____________
// TRIG => ________/              \__....
//          2 µs         10 µs       
//   
void send_burst(){ 
    distance_time.reset(); 
    trig = 0;
    wait_us(2);
    trig = 1;
    wait_us(10);
    trig = 0;
}
/*******************************************************************************************************/




/*******************************Fonction de minimum et maximum.*****************************************/
//But : Renvoie le min et le max entre deux entier.
//
int max (int a, int b) {
  return (a<b)?b:a;
}
int min (int a, int b) {
  return (a<b)?a:b;
}
/********************************************************************************************************/


/******************Routine d'interruption sur front montant du signal echo.*****************************/
//But : Lancer le timer.
//
//Donnée :  Le timer distance_time est un timer global.  
//                  _....
// ECHO => ....____/      Start timer.
//                 ^      
//
void echo_rise(){ 
    distance_time.start(); 
}
/*********************************************************************************************************/



/***************************Routine d'envoie de la distance via l'uart.***********************************/
//But : Transformer la distance en chaine de caractéres.
//      Envoier via la fonction send() et attendre un acquitement.
//      Error si on recoie 3 non-acquitement succésif ou atteinte du timeout.
//
//Donnée :  L'entier distance est une variable global.
//          La chaine message contient la distance sur 3 caractére + '\0'.        
//          L'envoie est répété 3 fois quand il y a erreur. Si 3 erreurs, arret des émission.
//          Il y a erreur si :  receive ne recoie pas d'acquitement dans un temps impartie.  
//                              controle renvoie 0, qui corespond à une erreur de checksum.
void transmit_data(){
    
    int essai, err;
    char message[4];
    char trame[2];
    message[0]=48+(distance/100)%10;
    message[1]=48+(distance/10)%10;
    message[2]=48+distance%10;
    message[3]=0;
        
    essai=0;
    send(message);
    
    
    pc.printf("%c%c%c\n\r",message[0],message[1],message[2]);
    err=receive(trame);                       //Bug avec le nouvelle affichage
    pc.printf("error%d\n\r",err);
 
    while((controle(trame)==1 || err!=0) && essai < 3){
        essai++;
        if( essai >= 3) while(1);
        send(message);
        err=receive(trame);
        pc.printf("error%d\n\r",err);
        
    }
}
/**********************************************************************************************************/




/*********************Routine d'interruption sur front descendant du signal echo.**************************/
//But : Stoper le timer, récupérer la distance en cm
//      et modifier le silence entre deux bip du buzzer
//
//Donnée :  Le timer distance_time est un timer global.
//          L'entier distance est une variable global.
//          Le buzzer buzzo est un objet global.    
//                  _____________________________________
// ECHO => ________/      Distant = Temps / 29/2         \__.... Stop timer.
//                                                       ^
//       
void echo_fall(){ 
    distance_time.stop();                               //Stop le timer.
    
    distance = distance_time.read_us() /29 / 2;    //Converti le temps (µs) en distance (cm).
    distance = min(distance,230);             //Definie la valeur max de la distance à 230.
    distance = max(distance,0);                    //Definie la valeur min de la distance à 0.
            
    buzzo.changeMode(distance);                         //Mise à jour de l'intervale entre chaque bip
}
/**********************************************************************************************************/

/******************Routine d'interruption sur front montant du signal pos.*****************************/
//But : Lancer les tâches.
//                  
//
void position_rise(){ 
       
        echo.rise(NULL);                            //Routine NULL sur interruption de echo en front montant.
        echo.fall(NULL);                            //Routine NULL sur interruption de echo en front descendant.
        mesure_time.detach();                       //Stop la routine send_burst.
        send_time.detach();                         //Stop la routine transmist_data.                              
        bip.detach();                               //Stop la routine l'actualisation du buzzer.
    
}
/*********************************************************************************************************/

/******************Routine d'interruption sur front descendant du signal pos.*****************************/
//But : Stop les tâches.
//                  
//
void position_fall(){ 
       
        echo.rise(&echo_rise);                      //Routine echo_rise sur interruption de echo en front montant.
        echo.fall(&echo_fall);                      //Routine echo_fall sur interruption de echo en front descendant.
        mesure_time.attach(&send_burst, 0.5);       //Lance send_burst toutes les 500ms.
        wait(0.1);                                  //Décalage entre send et transmit.
        send_time.attach(&transmit_data, 0.5);      //Lance transmist_data toutes les 500ms.
        wait(0.1);                                  //Décalage entre transmit et buzz.
        bip.attach_us(&buzzo, &Buzzer::run, 400);   //Lance l'actualisation du buzzer toutes les 400µs ( F = 2.5 kHz ).
    
}
/*********************************************************************************************************/



int main() {

        position.rise(&position_rise);                      //Routine position_rise sur interruption de position en front montant.
        position.fall(&position_fall);                      //Routine position_fall sur interruption de position en front descendant.

    while(1);

}