#ifndef MBED_ULTRASONIC_H
#define MBED_ULTRASONIC_H

#include "mbed.h"

/** ultrasonic class.
 *  Used for HC-SR04 ultrasonic range sensor
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "ultrasonic.h"
 *
 * ultrasonic usensor(PA_6, PA_7, 0.1, 0.25, &dist);
 *
 * int main() {
 *  //start mesuring the distance
 *  usensor.startUpdates();
 *  while(1){
 *      usensor.checkDistance();
 *   }
 * }
 * @endcode
 */
class ultrasonic
{
public:
    /**iniates the class with the specified trigger pin, echo pin, update speed and timeout
     * @param trigPin The "trigger" pin for the ultrasonic sensor.  This is DigitalOut.
     * @param echoPin The "echo" pin for the ultrasonic sensor.  This is InterruptIn.
     * @param updateSpeed How often the ultrasonic sensor pings and listens for a response.
     * @param timeout How long to wait before giving up on listening for a ping.
     **/
    ultrasonic(PinName trigPin, PinName echoPin, float updateSpeed, float timeout);
    /**iniates the class with the specified trigger pin, echo pin, update speed, timeout and method to call when the distance changes
     * @param trigPin The "trigger" pin for the ultrasonic sensor.  This is DigitalOut.
     * @param echoPin The "echo" pin for the ultrasonic sensor.  This is InterruptIn.
     * @param updateSpeed How often the ultrasonic sensor pings and listens for a response.
     * @param timeout How long to wait before giving up on listening for a ping.
     * @param onUpdate(int) The function to be called when the distance is detected to have changed.
     **/
    ultrasonic(PinName trigPin, PinName echoPin, float updateSpeed, float timeout, void onUpdate(int));
    /** returns the last measured distance**/
    int getCurrentDistance(void);
    /** auses measuring the distance**/
    void pauseUpdates(void);
    /** tarts mesuring the distance**/
    void startUpdates(void);
    /**attachs the method to be called when the distances changes
     * @param method(int) The function to be called when the distance is detected to have changed.
     */
    void attachOnUpdate(void method(int));
    /**
    * Method:
    * changes the speed at which updates are made
    * @param updateSpeed how often to update the ultrasonic sensor
    **/
    void changeUpdateSpeed(float updateSpeed);
    /**gets whether the distance has been changed since the last call of isUpdated() or checkDistance()**/
    int isUpdated(void);
    /**gets the speed at which updates are made**/
    float getUpdateSpeed(void);
    /**call this as often as possible in your code, eg. at the end of a while(1) loop,
    and it will check whether the method you have attached needs to be called**/
    void checkDistance(void);
private:
    DigitalOut _trig;
    InterruptIn _echo;
    Timer _t;
    Timeout _tout;
    int _distance;
    float _updateSpeed;
    int start;
    int end;
    volatile int done;
    void (*_onUpdateMethod)(int);
    void _startT(void);
    void _updateDist(void);
    void _startTrig(void);
    float _timeout;
    int d;
};
#endif