/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef __KL25Z_RTC_H
#define __KL25Z_RTC_H

#include "mbed.h"

// Usage:
// #include "mbed.h"
// #include "KL25Z_RTC.h"
// 
// DigitalOut myled(LED1);
// KL25Z_RTC rtc( 15);
// Serial pc(USBTX, USBRX);
// 
// void alm ( void);
// void sec ( void);
// 
// int main() {
//     
//     pc.baud( 230400);
//     pc.printf("RTC Management.\r\n");
// 
//     rtc.RTC_Start( &sec, &alm);
// 
//     while(1) {
//         // pc.printf("RTC [%0d].\r\n", rtc.RTC_Read());
//         wait( 1.0);
//     }
// }
// 
// void sec ( void)
// {
//     pc.printf("sec\r\n");
// }
// 
// void alm ( void)
// {
//     pc.printf("alrm\r\n");
// }    
//


class KL25Z_RTC
{
public:
    /** Constructor.
     * 
     * @param   alarm espressed in seconds
     */
    KL25Z_RTC( unsigned int alarm);
    
    /** Desctructor
    */
    ~KL25Z_RTC();
    
    /** Start the RTC module, using IRQ but without registering user defined callback functions.
     * Access to the elapsed time is possible using the "Read" methid.
     * The IRQ transaction is visible through "RTC_isIRQxxxxDone" methods.
     *
     * @param none
     * @return none
     */
    void RTC_Start( void);
    
    /** Start the module. Setting up the IRQ handler for the second IRQ and/or for the Alarm
     * @param   sec_ptr, alrm_ptr pointer to the IRQ handler
     * @return none
     */
    void RTC_Start( void(*sec_ptr)(void), void(*alrm_ptr)(void));
    
    /** Return the alarm value
     * @param   none
     * #return  the alarm value
     */
    unsigned int RTC_GetAlarm( void);
    
    /** Configure the alarm value
     * @param   alarm in seconds
     * @return  none
     */
    void RTC_SetAlarm( unsigned int alarm);
    
    /** Return the second elapsed
     * @param   none
     * @return  the second elapsed
     */
    unsigned int RTC_Read( void);

    /** Return the status of the IRQ for seconds elapsed.
     * Use this function for a polling like method, using the IRQ internally the module but without callbacks define.
     * The flag is reset after the read.
     *
     * @param   none
     * @return  1 if the IRQ is done. 
     */    
    unsigned int RTC_isIRQSecondDone( void);
    
    /** Return the status of the IRQ for the alarm.
     * Use this function for a polling like method, using the IRQ internally the module but without callbacks define.
     * The flag is reset after the read.
     *
     * @param   none
     * @return  1 if the IRQ is done.
     */        
    unsigned int RTC_isIRQAlarmDone( void);
    
private:
    
    static void _RTC_IRQHandler(void);
    static void _RTC_Seconds_IRQHandler(void);
        
};

#endif