/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "KL25Z_RTC.h"
#include "rtc_api.h"

unsigned int _alarm=0;                  // Value at which tha alarm is set.
volatile unsigned int _secIRQ_Done;
volatile unsigned int _alrmIRQ_Done;

void (*RTC_usr2_fptr)(void);               // Pointers to user function called after
void (*RTC_usr1_fptr)(void);               // IRQ assertion.


KL25Z_RTC::KL25Z_RTC( unsigned int alarm)
{
    if ( alarm != 0)
        _alarm = alarm;
}

KL25Z_RTC::~KL25Z_RTC() 
{
    NVIC_DisableIRQ( RTC_Seconds_IRQn);
    RTC_usr1_fptr = NULL;
    RTC->IER &= ~RTC_IER_TSIE_MASK;
    
    NVIC_DisableIRQ( RTC_IRQn);
    RTC_usr2_fptr = NULL;
    RTC->IER &= ~RTC_IER_TAIE_MASK;
    
    // Disable the clock to the RTC module.
    SIM->SCGC6 &= ~SIM_SCGC6_RTC_MASK;
}

unsigned int KL25Z_RTC::RTC_GetAlarm( void)
{
    return _alarm;
}

void KL25Z_RTC::RTC_SetAlarm( unsigned int alarm)
{
    _alarm = alarm;
}

void KL25Z_RTC::RTC_Start( void)
{
    rtc_init();
  
    NVIC_EnableIRQ( RTC_Seconds_IRQn);
    //NVIC_EnableIRQ( RTC_IRQn);   // causes startup problems?
    NVIC_SetVector( RTC_Seconds_IRQn, (uint32_t)&_RTC_Seconds_IRQHandler); 
    //NVIC_SetVector( RTC_IRQn, (uint32_t)&_RTC_IRQHandler); // causes startup problems?
 
    RTC->TAR = RTC->TSR + _alarm;
    RTC->IER = RTC_IER_TSIE_MASK | RTC_IER_TAIE_MASK;

}

void KL25Z_RTC::RTC_Start( void(*sec_ptr)(void), void(*alrm_ptr)(void))
{
    rtc_init();
    
    RTC->IER = 0;
    
    if ( sec_ptr != NULL) {
        NVIC_EnableIRQ( RTC_Seconds_IRQn);
        NVIC_SetVector( RTC_Seconds_IRQn, (uint32_t)&_RTC_Seconds_IRQHandler); 
        RTC_usr1_fptr = sec_ptr;
        RTC->IER |= RTC_IER_TSIE_MASK;
    } else {
        NVIC_DisableIRQ( RTC_Seconds_IRQn);
        RTC->IER &= ~RTC_IER_TSIE_MASK;
    }
    
    if ( alrm_ptr != NULL) {
        RTC->TAR = RTC->TSR + _alarm;
        NVIC_EnableIRQ( RTC_IRQn);
        NVIC_SetVector( RTC_IRQn, (uint32_t)&_RTC_IRQHandler); 
        RTC_usr2_fptr = alrm_ptr;
        RTC->IER |= RTC_IER_TAIE_MASK;
    } else {
        NVIC_DisableIRQ( RTC_IRQn);
        RTC->IER &= ~RTC_IER_TAIE_MASK;
    }
    
}

unsigned int KL25Z_RTC::RTC_Read( void)
{
    return RTC->TSR;
}

void KL25Z_RTC::_RTC_IRQHandler(void) 
{
    //
    if ( RTC->SR & 0x04) {
        // printf("RTC_Alarm\r\n");
        RTC->TAR = RTC->TSR + _alarm;
        // Run the user supplied function
        if ( RTC_usr2_fptr != NULL)
            RTC_usr2_fptr();
        // 
        _alrmIRQ_Done=1;
    }
    
}

void KL25Z_RTC::_RTC_Seconds_IRQHandler(void) 
{
    //
    _secIRQ_Done=1;
    
    // printf("RTC_Seconds_IRQHandler [%0d].\r\n", RTC->TSR);
    // Run the user supplied function
    if ( RTC_usr1_fptr != NULL)
        RTC_usr1_fptr();
}

unsigned int KL25Z_RTC::RTC_isIRQSecondDone( void)
{
    if ( _secIRQ_Done) {
        _secIRQ_Done=0;
        return 1;
    } else
        return 0;        
}

unsigned int KL25Z_RTC::RTC_isIRQAlarmDone( void)
{
    if ( _alrmIRQ_Done) {
        _alrmIRQ_Done=0;
        return 1;
    } else
        return 0;        
}

