#ifndef HMC6352_H
#define HMC6352_H

/**
 * Includes
 */
#include "mbed.h"

/**
 * Defines
 */
#define HMC6352_I2C_ADDRESS  (0x42>>1) //7-bit address. 0x43 write, 0x42 read.

//Commands.
#define HMC6352_EEPROM_WRITE 0x77
#define HMC6352_RAM_WRITE    0x47
#define HMC6352_RAM_READ     0x67
#define HMC6352_ENTER_CALIB  0x43
#define HMC6352_EXIT_CALIB   0x45
#define HMC6352_GET_DATA     0x41

//EEPROM locations.
#define HMC6352_OPMODE       0x08

//RAM registers.
#define HMC6352_RAM_OPMODE   0x74

//Operational mode register masks.
#define HMC6352_CM_MR_1HZ    0x00
#define HMC6352_CM_MR_5HZ    0x20
#define HMC6352_CM_MR_10HZ   0x40
#define HMC6352_CM_MR_20HZ   0x60

#define HMC6352_PERIODIC_SR  0x10

#define HMC6352_STANDBY      0x00
#define HMC6352_QUERY        0x01
#define HMC6352_CONTINUOUS   0x02

/**
 * Honeywell HMC6352 digital compass.
 */
class HMC6352 {

public:

    /**
     * Constructor.
     *
     * @param sda mbed pin to use for SDA line of I2C interface.
     * @param scl mbed pin to use for SCL line of I2C interface.
     */
    HMC6352(PinName sda, PinName scl);

    /**
     * Sample the device and return the result.
     *
     * @return In heading output mode, the current heading as a number between
     *         0-3599, representing 0-359.9 degrees.
     *         In raw magnetometer X output mode, the raw output of the X-axis
     *         magnetometer.
     *         In raw magnetometer Y mode, the raw output of the Y-axis
     *         magnetometer.
     *         In magnetometer X mode, the corrected output of the X-axis
     *         magnetometer.
     *         In magnetometer Y mode, the corrected output of the Y-axis
     *         magnetometer.
     */
    int sample(void);


    /**
     * Enter into or exit from calibration mode.
     *
     * @param enterOrExit 0x45 -> Exit
     *                    0x43 -> Enter
     */
    void setCalibrationMode(int enterOrExit);

    /**
     * Get the current operation mode.
     *
     * @return 0x00 -> Standby mode
     *         0x01 -> Query mode
     *         0x02 -> Continuous mode
     */
    int getOpMode(void);

    /**
     * Set the operation mode.
     *
     * @param mode 0x00 -> Standby mode
     *             0x01 -> Query mode
     *             0x02 -> Continuous mode
     * @param periodicSetReset 0x00 -> No periodic set/reset
     *                         0x01 -> Periodic set/reset
     * @measurementRate Measurement rate in Hz for continuous rate.
     *                  Possible rates: {1, 5, 10, 20}Hz.
     */
    void setOpMode(int mode, int periodicSetReset, int measurementRate = 0);

private:

    I2C* i2c_;
    int  operationMode_;

    /**
     * Write to EEPROM or RAM on the device.
     *
     * @param EepromOrRam 0x77 -> Writing to EEPROM
     *                    0x47 -> Writing to RAM
     * @param address Address to write to.
     * @param data Data to write.
     */
    void write(int EepromOrRam, int address, int data);

    /**
     * Read EEPROM or RAM on the device.
     *
     * @param EepromOrRam 0x72 -> Reading from EEPROM
     *                    0x67 -> Reading from RAM
     * @param address Address to read from.
     * @return The contents of the memory address.
     */
    int read(int EepromOrRam, int address);

};

#endif /* HMC6352_H */
