#ifndef Data_Types_BMU_H
#define Data_Types_BMU_H

#include "CAN_Data.h"
#include "CAN_IDs.h"
#include "mbed.h"

#define NO_CMUS 3
#define NO_READINGS_PER_CMU 12
#define NUM_MISSING_CELLS 3 //the number of cells short of 12 that are attached to the top CMU 
#define NO_TEMPERATURE_SENSORS 85
#define TEMPERATURE_MEASUREMENT_FREQ 4 //how often we measure temperature relative to the normal loop (note this isn't actually a frequency)

#define BATTERY_CAPACITY 42 //in Ah

// CAN pins
#define CAN_WRITE_PIN p29
#define CAN_READ_PIN p30

#define BATTERY_FAN_CONTROL_PIN p13

// EEPROM addresses
#define START_WRITE_ADDRESS 0x0040
 /* When writing to the eeprom, the data will be written to memory twice to reduce the chance of errors.
 The second set of data will be stored SECOND_ADDRESS_OFFSET after the first set of data. This should 
 be longer than than the number of bytes written to memory (otherwise data is overwritten) 
 but less than the page length */
#define SECOND_ADDRESS_OFFSET 20
#define MAX_WRITE_ADDRESS 0x7FC0

// Limit values

//Note: there are three categories of limiting values, for which the vehicle and driver response will differ:

//Dangerous errors, which will trigger a car shutdown and tell the driver to leave immediately

#define MAX_CELL_SAFE_VOLTAGE 4300 //mV
#define MIN_CELL_SAFE_VOLTAGE 2300 //mV
#define MAX_CELL_SAFE_TEMPERATURE 70
#define MAX_TEMPERATURE_DIFFERENCE 5 //this threshold also indicates a a dangerous situation i.e. suspected battery fire

//Moderate errors, which will trigger a car shutdown, but the driver can remain in the vehicle

#define MAX_CELL_VOLTAGE 4200 //mV
#define MIN_CELL_VOLTAGE 2500 //mV
#define MAX_CELL_DISCHARGE_TEMPERATURE 55
#define MIN_CELL_DISCHARGE_TEMPERATURE -20
#define OVERCURRENT_THRESHOLD 60000 //mA

//Low priority errors, for which the vehicle can continue driving, but some functionality may be reduced (e.g. array may be disabled)

#define MAX_CELL_CHARGE_VOLTAGE 4150 //mV
#define MAX_CELL_CHARGE_TEMPERATURE 45
#define MIN_CELL_CHARGE_TEMPERATURE 10

// Error states
#define NONE                                        0x00000000  
#define CELL_OVER_VOLTAGE                           0x00000001  //Moderate error
#define CELL_UNDER_VOLTAGE                          0x00000002  //Moderate error
#define CELL_OVER_CHARGE_TEMPERATURE                0x00000004  //Low priority error
#define CELL_OVER_DISCHARGE_TEMPERATURE             0x00000008  //Moderate error
#define CELL_UNDER_CHARGE_TEMPERATURE               0x00000010  //Low priority error
#define CELL_UNDER_DISCHARGE_TEMPERATURE            0x00000020  //Moderate error
#define MAX_TEMPERATURE_DIFFERENCE_ERROR            0x00000040  //Dangerous error
#define OVERCURRENT_ERROR                           0x00000080  //Moderate error
#define CELL_OVER_CHARGE_VOLTAGE                    0x00000100  //Low priority error
#define CELL_DANGEROUSLY_OVER_VOLTAGE               0x00000200  //Dangerous error
#define CELL_DANGEROUSLY_UNDER_VOLTAGE              0x00000400  //Dangerous error
#define CELL_DANGEROUSLY_OVER_TEMPERATURE           0x00000800  //Dangerous error

//CAN Buffer
#define CAN_BUFFER_SIZE 255 //Setting this to be quite large, should be equal to max # of ids expected to recieve

// Delay
#define LOOP_DELAY_S 1 // TODO is this delay the correct length?

struct individual_temperature{
    // Structure that will store a temperature value and also an ID 
    char ROMID[8]; //ROM array for temperature sensor, subset of it is useful as ID
    //ROMID 1-6 are the useful values for this case, ignore the rest
    float measurement;
    long ID; //This field is only filled when CAN message is decoded, just a re-encoding of ROMID
};

struct CMU_voltage{
    // Structure that contains voltage information suitable for CAN bus
    uint8_t ID; // CMU serial number
    uint8_t CMU_number; // Number CMU's from 0-(N-1) (N is number of CMU's) it will aid CAN communications the ID and CMU number may be the same not sure yet
    
    //to be removed
    //uint16_t first_cell_voltages[4]; // Split the cell voltages up easier to send over CAN
    //uint16_t last_cell_voltages[4];
    
    //this now
    uint16_t voltages[NO_READINGS_PER_CMU]; 
};

struct pack_voltage_extremes{
    // structure useful for CAN ID 0x6F8
    uint16_t voltage; // maximum or minimum voltage
    uint8_t CMU_number; // CMU number of extreme voltage
    uint8_t cell_number; // cell number of extreme voltage
};

struct pack_temperature_extremes{
    float temperature; // maxiumum or minimum temperature
    //uint8_t CMU_number; // CMU number of extreme voltage
    char ROMID[8]; //Since we arent tracking CMU numbers uses this instead
    unsigned long ID; //Same as temperature measurement, makes it much easier to compare probes.
};

struct BMU_data{
    // Data for the CAN bus see Tritium Communications Protocol
    // Cell voltage
    CMU_voltage cell_voltages[NO_CMUS]; // Store all voltage measurements in here
    pack_voltage_extremes max_cell_voltage;
    pack_voltage_extremes min_cell_voltage;
    uint32_t battery_voltage; // 5.9 Tritium data sheet
    float battery_current; //changed to a float because conversion is unnatural and pointless
    
    // SOC 
    float SOC;
    float percentage_SOC;
    
    // Cell Temperature
    individual_temperature temperature_measurements[NO_TEMPERATURE_SENSORS]; // Store all of the temperature measurements in here
    pack_temperature_extremes max_cell_temp;
    pack_temperature_extremes min_cell_temp;
    
    individual_temperature ambient_temperature;
    
    int32_t ivta_current;
};

#endif