#include "mbed.h"
#include <Timer.h>
#include "mbedtls/platform.h"
#include <string.h>
#include "mbedtls/cipher.h"
#include "mbedtls/entropy.h"
#include "mbedtls/ctr_drbg.h"
#include "cube_crypto.h"
#include "lib_CEBF746.h"
#include "lib_crc.h"
#include "CubeTPMHelper.h"



#if DEBUG_LEVEL > 0
#include "mbedtls/debug.h"
#endif
#if defined(MBEDTLS_MEMORY_BUFFER_ALLOC_C)
#include "mbedtls/memory_buffer_alloc.h"
#endif


#define sbi(buf,bit)     (buf) |= (1 << (bit)) 
#define cbi(buf,bit)     (buf) &= ~(1 << (bit))

 

#define PIN_TPM_SDA         PB_9        // SDA
#define PIN_TPM_SCL         PB_8        // SCL
#define PIN_TPM_RESET       PE_13       // RESET   
/*
 *  Global variables
 */
CubeTPMHelper TPM(PIN_TPM_SDA, PIN_TPM_SCL, PIN_TPM_RESET);

uint8_t cebf746_status_reg;
uint8_t cebf746_work_status_reg;


cube_sec_context Encrypt_ctx;
cube_sec_context Decrypt_ctx;

uint8_t spiTXBuf[255];     // SPI TX 踰꾪띁
uint8_t spiRXBuf[255];     // SPI RX 踰꾪띁
spiDataStr encryptedTxPacket; 
spiDataStr decryptedTxPacket; 


spiDataStr getEncTxPacket(void)
{
    //printf("%s:%d: getEncTxPacket()...\n", __FILE__, __LINE__);
    //print_cube_sec_context(Encrypt_ctx);
    SET_STATUS(Encrypt_ctx.cmd, ENCRYPT_STATUS, AES_ENCRYPT_OFF);
    return encryptedTxPacket;
}
spiDataStr getDecTxPacket(void)
{
    SET_STATUS(Decrypt_ctx.cmd, DECRYPT_STATUS, AES_DECRYPT_OFF);
    return decryptedTxPacket;
}
    
uint8_t cube_get_status(void)
{
    return Encrypt_ctx.cmd | Decrypt_ctx.cmd;
}

void cube_sec_init(void)
{
    int32_t ret;
    
    
    cube_sec_struct_init(&Encrypt_ctx);
    cube_sec_struct_init(&Decrypt_ctx);

    //key size 
    Encrypt_ctx.sec_key.key_size =32;


    if(!TPM.isInit())
    {
        printf("TPM isInit fail\n");
        return;
    }
    
    ret = cube_sec_key_reset(&Encrypt_ctx);
    if(ret != 0)
    {
        printf("cube_sec_key_reset fail\n");
        return;
    }
    
    memcpy(Decrypt_ctx.sec_key.key,Encrypt_ctx.sec_key.key , 32);
    memcpy(Decrypt_ctx.sec_key.iv,Encrypt_ctx.sec_key.iv , 16);
    Decrypt_ctx.sec_key.key_crc = Encrypt_ctx.sec_key.key_crc;
    Decrypt_ctx.sec_key.key_size = Encrypt_ctx.sec_key.key_size;
    mbedtls_aes_setkey_enc(&Encrypt_ctx.mbed_ctx, Encrypt_ctx.sec_key.key, 8*32);
    mbedtls_aes_setkey_dec(&Decrypt_ctx.mbed_ctx, Decrypt_ctx.sec_key.key, 8*32);
    
    print_cube_sec_key(Encrypt_ctx.sec_key);
    print_cube_sec_key(Decrypt_ctx.sec_key);
    
    printf("cube_sec_init [OK]\n");
    return;
    
}
void cube_sec_struct_init(cube_sec_context *cube_ctx)
{
    mbedtls_aes_init(&(cube_ctx->mbed_ctx)); 
    
    memset(cube_ctx->sec_key.key, 0, 32);
    memset(cube_ctx->sec_key.iv, 0, 16);
    memset(cube_ctx->sec_data.input_data, 0, 16);
    memset(cube_ctx->sec_data.output_data, 0, 16);
    cube_ctx->sec_key.key_crc = 0;
    cube_ctx->sec_key.key_size = 0;
    cube_ctx->sec_key.keyindex = 0;
    cube_ctx->sec_data.input_data_size = 0;
    cube_ctx->sec_data.output_data_size = 0;
    cube_ctx->cmd = 0;
    return; 
}

bool cube_sec_key_reset(cube_sec_context *cube_ctx)
{
    int32_t ret = 0;
    //printf("cube_sec_key_reset [start]\n");

    uint8_t pRandomNumKey[32]={0,};
    uint8_t pRandomNumIV[16]={0,};
    
    ret = cube_random_number_generate((unsigned char *)random_num_seed, pRandomNumKey, cube_ctx->sec_key.key_size);
    if(ret != 0)
    {
        printf("error cube_sec_key_reset %d \r\n", ret);
        return ret;
    }
    
    ret = cube_random_number_generate((unsigned char *)random_num_seed, pRandomNumIV, 16);
    if(ret != 0)
    {
        printf("error cube_sec_key_reset %d \r\n", ret);
        return ret;
    }

    ret = TPM.tpm_GenKey(1, (unsigned char *)pRandomNumKey, (unsigned char *)cube_ctx->sec_key.key);
    if(ret != 0)
    {
        printf("error tpm_GenKey %d \r\n", ret);
        return ret;
    }

    ret = TPM.tpm_GenIV(1, (unsigned char *)pRandomNumIV, (unsigned char *)cube_ctx->sec_key.iv);
    if(ret != 0)
    {
        printf("error tpm_GenKey %d \r\n", ret);
        return ret;
    }
    
    cube_crc_16(cube_ctx->sec_key.key_size, cube_ctx->sec_key.key, &(cube_ctx->sec_key.key_crc));
    print_cube_sec_key(cube_ctx->sec_key);
    return 0;
}

int32_t cube_AES_encrypt(cube_sec_context *cube_ctx)
{
    int32_t ret = 0;

    if(GET_STATUS(cube_ctx->cmd, MODE_STATUS) == AES_MODE_ECB)
    {
        printf("AES_MODE_ECB\n");
        ret = mbedtls_aes_crypt_ecb( &(cube_ctx->mbed_ctx),MBEDTLS_AES_ENCRYPT , cube_ctx->sec_data.input_data ,cube_ctx->sec_data.output_data);

    }
    else if(GET_STATUS(cube_ctx->cmd, MODE_STATUS) == AES_MODE_CBC)
    {
        ret = mbedtls_aes_crypt_cbc( &(cube_ctx->mbed_ctx), MBEDTLS_AES_ENCRYPT, cube_ctx->sec_data.input_data_size, cube_ctx->sec_key.iv, (const unsigned char *)cube_ctx->sec_data.input_data 
        ,(unsigned char *)cube_ctx->sec_data.output_data);
    }
    
    if(ret != 0)
    {
        printf("error cube_AES_decrypt %d \r\n", ret);
    }
    
    return ret;

};

int32_t cube_AES_decrypt(cube_sec_context *cube_ctx)
{
    int32_t ret = 0 ;
    
    if(GET_STATUS(cube_ctx->cmd, MODE_STATUS) == AES_MODE_ECB)
    {
        printf("AES_MODE_ECB\n");
        ret = mbedtls_aes_crypt_ecb( &(cube_ctx->mbed_ctx),MBEDTLS_AES_DECRYPT , cube_ctx->sec_data.input_data ,cube_ctx->sec_data.output_data);
    }
    else if(GET_STATUS(cube_ctx->cmd, MODE_STATUS) == AES_MODE_CBC)
    {
        ret = mbedtls_aes_crypt_cbc( &(cube_ctx->mbed_ctx), MBEDTLS_AES_DECRYPT, cube_ctx->sec_data.input_data_size, cube_ctx->sec_key.iv,  (const unsigned char *)cube_ctx->sec_data.input_data 
        ,(unsigned char *)cube_ctx->sec_data.output_data);
    }

    if(ret != 0)
    {
        printf("error cube_AES_decrypt %d \r\n", ret);
    }

    return ret;
}

bool cube_AES_read_data(cube_sec_context *cube_ctx, uint8_t *output_data, uint32_t size)
{
    bool ret = false;
    if(AES_WORK_OFF == GET_STATUS(cube_ctx->cmd, WORK_STATUS))
    {
        memcpy(output_data,&cube_ctx->sec_data.output_data, size);      
        SET_STATUS(cube_ctx->cmd, ENCRYPT_STATUS, AES_ENCRYPT_OFF);
        ret = true;
    }
    else
    {
        printf("thread busy\n");
    }
    
    return ret;
}

int32_t cube_random_number_generate(const uint8_t *seed, uint8_t *num, uint8_t size)
{
    int32_t ret = 0;
    /*
     * Setup random number generator
     * (Note: later this might be done automatically.)
     */
    mbedtls_entropy_context entropy;    /* entropy pool for seeding PRNG */
    mbedtls_ctr_drbg_context drbg;      /* pseudo-random generator */

    mbedtls_entropy_init(&entropy);
    mbedtls_ctr_drbg_init(&drbg);

    /* Seed the PRNG using the entropy pool, and throw in our secret key as an
     * additional source of randomness. */
    ret =(int) mbedtls_ctr_drbg_seed(&drbg, mbedtls_entropy_func, &entropy,
                                      (const unsigned char *)seed, sizeof seed );
    //printf("mbedtls_ctr_drbg_seed [ok]\n");
    if (ret != 0) {
        printf("error mbedtls_ctr_drbg_seed return %d \r\n", ret);
        return ret;
    }
    printf("num = %d\n", size);
    mbedtls_ctr_drbg_random(&drbg, num, size);
    
    mbedtls_ctr_drbg_free(&drbg);
    mbedtls_entropy_free(&entropy);
    
    //printf("mbedtls_ctr_drbg_random [ok]\n");
    //print_hex("random seed : ",(const unsigned char *) num, 32);

    return ret;
    
}

/** @brief This function spi cmd set to AES crypt struct .
 *
 *  @param[out] AES Cryto structure
 *  @param[in] SPI Communication cmd setting value
 *  @param[in] encryption or decryption data 16 byte
 *  @param[in] crypto key of CRC
 */

bool cube_crypt_spi_cmd_set(sec_spi_data *spi_data)
{
    bool ret = false;
    

    printf("GET_STATUS(spi_data->cmd, WORK_STATUS) : %d \r\n" , GET_STATUS(spi_data->cmd, WORK_STATUS));
    
    
    if( (AES_WORK_ON | AES_ENCRYPT_ON) == GET_STATUS(spi_data->cmd, (WORK_STATUS | ENCRYPT_STATUS)) )
    {   
        memcpy(Encrypt_ctx.sec_data.input_data, spi_data->data, 16);
        Encrypt_ctx.cmd = spi_data->cmd;
        
        printf("AES_ENCRYPT_ON\r\n");
        ret = true;     
    }
    
    if( (AES_WORK_ON | AES_DECRYPT_ON) == GET_STATUS(spi_data->cmd, (DECRYPT_STATUS|WORK_STATUS)) )
    {
        memcpy(Decrypt_ctx.sec_data.input_data, spi_data->data, 16);
        Decrypt_ctx.cmd = spi_data->cmd;
        
        ret = cube_crc_compare(Decrypt_ctx.sec_key.key_crc , spi_data->key_crc);
        if(!ret)
        {
             printf("KeyCRC error\r\n");
             return ret;
        }
        printf("AES_DECRYPT_ON\r\n");
        ret = true;     
    }
    
    /*
    if( AES_ENCRYPT_ON == GET_STATUS(spi_data->cmd, ENCRYPT_STATUS) )
    {   
         printf("AES_ENCRYPT_ON\r\n");
        if( AES_WORK_ON == GET_STATUS(spi_data->cmd, WORK_STATUS) )
        {
            printf("cube_crypt_spi_cmd_set : AES_ENCRYPT_ON\r\n");
            memcpy(Encrypt_ctx.sec_data.input_data, spi_data->data, 16);
            Encrypt_ctx.cmd = spi_data->cmd;
            ret = true;     
            return ret;
        }
        else
        {
            ret = cube_AES_read_data(&Encrypt_ctx, (uint8_t *)spi_data->data, 16);
            print_hex("encrypt data : ",(const unsigned char *)spi_data->data, 16);
            return ret; 
        }
    }
    else if( AES_DECRYPT_ON == GET_STATUS(spi_data->cmd, DECRYPT_STATUS) )
    {
        printf("AES_DECRYPT_ON\r\n");
        if( AES_WORK_ON == GET_STATUS(spi_data->cmd, WORK_STATUS) )
        {
            printf("cube_crypt_spi_cmd_set : AES_ENCRYPT_ON\r\n");
            memcpy(Decrypt_ctx.sec_data.input_data, spi_data->data, 16);
            Decrypt_ctx.cmd = spi_data->cmd;
            ret = true;     
            return ret;
        }
        else
        {
            ret = cube_AES_read_data(&Decrypt_ctx, (uint8_t *)spi_data->data, 16);
            print_hex("decrypt data : ",(const unsigned char *)spi_data->data, 16);
            return ret;
        }
    }
    */
    
    return ret; 
}



void print_mbedtls_aes_context(mbedtls_aes_context printStr)
{
    printf("-------------mbedtls_aes_context---------\n");
    printf("number of rounds : %d\n", printStr.nr);
    printf("AES round keys : 0x%X\n", printStr.rk);
    printf("unaligned data : ");
    for(int iCnt = 0 ; iCnt < 68 ; iCnt++)
    {
        printf("%X",printStr.buf[iCnt]);
    }
    printf("\n");
    //printf("-----------------------------------------\n");
}
void print_cube_sec_data(cube_sec_data printStr)
{
    printf("--------------cube_sec_data--------------\n");
    printf("input_data : ");
    for(int iCnt = 0 ; iCnt < 16 ; iCnt++)
    {
        printf("%02X ",printStr.input_data[iCnt]);
    }
    printf("\n");

    printf("output_data : ");
    for(int iCnt = 0 ; iCnt < 16 ; iCnt++)
    {
        printf("%02X ",printStr.output_data[iCnt]);
    }
    printf("\n");

    printf("input_data_size : %d\n", printStr.input_data_size);
    printf("output_data_size : %d\n", printStr.output_data_size);

    //printf("-----------------------------------------\n");
}
void print_cube_sec_key(cube_sec_key printStr)
{
    printf("-------------cube_sec_key------------------\n");
    printf("iv : ");
    for(int iCnt = 0 ; iCnt < 16 ; iCnt++)
    {
        printf("%02X ",printStr.iv[iCnt]);
    }
    printf("\n");

    printf("key : ");
    for(int iCnt = 0 ; iCnt < 32 ; iCnt++)
    {
        printf("%02X ",printStr.key[iCnt]);
    }
    printf("\n");

    printf("key_size : %d\n", printStr.key_size);
    printf("key_crc : %04X\n", printStr.key_crc);

    //printf("-----------------------------------------\n");
}
void print_cube_sec_context(cube_sec_context printStr)
{
    printf("=============cube sec context print================\n");
    print_mbedtls_aes_context(printStr.mbed_ctx);
    printf("---------------- cmd --------------------\n");
    printf("cmd : %02X\n", printStr.cmd);
    //printf("-----------------------------------------\n");
    print_cube_sec_data(printStr.sec_data);
    print_cube_sec_key(printStr.sec_key);
    printf("===================================================\n");
}

void print_hex(const char *title, const unsigned char buf[], size_t len)
{
    printf("%s: ", title);

    for (size_t i = 0; i < len; i++)
        printf("%02x", buf[i]);

    printf("\r\n");
}


void cube_Thread_AES_encrypt(void const *argument)
{   
    
    while(1)
    {
        if(GET_STATUS(Encrypt_ctx.cmd, WORK_STATUS) == AES_WORK_ON)
        {
            printf("cube_Thread_AES_encrypt start\n");
            //sbi(cebf746_status_reg, 6);
            //sbi(cebf746_status_reg, 1);
            cube_AES_encrypt(&Encrypt_ctx);
            
            memcpy(spiTXBuf, Encrypt_ctx.sec_data.output_data, CEB_BUF_SIZE);        // encrypted data copy
            spiTXBuf[CEB_BUF_SIZE + 0] = ((Encrypt_ctx.sec_key.key_crc >> 8) & 0xFF);    // upper key crc copy 
            spiTXBuf[CEB_BUF_SIZE + 1] = ((Encrypt_ctx.sec_key.key_crc     ) & 0xFF);    // lower key crc copy    
            cebf746_set_packet(&encryptedTxPacket, CEB_BUF_SIZE+2 , spiTXBuf);
            
            
            print_cube_sec_data(Encrypt_ctx.sec_data);
            print_cube_sec_key(Encrypt_ctx.sec_key);
            SET_STATUS(Encrypt_ctx.cmd,WORK_STATUS,AES_WORK_OFF);
            printf("cube_Thread_AES_encrypt end\n");            
            //cbi(cebf746_status_reg, 6);
        }        
        Thread::wait(100);
    }
}

void cube_Thread_AES_decrypt(void const *argument)
{    
    while(1)
    {
        if(GET_STATUS(Decrypt_ctx.cmd, WORK_STATUS) == AES_WORK_ON)
        {
            printf("cube_Thread_AES_decrypt start\n");
            cube_AES_decrypt(&Decrypt_ctx);
            memcpy(spiTXBuf, Decrypt_ctx.sec_data.output_data, CEB_BUF_SIZE);        // encrypted data copy
            cebf746_set_packet(&decryptedTxPacket, CEB_BUF_SIZE , spiTXBuf); 
            print_cube_sec_data(Decrypt_ctx.sec_data);
            print_cube_sec_key(Decrypt_ctx.sec_key);
            SET_STATUS(Decrypt_ctx.cmd,WORK_STATUS,AES_WORK_OFF);
        }
        
        Thread::wait(100);
    }
}


