
/* CsrLocation class for mbed Microcontroller
 * Copyright 2014 CSR plc
 */


#include "mbed.h"
#include "CsrLocation.h"

#define APP_DBG_PORT_BAUD       115200
#define CSR_APP_LOG_INFO(...)   sSerialDebug.printf(__VA_ARGS__)

/* appliation commands */
typedef enum AppCmd
{
    APP_CMD_IDLE, 			// No special command
    APP_CMD_HELP,			// Show the supported commands
    APP_CMD_START,			// Start location
    APP_CMD_STOP,			// Stop location
    APP_CMD_PM_FULL,		// Set full power mode
    APP_CMD_PM_PTF,			// Set low power PTF mode
    APP_CMD_PTF_GETPOS,		// Get location immediately in low power PTF mode
    APP_CMD_NMEA,			// proto mode is NMEA
    APP_CMD_OSP,            // proto mode is OSP
    APP_CMD_SWITCH_NMEA,    // Debug command, switch chip to NMEA protocol at 4800bps
    APP_CMD_SWITCH_OSP,     // Debug command, switch chip to NMEA protocol at 4800bps
    APP_CMD_START_FAILED,   // Process start failed case
    APP_CMD_STOP_FAILED,    // Process stop failed case
    APP_CMD_ONOFF_ON,		// Debug command, pull onoff pin high level
    APP_CMD_ONOFF_OFF,      // Debug command, pull onoff pin low level
    APP_CMD_RESET_ON,       // Debug command, pull reset pin high level
    APP_CMD_RESET_OFF       // Debug command, pull reset pin low level
}eAppCmd;


static void _AppShowCmd(void);
static void _AppOutputCallback(uint32_t  msgId, void * const pMsgData, uint32_t msgLength);
static void _AppEventCallback(eCsrLocEventType event, uint32_t data);
static void _ConsoleRxHandler(void);
static void _AppCmdProcess(char *pCmd);


static int sAppCmd  = APP_CMD_IDLE;
static ePowerMode sPwrMode = PWR_FULL;
static eProto     sProto = PROTO_OSP;

static DBG_SERIAL_TYPE  sSerialDebug(PINMAP_UART_DEBUG_TX, PINMAP_UART_DEBUG_RX);
static LOC_SERIAL_TYPE  sSerialLoc(PINMAP_UART_LOC_TX, PINMAP_UART_LOC_RX);
static DigitalOut		sPinOnoff(PINMAP_GPIO_LOC_ONOFF);
static DigitalOut		sPinReset(PINMAP_GPIO_LOC_RESET);
static DigitalOut		sLedLocOn(LOC_LED1);
static DigitalOut		sLedPosReport(LOC_LED2);


int main(void)
{
    CsrLocation  *pCsrLoc;
	tCsrLocConfig locConfig;

	/* initialize the debug serial port */
    sSerialDebug.baud(APP_DBG_PORT_BAUD);
    sSerialDebug.attach(&_ConsoleRxHandler);

	/* initialize the CsrLocConfig */
	locConfig.pSerialDebug 	= &sSerialDebug;
	locConfig.pSerialLoc 	= &sSerialLoc;
	locConfig.pPinOnoff 	= &sPinOnoff;
	locConfig.pPinReset 	= &sPinReset;

	/* new the CsrLocation instance */
    pCsrLoc = new CsrLocation(&locConfig);
    if(pCsrLoc == NULL)
    {
        CSR_APP_LOG_INFO("Failed to new csrLocation.\r\n");
        sSerialDebug.attach(NULL);
        return -1;
    }
    else
    {
        CSR_APP_LOG_INFO("Success to new csrLocation.\r\n");
    }

    /* Register output callback and event callback functions */
    pCsrLoc->CsrLocRegOutput(_AppOutputCallback, _AppEventCallback);

    _AppShowCmd();

    while(1)
    {
        switch(sAppCmd)
        {
        case APP_CMD_HELP:
            sAppCmd = APP_CMD_IDLE;
            _AppShowCmd();
            break;
        case APP_CMD_START:
            sAppCmd = APP_CMD_IDLE;
            if(pCsrLoc->CsrLocGetState() == CSR_LOC_STATE_RUN)
            {
	            CSR_APP_LOG_INFO("Cannot execute this command as location is in running state, please stop location firstly.\r\n");
	            break;
            }
            CSR_APP_LOG_INFO("start location.\r\n");
            sLedLocOn = 1;
            pCsrLoc->CsrLocStart(sPwrMode, sProto);
            break;
        case APP_CMD_STOP:
            sAppCmd = APP_CMD_IDLE;
            if(pCsrLoc->CsrLocGetState() == CSR_LOC_STATE_IDLE)
            {
	            CSR_APP_LOG_INFO("Cannot execute this command as location is not in running state.\r\n");
	            break;
            }
            CSR_APP_LOG_INFO("stop location.\r\n");
            sLedLocOn = 0;
            pCsrLoc->CsrLocStop();
            break;
        case APP_CMD_START_FAILED:
            sAppCmd = APP_CMD_IDLE;
            CSR_APP_LOG_INFO("reset as start failed.\r\n");
            sLedLocOn = 0;
            pCsrLoc->CsrLocStop();
            pCsrLoc->CsrLocReset();
            break;
        case APP_CMD_STOP_FAILED:
            sAppCmd = APP_CMD_IDLE;
            CSR_APP_LOG_INFO("reset as stop failed.\r\n");
            sLedLocOn = 0;
            pCsrLoc->CsrLocStop();
            pCsrLoc->CsrLocReset();
            break;
        case APP_CMD_IDLE:
            pCsrLoc->CsrLocUpdate();
            break;
        case APP_CMD_ONOFF_ON:
            CSR_APP_LOG_INFO("onoff on.\r\n");
            sAppCmd = APP_CMD_IDLE;
            sPinOnoff = 1;
            break;
        case APP_CMD_ONOFF_OFF:
            CSR_APP_LOG_INFO("onoff off.\r\n");
            sAppCmd = APP_CMD_IDLE;
            sPinOnoff = 0;
            break;
        case APP_CMD_RESET_ON:
            CSR_APP_LOG_INFO("reset on.\r\n");
            sAppCmd = APP_CMD_IDLE;
            sPinReset = 1;
            break;
        case APP_CMD_RESET_OFF:
            CSR_APP_LOG_INFO("reset off.\r\n");
            sAppCmd = APP_CMD_IDLE;
            sPinReset = 0;
            break;
        case APP_CMD_PTF_GETPOS:
            sAppCmd = APP_CMD_IDLE;
            if(pCsrLoc->CsrLocGetState() != CSR_LOC_STATE_RUN)
            {
	            CSR_APP_LOG_INFO("Cannot execute this command as location is not in running state, please start location in PTF mode firstly.\r\n");
	            break;
            }
            CSR_APP_LOG_INFO("lpm get pos.\r\n");
            pCsrLoc->CsrLocLpmGetPos();
            break;
        case APP_CMD_NMEA:
            sAppCmd = APP_CMD_IDLE;
            if(pCsrLoc->CsrLocGetState() == CSR_LOC_STATE_RUN)
            {
	            CSR_APP_LOG_INFO("Cannot execute this command as location is in running state, please stop location firstly.\r\n");
	            break;
            }
            if(sPwrMode != PWR_FULL)
            {
                CSR_APP_LOG_INFO("Cannot select NMEA protocol because current lpm is not supported for NMEA protocol, pls switch to FPM firstly.\r\n");
            }
            else
            {
                CSR_APP_LOG_INFO("select NMEA protocol.\r\n");
                sProto = PROTO_NMEA;
            }
            break;
        case APP_CMD_OSP:
            sAppCmd = APP_CMD_IDLE;
            if(pCsrLoc->CsrLocGetState() == CSR_LOC_STATE_RUN)
            {
	            CSR_APP_LOG_INFO("Cannot execute this command as location is in running state, please stop location firstly.\r\n");
	            break;
            }
            CSR_APP_LOG_INFO("select OSP protocol.\r\n");
            sProto = PROTO_OSP;
            break;
        case APP_CMD_SWITCH_NMEA:
            sAppCmd = APP_CMD_IDLE;
            if(pCsrLoc->CsrLocGetState() == CSR_LOC_STATE_RUN)
            {
	            CSR_APP_LOG_INFO("Cannot execute this command as location is in running state, please stop location firstly.\r\n");
	            break;
            }
            CSR_APP_LOG_INFO("switch to NMEA protocol.\r\n");
            pCsrLoc->CsrLocDebugSwitch2Nmea();
            break;
        case APP_CMD_PM_FULL:
            sAppCmd = APP_CMD_IDLE;
            if(pCsrLoc->CsrLocGetState() == CSR_LOC_STATE_RUN)
            {
	            CSR_APP_LOG_INFO("Cannot execute this command as location is in running state, please stop location firstly.\r\n");
	            break;
            }
            CSR_APP_LOG_INFO("fpm set.\r\n");
            sPwrMode = PWR_FULL;
            break;
        case APP_CMD_PM_PTF:
            sAppCmd = APP_CMD_IDLE;
            if(pCsrLoc->CsrLocGetState() == CSR_LOC_STATE_RUN)
            {
	            CSR_APP_LOG_INFO("Cannot execute this command as location is in running state, please stop location firstly.\r\n");
	            break;
            }
            if(sProto == PROTO_NMEA)
            {
                CSR_APP_LOG_INFO("Cannot set lpm because LPM is not supported for current NMEA protocol, pls switch to OSP firstly.\r\n");
            }
            else
            {
                CSR_APP_LOG_INFO("lpm ptf set.\r\n");
                sPwrMode = PWR_PTF;
            }
            break;
        }
    }
}

static void _AppShowCmd(void)
{
    CSR_APP_LOG_INFO("Location commands:\r\n");
    CSR_APP_LOG_INFO("    help     - help to show supported commands\r\n");
    CSR_APP_LOG_INFO("    start    - begin location\r\n");
    CSR_APP_LOG_INFO("    stop     - end location\r\n");
    CSR_APP_LOG_INFO("    fpm      - full power mode\r\n");
    CSR_APP_LOG_INFO("    ptf      - ptf low power mode\r\n");
    CSR_APP_LOG_INFO("    getpos   - get location immediately in low power ptf mode\r\n");
    CSR_APP_LOG_INFO("    osp      - OSP mode\r\n");
    CSR_APP_LOG_INFO("    nmea     - NMEA mode\r\n");
    
}

static void _AppOutputCallback(uint32_t  msgId, void * const pMsgData, uint32_t msgLength)
{
    switch(msgId)
    {
    case LOC_OUTPUT_LOCATION:
    {
        tLocPosResp *pPosRsp = (tLocPosResp *)pMsgData;
        CSR_APP_LOG_INFO("Loc: lat=%f, lon=%f, alt=%f\r\n", pPosRsp->lat, pPosRsp->lon, pPosRsp->alt);
        sLedPosReport = !sLedPosReport;
        break;
    }
    case LOC_OUTPUT_SV_STATUS:
    {
        tLocSvStatus *pSvStatus = (tLocSvStatus *)pMsgData;
        CSR_APP_LOG_INFO("SV:week=%u, tow=%lu, GPS Num=%u, GLO Num=%u, GPS Mask=0x%lx, GLO Mask=0x%lx\r\n", 
                pSvStatus->gps_week, pSvStatus->tow, pSvStatus->numOfSVs, pSvStatus->numOfGloSVs, 
                pSvStatus->svUsedInFixMask, pSvStatus->gloSvUsedInFixMask);
        break;
    }

    default :
        break;
    }   
}

static void _AppEventCallback(eCsrLocEventType event, uint32_t data)
{
    switch(event)
    {
    case CSR_LOC_EVENT_START_RESULT:
        if(data != 0)
        {
            CSR_APP_LOG_INFO("start failed.\r\n");
            sAppCmd = APP_CMD_START_FAILED;
        }
        else
        {
            CSR_APP_LOG_INFO("start OK.\r\n");
        }
        break;
    case CSR_LOC_EVENT_STOP_RESULT:
        if(data != 0)
        {
            CSR_APP_LOG_INFO("stop failed.\r\n");
            sAppCmd = APP_CMD_STOP_FAILED;
        }
        else
        {
            CSR_APP_LOG_INFO("stop OK.\r\n");
        }
        break;
    default:
        break;
    }
}

static void _ConsoleRxHandler(void)
{
	static char cmd[32] = {0};
    char ch;

    ch = sSerialDebug.getc();
    sSerialDebug.putc(ch);
	if(ch == '\r')
	{
        sSerialDebug.putc('\n');
        if(strlen(cmd) > 0)
        {
	        _AppCmdProcess(cmd);
	        memset(cmd, 0, sizeof(cmd));
        }
	}
	else
	{
		cmd[strlen(cmd)] = ch;
	}
}
	
static void _AppCmdProcess(char *pCmd)
{
	if(strcmp(pCmd, "help") == 0)
    {
        sAppCmd = APP_CMD_HELP;
    }
    else if(strcmp(pCmd, "start") == 0)
    {
        sAppCmd = APP_CMD_START;
    }
    else if(strcmp(pCmd, "stop") == 0)
    {
        sAppCmd = APP_CMD_STOP;
    }
    else if(strcmp(pCmd, "fpm") == 0)
    {
        sAppCmd = APP_CMD_PM_FULL;
    }
    else if(strcmp(pCmd, "ptf") == 0)
    {
        sAppCmd = APP_CMD_PM_PTF;
    }
    else if(strcmp(pCmd, "getpos") == 0)
    {
        sAppCmd = APP_CMD_PTF_GETPOS;
    }
    else if(strcmp(pCmd, "nmea") == 0)
    {
        sAppCmd = APP_CMD_NMEA;
    }
    else if(strcmp(pCmd, "osp") == 0)
    {
        sAppCmd = APP_CMD_OSP;
    }
    else if(strcmp(pCmd, "switchnmea") == 0)
    {
        sAppCmd = APP_CMD_SWITCH_NMEA;
    }
    else if(strcmp(pCmd, "switchosp") == 0)
    {
        sAppCmd = APP_CMD_SWITCH_OSP;
    }
    else if(strcmp(pCmd, "onoffon") == 0)
    {
        sAppCmd = APP_CMD_ONOFF_ON;
    }
    else if(strcmp(pCmd, "onoffoff") == 0)
    {
        sAppCmd = APP_CMD_ONOFF_OFF;
    }
    else if(strcmp(pCmd, "reseton") == 0)
    {
        sAppCmd = APP_CMD_RESET_ON;
    }
    else if(strcmp(pCmd, "resetoff") == 0)
    {
        sAppCmd = APP_CMD_RESET_OFF;
    }
    else
    {
        CSR_APP_LOG_INFO("\r\nUnknown command %s\r\n", pCmd);
    }
	
	CSR_APP_LOG_INFO("\r\n");
}
