
/* CsrLocation class for mbed Microcontroller
 * Copyright 2014 CSR plc
 */


#ifndef CSRLOCATION_H
#define CSRLOCATION_H


#define CSR_LOC_SDK_VER                 "CSR-LOC-SDK-0.5"

/* Data type definitions */
#undef  FALSE
#define FALSE (0)

#undef  TRUE
#define TRUE (1)

/* Unsigned fixed width types */
typedef unsigned char CsrUint8;
typedef unsigned short CsrUint16;
typedef unsigned int CsrUint32;

/* Signed fixed width types */
typedef signed char CsrInt8;
typedef signed short CsrInt16;
typedef signed int CsrInt32;

/* Boolean */
typedef CsrUint8 CsrBool;

/* String types */
typedef char CsrCharString;
typedef CsrUint8 CsrUtf8String;
typedef CsrUint16 CsrUtf16String;   /* 16-bit UTF16 strings */
typedef CsrUint32 CsrUint24;

/*
 * Floating point
 *
 * Note: If a given compiler does not support floating point, it is
 * OK to omit these definitions;  alternative versions of the code using
 * these types may be available.  Consult the relevant documentation
 * or the customer support group for information on this.
 */
typedef float CsrFloat;
typedef double CsrDouble;

typedef CsrUint16 CsrResult;
#define CSR_RESULT_SUCCESS  ((CsrResult) 0x0000)
#define CSR_RESULT_FAILURE  ((CsrResult) 0xFFFF)
/* The end of data type definitions */

/* OSP protocol related definitions */
#define MAX_VERSION_LENGTH 80

/* OSP message header */
#define OSP_MSG_HEAD0                   (0xA0)
#define OSP_MSG_HEAD1                   (0xA2)
/* OSP message footer */
#define OSP_MSG_TAIL0                   (0xB0)
#define OSP_MSG_TAIL1                   (0xB3)

/* NMEA message header */
#define NMEA_MSG_HEAD0                  ('$')
#define NMEA_MSG_HEAD1                  ('G')
/* NMEA message footer */
#define NMEA_MSG_TAIL0                  ('*')

#define CSR_SWAPIN16(bytestream) (((CsrUint16)*((bytestream)+0) << 8) | ((CsrUint16)*((bytestream)+1)))

#define CSR_SWAPIN32(bytestream)\
   ( ((CsrUint32)*((bytestream)+0) << 24)\
   | ((CsrUint32)*((bytestream)+1) << 16)\
   | ((CsrUint32)*((bytestream)+2) <<  8)\
   | ((CsrUint32)*((bytestream)+3)      ))


/* import macros for little endian: */
/* NOTE: must use {} around these macros when calling in a loop */
#define BINARY_IMPORT_UINT8(bytestream) (                      *((bytestream)++))
#define BINARY_IMPORT_UINT16(bytestream) ((CsrUint16) CSR_SWAPIN16((bytestream))); bytestream+=2
#define BINARY_IMPORT_UINT32(bytestream) ((CsrUint32) CSR_SWAPIN32((bytestream))); bytestream+=4
#define BINARY_IMPORT_SINT32(bytestream) ((CsrInt32)  CSR_SWAPIN32((bytestream))); bytestream+=4


#define OSP_MAKE_MSG_ID(mid, sid) ((((mid) & 0xFF) << 8) | ((sid) & 0xFF))

#define OSP_MSG_SW_VERSION              OSP_MAKE_MSG_ID(0x06, 0x0)
#define OSP_MSG_OK_TO_SEND              OSP_MAKE_MSG_ID(0x12, 0x0)
#define OSP_MSG_GEODETIC_NAVIGATION     OSP_MAKE_MSG_ID(0x29, 0x0)
#define OSP_MSG_MULTI_CONSTELLATION     OSP_MAKE_MSG_ID(0x43, 0x0)
    #define OSP_MSG_GNSS_NAV_DATA       OSP_MAKE_MSG_ID(0x43, 0x01)
    #define OSP_MSG_GNSS_SAT_DATA       OSP_MAKE_MSG_ID(0x43, 0x10)
#define OSP_MSG_HW_CONFIG_REQ           OSP_MAKE_MSG_ID(0x47, 0x0)
#define OSP_MSG_PWR_MODE_RSP            OSP_MAKE_MSG_ID(0x5A, 0x0)
    #define OSP_MSG_PWR_MODE_FPM_RSP    OSP_MAKE_MSG_ID(0x5A, 0x0)
    #define OSP_MSG_PWR_MODE_LPM_RSP    OSP_MAKE_MSG_ID(0x5A, 0x6)

   
#define GNSS_SAT_DATA_NUM_OF_SATS       (15)
#define CODEC_GLO_MAX_CHANNELS          (14)
/* The end of OSP protocol definitions */


#define LOC_MAX_GNSS_SVS                (32)
#define LOC_GLO_FREQ_OFFSET             (77)
#define LOC_GLO_FREQ_ID_START           (70)
#define LOC_GLO_FREQ_ID_END             (83)
#define LOC_NUM_OF_GLO_FREQ_CHANNELS    (1+LOC_GLO_FREQ_ID_END-LOC_GLO_FREQ_ID_START)

#define MAX_PORT_NUM_STRING_LENGTH      (16)
#define MAX_SERIAL_BUF_LEN              (2048)
#define MAX_SERIAL_PKT_LEN              (512)

#define BAUDRATE_NMEA                   4800
#define BAUDRATE_OSP                    115200

#define PROTO_CHECK_TIMEOUT             (2.0)

/** Indicates the outputted location information */
#define LOC_OUTPUT_LOCATION             (1)
/* Indicates the outputted sv status information */
#define LOC_OUTPUT_SV_STATUS            (2)

#define CSR_LOG_INFO(...) \
{\
    if(csrLocInst.pSerialDebug != NULL)\
    {\
        (csrLocInst.pSerialDebug->printf(__VA_ARGS__));\
    }\
}

/** Location enent definitions */
typedef enum
{
    /** Start result event */
    CSR_LOC_EVENT_START_RESULT,
    /** Stop result event */
    CSR_LOC_EVENT_STOP_RESULT,
}eCsrLocEventType;

/** Power mode selection */
typedef enum
{
    /** full power mode */
    PWR_FULL,
    /** Low power push to fix mode */
    PWR_PTF,
}ePowerMode;

/* Protocol detection state */
typedef enum
{
    STATE_START1,       /* Indicates the first byte of the OSP or NMEA message header*/
    STATE_START2,       /* Indicates the second byte of the OSP or NMEA message header */
    STATE_SIZE1,        /* Indicates the first byte of the OSP message length */
    STATE_SIZE2,        /* Indicates the second byte of the OSP message length  */
    STATE_PAYLOAD,      /* Indicates the start of payload of the OSP message */
    STATE_CHECKSUM1,    /* Indicates the first byte of the OSP message checksum  */
    STATE_CHECKSUM2,    /* Indicates the second byte of the OSP message checksum  */
    STATE_END1,         /* Indicates the first byte of the OSP or NMEA message footer  */
    STATE_END2          /* Indicates the second byte of the OSP message footer  */
}eProtoDetState;

/* Csr Location state */
typedef enum
{
    CSR_LOC_STATE_IDLE,
    CSR_LOC_STATE_RUN,
}eCsrLocState;

/* Locatin chip protocol detection state */
typedef enum
{
   PROTO_STATE_DET_INVALID = 0,
   PROTO_STATE_DET_OSP,
   PROTO_STATE_DET_NMEA,
   PROTO_STATE_SWI_OSP_FROM_NMEA,
   PROTO_STATE_DET_OSP_FROM_NMEA,
   PROTO_STATE_DET_OK,
}eProtoState;

/* Locaiton chip status */
typedef enum
{
    /* Location chip is going to hibernation mode and cannot accept message any more */
    ENGINE_STATUS_NOTOK2SEND,
    /* Locaitn come back from hibernation mode and can accept message now */
    ENGINE_STATUS_OK2SEND
}eEngineStatus;

/* OSP data type to be sent to location chip */
typedef enum
{
    SEND_DATA_TYPE_OSP_STOP_REQ,
    SEND_DATA_TYPE_OSP_VER_REQ,
    SEND_DATA_TYPE_OSP_LPM_REQ,
    SEND_DATA_TYPE_OSP_FPM_REQ,
    SEND_DATA_TYPE_OSP_SWITCH2NMEA_REQ,
    SEND_DATA_TYPE_NMEA_SWITCH2OSP_REQ
}eSendDataType;

/** Structure to hold Position Response Message Information. */
typedef struct LocPosResp
{
    /** Week part of GPS time */
    CsrUint16            gps_week;
    /** Time of second part of GPS time */
    CsrUint32            tow;
    /** Latitude */
    CsrDouble            lat;
    /** Longitude */
    CsrDouble            lon;
    /** Altitude */
    CsrDouble            alt;
} tLocPosResp;

/** Structure to hold Satellite Information. */
typedef struct LocSvInfo
{
    /** Prn or svId */
    CsrUint8             prn;
    /** The ratio of carrier and noise */
    CsrFloat             cno;
    /** elevation */
    CsrFloat             elevation;
    /** azimuth */
    CsrFloat             azimuth;
    /** satellite state */
    CsrUint16            state;
} tLocSvInfo;

/** Structure to hold Satellite Information for GLONASS */
typedef struct LocGloSvInfo
{
    /** Prn or svId */
    CsrUint8             prn;
    /** Slot number */
    CsrUint8             sno;
    /** The ratio of carrier and noise */
    CsrFloat             cno;
    /** elevation */
    CsrFloat             elevation;
    /** azimuth */
    CsrFloat             azimuth;
    /** satellite state */
    CsrUint16            state;
} tLocGloSvInfo;

/** Structure to hold Satellite Status Information. */
typedef struct LocSvStatus
{
    /** Week part of GPS time */
    CsrUint16           gps_week;
    /** Time of second part of GPS time */
    CsrUint32           tow;
    /** Time of millisecond part of GPS time */
    CsrUint32           tow_sub_ms;
   
    /**Number of GPS SVs currently visible **/
    CsrUint8            numOfSVs;
    /**Number of GLONASS SVs currently visible **/
    CsrUint8            numOfGloSVs;
    /** GPS SVs information */
    tLocSvInfo          svList[LOC_MAX_GNSS_SVS];
    /** GLONASS SVs information */
    tLocGloSvInfo       gloSvList[LOC_NUM_OF_GLO_FREQ_CHANNELS];
    /** Bit mask indicating which SVs have ephemeris data **/
    CsrUint32           ephemerisMask;
    /** Bit mask indicating which GLONASS SVs have ephemeris data **/
    CsrUint32           gloEphemerisMask;
    /** Bit mask indicating which SVs were used in latest sent fix **/
    CsrUint32           svUsedInFixMask;
    /** Bit mask indicating which GLONASS SVs were used in latest sent fix **/
    CsrUint32           gloSvUsedInFixMask;
    /** Bit mask indicating which QZSS SVs were used in latest sent fix **/   
    CsrUint32           qzssSvUsedInFixMask;
    /** Bit mask indicating which SBAS SVs were used in latest sent fix **/   
    CsrUint32           sbasSvUsedInFixMask;
} tLocSvStatus;


/** Applicaiton register this out callback function and CsrLocaiton class will pass outputted information to application */
typedef void (*csr_app_output_callback)(CsrUint32  msgId, void * const pMsgData, CsrUint32 msgLength);

/** Applicaiton register this event callback function and CsrLocaiton class will pass internal porcessing event to application */
typedef void (*csr_app_event_callback)(eCsrLocEventType event, CsrUint32 data);

/** tCsrLocConfig structure
 * Application needs to decides and pass the configuration into CsrLocation class.
 */
typedef struct CsrLocConfig
{
    /** Debug serial port to print debug information */
    Serial          *pSerialDebug;
    /** location serail port to communicate between mbed host side and location chip */
    Serial          *pSerialLoc;
    /** GPIO pin to control location chip on, a rising edge is uset to activate location chip. Please note, before activate chip, reset pin should be pull high */
    DigitalOut      *pPinOnoff;
    /** GPIO pin to control location chip reset, low level will keep location chip in hibernation state and high level will permit location chip to be activated */
    DigitalOut      *pPinReset;
}tCsrLocConfig;

/* General OSP mesasge format */
typedef struct OspMsg
{
   CsrUint32        msgId;   
   CsrUint32        length;
   CsrUint8         payload[4];
} tOspMsg;

/* keep the internal data of CsrLocation class */
typedef struct CsrLocInst
{
    CsrBool         bStopFlag;

    eCsrLocState    locState;
    eProtoState     protoState;
    ePowerMode      pwrMode;
    CsrUint32       baudRate;
    CsrBool         isOspHeader;
    CsrBool         isNmeaHeader;
    CsrInt32        computedCheckSum;
    CsrInt32        checksum;
    CsrInt32        msgSize;
    CsrInt32        decodeIndex;
    eProtoDetState  protoDetState;
    Timeout         *pTimeoutChk;    /* timeout process */
    CsrBool         bTimeoutFlag;
    eEngineStatus   engStatus;

    tLocSvStatus    svStatus;   /* 2 kind of messages contribute the svStaus */

    Serial          *pSerialDebug;
    Serial          *pSerialLoc;
    DigitalOut      *pPinOnoff;
    DigitalOut      *pPinReset;

    CsrUint8        serialBuf[MAX_SERIAL_BUF_LEN]; /* buffer the serial data from uart callback function */
    CsrUint8        serialPkt[MAX_SERIAL_PKT_LEN]; /* decoded osp data */
    CsrUint16       in;
    CsrUint16       out;
    
    csr_app_output_callback appOutCb;
    csr_app_event_callback  appEventCb;
}tCsrLocInst;

/** CsrLocation class.
 *  A location interface to control location chip and get position and satellite information.
 */
class CsrLocation
{
public:
    /** Constructor: CsrLocaiton
     * Create the CsrLocation, accept specified configuration
     * @param pLocConfig Configuration including debug serial port, location communication serail port, onoff pin, reset pin
     */
    CsrLocation(tCsrLocConfig *pLocConfig);
    
    /** Destructor: CsrLocation
     * Free allocated resource
     */
    ~CsrLocation();
    
    /** Register output callback and enent callback functions
     * @param app_output_cb CsrLocation class output the loaction and satellite information to application
     * @param app_event_cb CsrLocation class output the start and stop result to application
     */
    void CsrLocRegOutput(csr_app_output_callback app_output_cb, csr_app_event_callback app_event_cb);

    /** hw reset to get location chip into hibernation mode */
    void CsrLocReset(void);

    /** Start location request, activate location chip */
    void CsrLocStart(ePowerMode pwrMode);

    /** Process location data from chip and update location and satellite information */
    void CsrLocUpdate(void);

    /** Stop location request, get location chip into hibernation mode */
    void CsrLocStop(void);

    /** Speical for low power PTF mode. 
      * During low power PTF mode, after reporting position, chip will go to hibernation mode automatically.
      * After 30 seconds, chip will recover and report position again. Then chip will go to hibernation mode again.
      * During the hibernation, application can call CsrLocLpmGetPos to get position immediately and no need to wait for the whole interval.
      */
    void CsrLocLpmGetPos(void);

    /* A debug interface to switch location chip protocol from OSP at 115200bps to NMEA at 4800bps */
    void CsrLocDebugSwitch2Nmea(void);
    
private:
    /* Internal kept data */
    tCsrLocInst csrLocInst;

    /* Initialize the serial port and open it */
    void _CsrLocUartInit(void);

    /* Process the raw stream from location seraial port */
    void _CsrLocProcessRawStream(void);

    /* Detect the OSP protocol outputted from location serial port */
    void _CsrLocDetProtoOsp(CsrUint8 data);

    /* Detect the NMEA protocol outputted from location serial port */
    void _CsrLocDetProtoNmea(CsrUint8 data);

    /* Process the raw OSP stream, remove the OSP header, size, check sum, and save the OSP data into interal buffer */
    void _CsrLocProcessRawOspStream(CsrUint8 data);

    /* Process the saved OSP data and decode them */
    void _CsrLocProcessRawOspPkt(void);

    /* Calculate the OSP message size to allcate buffer to save the decoded OSP data */
    CsrUint32 _CsrLocCalcMsgSize(void);

    /* Decode OSP data into pakcet data structure */
    CsrResult _CsrLocDecodeOspPkt( CsrUint8 *payload, CsrUint32 payload_length, CsrUint32 *message_id, void   *message_structure, CsrUint32 *message_length);

    /* Process the decode OSP packet and pass to application when needed */
    void _CsrLocProcessOspPkt(tOspMsg *pOspMsg);

    /* Timeout process, such as detect OSP and NMEA protocol */
    void _CsrLocTimeout(void);

    /* Location serial port data recevier */
    void _CsrLocRxHandler(void);

    /* Send special OSP messges to location chip */
    void _CsrLocSendData(eSendDataType type);

    /* Trigger a pulse on the onoff pin */
    void _CsrLocHwOnoff(void);

    /* Trigger a reset on the reset pin */
    void _CsrLocHwReset(void);
};



#endif /* CSRLOCATION_H */
