/* mbed Microcontroller Library
 * Copyright (c) 2006-2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "CsrLocation.h"
#include "GPSProviderImplBase.h"

#define PINMAP_UART_LOC_TX     D8
#define PINMAP_UART_LOC_RX     D2
#define PINMAP_GPIO_LOC_ONOFF  D9
#define PINMAP_GPIO_LOC_RESET  D4
#define PINMAP_GPIO_LOC_WAKEUP D3
#define PINMAP_GPIO_BTN        D5
#define PINMAP_GPIO_TEST       D10
#define LOC_LED1               D7
#define LOC_LED2               D6

static RawSerial  sSerialLoc(PINMAP_UART_LOC_TX, PINMAP_UART_LOC_RX);
static DigitalOut sPinOnoff(PINMAP_GPIO_LOC_ONOFF);
static DigitalOut sPinReset(PINMAP_GPIO_LOC_RESET);
static DigitalIn  sWakeup(PINMAP_GPIO_LOC_WAKEUP);

static void _AppOutputCallback(uint32_t msgId, void *const pMsgData, uint32_t msgLength);
static void _AppEventCallback(eCsrLocEventType event, uint32_t data);

extern Serial sSerialDebug;
#define CSR_APP_LOG_INFO(...)   sSerialDebug.printf(__VA_ARGS__)

GPSProviderImplBase *
createGPSProviderInstance(void)
{
    static CSRLocation csrGPS(sSerialLoc, sPinOnoff, sPinReset, sWakeup, &sSerialDebug);

    /* Register output callback and event callback functions */
    csrGPS.CsrLocRegOutput(_AppOutputCallback, _AppEventCallback);

    return &csrGPS;
}

void
_AppOutputCallback(uint32_t msgId, void *const pMsgData, uint32_t msgLength)
{
    switch (msgId) {
        case LOC_OUTPUT_LOCATION: {
            tLocPosResp *pPosRsp = (tLocPosResp *)pMsgData;
            CSR_APP_LOG_INFO("Loc: lat=%f, lon=%f, alt=%f\r\n", pPosRsp->lat, pPosRsp->lon, pPosRsp->alt);
            break;
        }
        case LOC_OUTPUT_SV_STATUS: {
            tLocSvStatus *pSvStatus = (tLocSvStatus *)pMsgData;
            CSR_APP_LOG_INFO("SV:week=%u, tow=%lu, GPS Num=%u, GLO Num=%u, GPS Mask=0x%lx, GLO Mask=0x%lx\r\n",
                             pSvStatus->gps_week, pSvStatus->tow, pSvStatus->numOfSVs, pSvStatus->numOfGloSVs,
                             pSvStatus->svUsedInFixMask, pSvStatus->gloSvUsedInFixMask);
            break;
        }
        case LOC_OUTPUT_NMEA: {
            CSR_APP_LOG_INFO("%s\r\n", (char *)pMsgData);
            break;
        }

        default:
            break;
    }
}

void _AppEventCallback(eCsrLocEventType event, uint32_t data)
{
    switch (event) {
        case CSR_LOC_EVENT_START_RESULT:
            if (data != 0) {
                CSR_APP_LOG_INFO("start failed.\r\n");
                // sAppCmd = APP_CMD_START_FAILED;
            } else {
                CSR_APP_LOG_INFO("start OK.\r\n");
            }
            break;
        case CSR_LOC_EVENT_STOP_RESULT:
            if (data != 0) {
                CSR_APP_LOG_INFO("stop failed.\r\n");
                // sAppCmd = APP_CMD_STOP_FAILED;
            } else {
                CSR_APP_LOG_INFO("stop OK.\r\n");
            }
            break;
        default:
            break;
    }
}
