#include "main.h"
#include "mbed.h"
#include "PinDetect.h"
//#include "USBSerial.h"
#include "Adafruit_SSD1306.h"
#include "SDFileSystem.h"
#include "receive_data.h"
#include "log_data.h"
#include "process_data.h"
#include "debug.h"
#include "SystemTime.h"
#include "sync.h"
#include "pins.h"
#include "DS3231.h"

#define X_POS(x) (32 + x%64)
#define Y_POS(y) (16 + y%48)

// Capture button stuff
#define PIN_DETECT_SAMPLE_PERIOD_uS 20000   // 20 ms sample period
#define SYNC_HELD_SAMPLES   (SYNC_HOLD_TIME_MS * 1000 / PIN_DETECT_SAMPLE_PERIOD_uS)


#ifdef USE_OLED
    SPI spi0(P0_9, P0_8, P0_10); // mosi, miso, sclk
    Adafruit_SSD1306 oled(spi0, P0_11, P0_12, P0_13); // DC, RST, CS
#endif

DS3231 rtc(I2C_SDA, I2C_SCL);

// Mode button
PinDetect captureButton(P0_16, PullUp);

// State
enum state {IDLE, CAPTURE, SYNC};
enum state State;

Timer captureTimer;

// This is used to ensure the rising edge after a hold is ignored
bool ignore_edge = false;
void buttonHeld() {
    if (State == IDLE)
        State = SYNC;
    else
        State = IDLE;
    
    // Button was held down, so don't count the next assert
    ignore_edge = true;
}

void buttonPressed() {
    // Don't ignore subsequent edges
    if (ignore_edge) {
        ignore_edge = false;
        return;
    }
    
    if (State == IDLE)
        State = CAPTURE;
    else if (State == CAPTURE)
        State = IDLE;
}

void printSplit(int split)
{
    int min = split / 60000;
    int sec = (split / 1000) % 60;
    int hund = (split / 10) % 100;

    OLED_PRINTPF("%1d", min, X_POS(0), Y_POS(20));
    OLED_PRINTPF("%02d", sec, X_POS(5), Y_POS(20));
    OLED_DRAWPIXEL(X_POS(14), Y_POS(24), 0x1);
    OLED_PRINTPF("%02d", hund, X_POS(15), Y_POS(20));
}

int main(void)
{

    SystemTime::start();

    State = IDLE;
    
    // After button is held, the next rising edge will call buttonPressed.
    //   Because of this, inside the callbacks need to ensure that edge
    //   does not do anything unintended.
    captureButton.attach_deasserted_held(buttonHeld);
    captureButton.attach_asserted(buttonPressed);
    captureButton.setSampleFrequency();
    captureButton.setSamplesTillHeld(SYNC_HELD_SAMPLES);
    
    VectorInt16 *data;
    while (true) {
        if (State == IDLE){
            PC_PRINTLN("Idling...");
            OLED_PRINTP("Idling...", X_POS(0), Y_POS(20));
            OLED_CLEAR();
        } else if (State == CAPTURE) {
            PC_PRINTLN("Starting capture...");
            PC_PRINTLN("Init SD card...");
            log_init();
            PC_PRINTLN("Init peak detect...");
            process_init();
            PC_PRINTLN("Init data receipt...");
            receive_init();           
            PC_PRINTLN("Capturing data...");
            
            captureTimer.start();
            int split = 0;
            printSplit(split);
            while (State == CAPTURE) {
                data = receive_data();
//                PC_PRINTLNF("x: %d ", data->x);
//                PC_PRINTLNF("y: %d ", data->y);
//                PC_PRINTLNF("z: %d ", data->z);
                log_data(captureTimer.read_ms(), data);

                if (process_data((int) (data->x), (int) (data->y), &split)) {
                    PC_PRINTLNF("Split time: %d", split);
                    printSplit(split);

//                    log_data(captureTimer.read_ms(), split);
                }
            }
            captureTimer.stop();
            captureTimer.reset();
            receive_close();
            process_close();
            log_close();
        } else if (State == SYNC) {
            OLED_PRINTP("Ready to sync...", 0, 0);
            sync_init();
            sync();
        }
    }
}
