/* mbed AX-12+ Servo Library
 *
 * Copyright (c) 2010, cstyles (http://mbed.org)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef MBED_AX12_H
#define MBED_AX12_H


#include "global.h"

//#define AX12_WRITE_DEBUG 0
#define AX12_READ_DEBUG   
//#define AX12_TRIGGER_DEBUG 0
#define AX12_DEBUG 0

/****** à utiliser pour le debug !!  ******/

//#define AX12_DEBUG_WRITE 0
//#define AX12_DEBUG_READ 0

/******************************************/


#define AX12_REG_ID 0x03
#define AX12_REG_BAUD 0x04
#define AX12_REG_DELAY_TIME 0x05
#define AX12_REG_CW_LIMIT 0x06
#define AX12_REG_CCW_LIMIT 0x08
#define AX12_REG_TEMP_MAX 0x0B
#define AX12_REG_LOWEST_VOLTAGE 0x0C
#define AX12_REG_HIGHEST_VOLTAGE 0x0D
#define AX12_REG_MAX_TORQUE 0x0E
#define AX12_REG_SATUS_RETURN 0x10
#define AX12_REG_ALARM_LED 0x11
#define AX12_REG_ALARM_SHUTDOWN 0x12
#define AX12_REG_DOWN_CALIBRATION 0x14
#define AX12_REG_UP_CALIBRATION 0x16
#define AX12_REG_TORQUE_ENABLE 0x18
#define AX12_REG_LED 0x19
#define AX12_REG_CW_MARGIN 0x1A
#define AX12_REG_CCW_MARGIN 0x1B
#define AX12_REG_CW_SLOPE 0x1C
#define AX12_REG_CCW_SLOPE 0x1D
#define AX12_REG_GOAL_POSITION 0x1E
#define AX12_REG_MOVING_SPEED 0x20
#define AX12_REG_TORQUE_LIMIT 0x22
#define AX12_REG_POSITION 0x24
#define AX12_REG_PRESENT_SPEED 0x26
#define AX12_REG_PRESENT_LOAD 0x28
#define AX12_REG_VOLTS 0x2A
#define AX12_REG_TEMP 0x2B
#define AX12_REG_INSTRUCTION 0x2C
#define AX12_REG_MOVING 0x2E
#define AX12_REG_LOCK 0x2F
#define AX12_REG_PUNCH 0x30


#define AX12_MODE_POSITION  0
#define AX12_MODE_ROTATION  1

#define AX12_CW 1
#define AX12_CCW 0

//--- Instruction ---
#define INST_PING 0x01
#define INST_READ 0x02
#define INST_WRITE 0x03
#define INST_REG_WRITE 0x04
#define INST_ACTION 0x05
#define INST_RESET 0x06
#define INST_DIGITAL_RESET 0x07
#define INST_SYSTEM_READ 0x0C
#define INST_SYSTEM_WRITE 0x0D
#define INST_SYNC_WRITE 0x83
#define INST_SYNC_REG_WRITE 0x84

#define DEFAULT_RETURN_PACKET_SIZE 6

#define BROADCASTING_ID 0xfe

/** Servo control class, based on a PwmOut
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "AX12.h"
 *
 * int main() {
 *
 *   AX12 myax12 (p9, p10, 1);
 *
 *   while (1) {
 *       myax12.SetGoal(0);    // go to 0 degrees
 *       wait (2.0);
 *       myax12.SetGoal(300);  // go to 300 degrees
 *       wait (2.0);
 *   }
 * }
 * @endcode
 */
class AX12
{

public:
    
    /** Create an AX12 servo object connected to the specified serial port, with the specified ID
     *
     * @param pin tx pin
     * @param pin rx pin
     * @param int ID, the Bus ID of the servo 1-255
     */
    AX12(PinName tx, PinName rx, int ID, int baud=1000000);
    
    /** Nouvelle fonction de commande de position du servomoteur avec sécurité d'arriver à la bonne position 
        Si une erreur se produit et que le servo ne recoit vraiment rien cette fonction dispose d'une sortie en timeout error;
    */
    int Set_Secure_Goal(int degres);
    
    /** Retourne le temps d'attente avant l'envoi de la trame de retour par l'actionneur ( en micro seconde )
    */
    int Get_Return_Delay_Time(void);
    
    
    /** Retourne la vitesse de communication de l'actionneur ( en Bits/seconde )
    */
    int Get_Baud_Rate(void);
    
    
    /** Reglage du courant minimum necessaire au bon fonctionnement de l'actionneur 
    //  minimum >>  Ox000    >>  decimal  0
    //  maximum >>  0x3FF    >>  decimal  1023
    //  deflaut >>  0x20     >>  decimal  32
    */
    int Set_Punch(int punch);
    
    
    /** Retourne le courant minimum au fonctionnement de l'actionneur
    */
    int Get_Punch (void);
    
    
    /** Retourne l'ampleur de la charge sur l'actionneur
    */
    int Get_Load_Value (void);
    
    void read_all_info(unsigned char, unsigned char);
    
    /** Reset
    */
    int Reset(int);
    
    /** Retourne la direction de la charge sur l'actionneur
    */
    int Get_Load_Direction (void);
    
    
    /** Retourne la vitesse angulaire actuelle de l'actionneur
    */
    int Get_Present_Speed (void);
    
    
    /** Retourne la valeur en degres du registre CCW angle limit qui est l'angle limite maximum de l'actionneur
    */
    int Get_CCW_Angle_Limit (void);
    
    
    /** Retourne la valeur en degres du registre CW angle limit qui est l'angle limite minimum de l'actionneur
    */
    int Get_CW_Angle_Limit (void);
    
    
    /** Retourne la valeur du registre Torque Enable 
    */
    int Get_Torque_Enable(void);
    
    
    /** 
    1  >>>  
    0  >>>  
    */
    int Set_Torque_Enable(int etat); 
    
    
    /** Retourne les données de compensation des différences entre les potentiomètres 
    utilisés dans l'AX12    (Up)     ????????
    */
    int Get_Up_Calibration (void);
    
    
    /** Retourne les données de compensation des différences entre les potentiomètres 
    utilisés dans l'AX12    (Dowm)     ????????
    */
    int Get_Down_Calibration(void);
    
    
    /** Retourne l'ID de l'AX_12 avec lequel on dialogue 
    utile seulement dans le cas d'un broadcast ID 
    */
    int Get_ID(void);
    
    
    /** Reglage du couple maximum de l'actionneur 
    //  minimum >>  Ox000    >>  decimal  0
    //  maximum >>  0x3FF    >>  decimal  1023
    //  deflaut >>           >>  decimal  
    */
    int Set_Max_Torque(int torque);
    
    
    /** Reglage de la desactivation des actionneurs si une erreur le concernant se produit 
    Bit                 Function
    Bit 7               0
    Bit 6               If set to 1, torque off when an Instruction Error occurs 
    Bit 5               If set to 1, torque off when an Overload Error occurs 
    Bit 4               If set to 1, torque off when a Checksum Error occurs 
    Bit 3               If set to 1, torque off when a Range Error occurs 
    Bit 2               If set to 1, torque off when an Overheating Error occurs 
    Bit 1               If set to 1, torque off when an Angle Limit Error occurs 
    Bit 0               If set to 1, torque off when an Input Voltage Error occurs
    */
    int Set_Alarm_Shutdown(int valeur); 
    
    
    /** Reglage de l'activation de l'alarme
    Bit                 Function
    Bit 7               0
    Bit 6               If set to 1, the LED blinks when an Instruction Error occurs
    Bit 5               If set to 1, the LED blinks when an Overload Error occurs
    Bit 4               If set to 1, the LED blinks when a Checksum Error occurs
    Bit 3               If set to 1, the LED blinks when a Range Error occurs
    Bit 2               If set to 1, the LED blinks when an Overheating Error occurs
    Bit 1               If set to 1, the LED blinks when an Angle Limit Error occurs
    Bit 0               If set to 1, the LED blinks when an Input Voltage Error occurs
    */
    int Set_Alarm_LED(int valeur);
    
    
    /** Reglage de la réponse à une instruction
    * @param mode  
    *  0   >>  ne repond a aucune instructions
    *  1   >>  repond seulement aux instructions READ_DATA 
    *  2   >>  repond à toutes les instructions 
    */
    int Set_Status_Return_Level(int etat); 
    
    
    /**  Reglage de la tension minimale
    * @param mode
    *  minimum >>  0x32    >>  decimal  50
    *  maximum >>  0xFA    >>  decimal  250
    *  deflaut >>  0x3C    >>  decimal  60
    */  
    int Set_Lowest_Voltage(int val_lowest_voltage); 
    
    
     /**  Reglage de la tension maximale
    * @param mode
    *  minimum >>  Ox32    >>  decimal  50
    *  maximum >>  0xFA    >>  decimal  250
    *  deflaut >>  0xBE    >>  decimal  190
    */  
    int Set_Highest_Voltage(int val_highest_voltage); 
    
    
    // Reglage du return time delay  EN MICRO SECONDE              2uSec * val_delay_time
    //  minimum >>  0       us 
    //  maximum >>  508     us
    //  deflaut >>  125     us
    int Set_Delay_Time (int val_delay_time );
    
    
    /** Set Highest Limit Temperature
    * @param mode
    *    minimum >>  Ox00    >>  decimal  0
    *    maximum >>  0x96    >>  decimal  150
    */
    int Set_Temperature_Max (int val_temperature );
    
    
    
    /** Set the state of LED 
    * @param mode
    *    0 = off, default
    *    1 = on 
    */  
    int Set_Etat_LED(int etat);
    
    

    /** Set the mode of the servo
     * @param mode
     *    0 = Positional, default
     *    1 = Continuous rotation
     */
    int Set_Mode(int mode);

    /** Set baud rate of all attached servos
     * @param mode
     *    0x01 = 1,000,000 bps
     *    0x03 =   500,000 bps
     *    0x04 =   400,000 bps
     *    0x07 =   250,000 bps
     *    0x09 =   200,000 bps
     *    0x10 =   115,200 bps
     *    0x22 =    57,600 bps
     *    0x67 =    19,200 bps
     *    0xCF =     9,600 bp
     */
    int Set_Baud(int baud);


    /** Set goal angle in integer degrees, in positional mode
     *
     * @param degrees 0-300
     * @param flags, defaults to 0
     *    flags[0] = blocking, return when goal position reached
     *    flags[1] = register, activate with a broadcast trigger
     *
     */
    int Set_Goal(int degrees, int flags = 0);

    int Set_Goal_speed(int degrees, int flags = 0);

    /** Set the speed of the servo in continuous rotation mode
     *
     * @param speed, -1.0 to 1.0
     *   -1.0 = full speed counter clock wise
     *    1.0 = full speed clock wise
     */
    int Set_CR_Speed(float speed);


    /** Permet de définir l'angle limite minimum de l'actionneur ( prend une valeur d'entrée en degres )
    //  minimum >>  0°   
    //  maximum >>  300°  
    //  deflaut >>  0°        
    */
    int Set_CW_Angle_Limit(int degrees);

    /** Permet de définir l'angle limite maximum de l'actionneur ( prend une valeur d'entrée en degres )
    //  minimum >>  0°   
    //  maximum >>  300°  
    //  deflaut >>  300°        
    */
    int Set_CCW_Angle_Limit(int degrees);

    // Change the ID

    /** Change the ID of a servo
     *
     * @param CurentID 1-255
     * @param NewID 1-255
     *
     * If a servo ID is not know, the broadcast address of 0 can be used for CurrentID.
     * In this situation, only one servo should be connected to the bus
     */
    int Set_ID(int CurrentID, int NewID);


    /** Poll to see if the servo is moving
     *
     * @returns true is the servo is moving
     */
    int isMoving(void);

    /** Send the broadcast "trigger" command, to activate any outstanding registered commands
     */
    void trigger(void);

    /** Send the "reset" command, to activate any outstanding registered commands
     */
    void reset();

    /** Read the current angle of the servo
     *
     * @returns float in the range 0.0-300.0
     */
    float Get_Position();

    /** Read the temperature of the servo
     *
     * @returns float temperature
     */
    float Get_Temp(void);

    /** Read the supply voltage of the servo
     *
     * @returns float voltage
     */
    float Get_Volts(void);
    
    // Lecture du couple maximum ( retourne la valeur du registre Max Torque de l'AX12 ) 
    int Get_Max_Torque (void);
    

    int read(int ID, int start, int length, char* data);
    int write(int ID, int start, int length, char* data, int flag=0);
    void multiple_goal_and_speed(int number_ax12,char* tab);
    float read_and_test(float angle,char* Tab);
    
private :

    SerialHalfDuplex _ax12;
    int _ID;
    int _baud;


};

#endif
