#include "Thermistor.h"
#include "math.h"

//#define ABSTMP (273.0)    ///< 絶対温度と０℃との差

/// コンストラクタ
/// @param B B定数
/// @param T0 抵抗値がR0Ωとなる摂氏温度
/// @param Rs0 T0℃でのサーミスタ抵抗値[Ω]、＋入力端子とGNDとの間に設置
Thermistor::Thermistor(double B, double T0, double Rs0)
{
    b = B; t0 = T0+ABSTMP;
    rs0 = Rs0;
    // 温度補正パラメタを「補正なし」に初期化
    a2 = 0.0;
    a1 = 1.0;
    a0 = 0.0;
};


/// AD変換結果から温度を算出する
/// @param adVal AD変換結果
/// @return 摂氏温度
double Thermistor::toTempValue(double rs)
{
    // サーミスタ抵抗値から温度を算出
    t = b * t0 / (b + t0 * log(rs/rs0));
    // 2次式による補正
    t = a2 * t * t + a1 * t + a0 - ABSTMP;
    return t;
}

/// 3点を指定し２次式で補正する温度校正
/// @param t1  補正点１の測定値
/// @param t1r 補正点１の正しい温度
/// @param t2  補正点２の測定値
/// @param t2r 補正点２の正しい温度
/// @param t3  補正点３の測定値
/// @param t3r 補正点３の正しい温度
void Thermistor::adjust(double t1, double t1r, double t2, double t2r, double t3, double t3r)
{
    t1  += ABSTMP;
    t1r += ABSTMP;
    t2  += ABSTMP;
    t2r += ABSTMP;
    t3  += ABSTMP;
    t3r += ABSTMP;
    // ２次式の係数を求める
    a2 = (t1r*(t2 - t3) + t2r*t3 - t2*t3r + t1*(-t2r + t3r))/((t1 - t2)*(t1 - t3)*(t2 - t3));
    a1 = (-(t2r*t3*t3) + t1r*(-t2*t2 + t3*t3) + t1*t1*(t2r - t3r) + t2*t2*t3r)/((t1 - t2)*(t1 - t3)*(t2 - t3));
    a0 = (t1r*t2*(t2 - t3)*t3 + t1*(-(t1*t2r*t3) + t2r*t3*t3 + t1*t2*t3r - t2*t2*t3r))/((t1 - t2)*(t1 - t3)*(t2 - t3));
}

/// 2点を指定し１次式で補正
/// @param t1  補正点１の測定値
/// @param t1r 補正点１の正しい温度
/// @param t2  補正点２の測定値
/// @param t2r 補正点２の正しい温度
void Thermistor::adjust(double t1, double t1r, double t2, double t2r)
{
    t1  += ABSTMP;
    t1r += ABSTMP;
    t2  += ABSTMP;
    t2r += ABSTMP;
    // １次式の係数を求める
    a2 = 0.0;
    a1 = (t1r - t2r)/(t1 - t2);
    a0 = (-(t1r*t2) + t1*t2r)/(t1 - t2);
}

