#ifndef AD7714_H
#define AD7714_H
#include "mbed.h"

/**
*  AD7714クラス
*
*   AD7714の設定し運用する
*   サーミスタ温度計に必要な機能を実装したものなので
*   AD7714のすべての機能を制御することはできない
*/
class AD7714{
private:

    static const int CmdRead = 0x08;
    static const int CmdWrite = 0x00;

    int channel;
    int data;

    SPI *spi;       ///< SPI通信のインスタンス


public:
    static const int FclkIn = 1000000;  ///< AD7714へのクロック　1MHz
    static const int bits = 24;         ///< 変換ビット長
    /// 内部レジスタを選択するための数値
    static struct CRegister {
        static const int Comun   = 0x00;    ///< コミュニケーションレジスタ
        static const int Mode    = 0x10;    ///< モードレジスタ
        static const int FltHi   = 0x20;    ///< フィルタ上位レジスタ
        static const int FltLo   = 0x30;    ///< フィルタ下位レジスタ
        static const int Test    = 0x40;    ///< 動作テスト、使用禁止
        static const int Data    = 0x50;    ///< データレジスタ
        static const int ZeroCal = 0x60;    ///< ゼロキャリブレーションレジスタ
        static const int FullCal = 0x70;    ///< フルスケールキャリブレーションレジスタ
    } Reg;  ///< レジスタを指定するための構造体
    /// 入力チャネルの指定
    static struct CCh {
        static const int A1A6 = 0;  ///< 擬似差動入力 ch1 - ch6
        static const int A2A6 = 1;  ///< 擬似差動入力 ch2 - ch6
        static const int A3A6 = 2;  ///< 擬似差動入力 ch3 - ch6
        static const int A4A6 = 3;  ///< 擬似差動入力 ch4 - ch6
        static const int A1A2 = 4;  ///< 差動入力 ch1 - ch2
        static const int A3A4 = 5;  ///< 差動入力 ch3 - ch4
        static const int A5A6 = 6;  ///< 差動入力 ch5 - ch6
        static const int A6A6 = 7;  ///< テスト入力 ch6 - ch6
    } Ch;   ///< 入力チャネルを指定するための構造体


    /// 内部レジスタに値を書き込む
    void write(int reg, int data);
    /// レジスタを読む
    int read(int reg);
    /// バイトデータを書き込む
    int writeByte(int txd);

    /// コンストラクタ
    AD7714();
    /// リセット
    void reset(void);
    /// ゼロとフルスケールのキャリブレーション
    void calib(void);
    /// 変換中かどうかの判定 @return ture 変換中
    bool isBusy();
    /// 入力チャネルを指定する
    void setCh(int ch){channel = ch;};
    /// デジタルフィルタを設定する
    void setFilter(int filter);
    /// 変換値を読み出す
    int getValue(){return read(CRegister::Data);};
};

/// AD7714によるレシオメトリックにより抵抗値を測定するADCクラス
/// AD7714クラスから派生
class RatioMetric7714 : public AD7714 {
private:
    double af;
    double rpu;
    double rmu;
    double rml;
    double rru;
    double rrl;
public:
    /**
    * コンストラクタ
    */
    RatioMetric7714(double Rpu, double Rmu, double Rml, double Rru, double Rrl);
    /// 指定されたAD値を抵抗値に変換
    double toResistorValue(int adcVal);
    /// 現在のAD値を抵抗値に変換して返す
    double getResistorValue(void);
};




#endif
