/** MARMEX_VB Camera control library
 *
 *  @class   MARMEX_VB
 *  @version 0.3
 *  @date    16-Jun-2014
 *
 *  Released under the Apache License, Version 2.0 : http://mbed.org/handbook/Apache-Licence
 *
 *  MARMEX_VB Camera control library for mbed
 */

#ifndef MBED_MARMEX_VB
#define MBED_MARMEX_VB

/** MARMEX_VB class
 *
 *  MARMEX_VB camera control library
 *  This driver provide controls and data transfer interface for MARMEX_VB
 *
 *  Example:
 *  @code
 *
 *  #include "mbed.h"
 *  #include "MARMEX_OB_oled.h"
 *  #include "MARMEX_VB.h"
 *
 *  MARMEX_OB_oled  oled1( p5, p7,  p20, p16, p15 );            // mosi, sclk, cs, rst, power_control             -- maple-mini-type-b-board-slot1
 *  MARMEX_VB       camera( p5, p6, p7, p22, p26, p28, p27 );   // mosi, miso, sclk, cs, reset, I2C_SDA, I2C_SCL  -- maple-mini-type-b-board-slot2
 *  BusOut          led( LED3, LED4 );
 *
 *  #define X_OFFSET        ((MARMEX_VB::PIXEL_PER_LINE - MARMEX_OB_oled::WIDTH ) / 2)
 *  #define Y_OFFSET        ((MARMEX_VB::LINE_PER_FRAME - MARMEX_OB_oled::HEIGHT) / 2)
 *
 *  int main()
 *  {
 *      led    = 0x3;
 *
 *      oled1.cls();
 *
 *      short   buf[ MARMEX_OB_oled::WIDTH ];
 *
 *      while ( 1 ) {
 *
 *          led    = 0x1;
 *          camera.open_transfer();
 *
 *          for ( int line = 0; line < 128; line++  ) {
 *              camera.read_a_line( buf, line + Y_OFFSET, X_OFFSET, 128 );
 *              oled1.blit565( 0, line, 128, 1, buf );
 *          }
 *
 *          camera.close_transfer();
 *          led    = 0x2;
 *     }
 *  }
 *  @endcode
 */

#define OPTIMIZATION_ENABLED

#define  DEFAULT_PICTURE_SIZE   QCIF
//#define  DEFAULT_PICTURE_SIZE   VGA
//#define  DEFAULT_PICTURE_SIZE   QVGA
//#define  DEFAULT_PICTURE_SIZE   QQVGA

class MARMEX_VB
{
public:
    typedef enum  {
        QCIF    = 1,            /**< QCIF       */
        VGA,                    /**< VGA        */
        QVGA,                   /**< QVGA       */
        QQVGA,                  /**< QQVGA      */
    } CameraResolution;

    typedef enum  {
        OFF    = 0,             /**< ON         */
        ON,                     /**< OFF        */
    } SwitchState;

    /** General parameters for MARMEX_VB */
    enum  {
        BYTE_PER_PIXEL  = 2,    /**< bytes per pixel  */

        QCIF_PIXEL_PER_LINE     = 176,  /**< pixels in a line (QCIF horizontal size)  */
        QCIF_LINE_PER_FRAME     = 144,  /**< lines in a frame (QCIF vertical size)    */
        VGA_PIXEL_PER_LINE      = 640,  /**< pixels in a line (VGA  horizontal size)  */
        VGA_LINE_PER_FRAME      = 480,  /**< lines in a frame (VGA  vertical size)    */
    };

    /** General parameters for MARMEX_OB_oled */
    enum  {
        NO_ERROR        = 0     /**< zero  */
    };

    /** Create a MARMEX_VB instance connected to specified SPI, DigitalOut and I2C pins
     *
     * @param SPI_mosi   SPI-bus MOSI pin
     * @param SPI_miso   SPI-bus MISO pin
     * @param SPI_sclk   SPI-bus SCLK pin
     * @param SPI_cs     SPI-bus Chip Select pin
     * @param cam_reset  Camera reset pin
     * @param I2C_sda    I2C-bus SDA pin
     * @param I2C_scl    I2C-bus SCL pin
     */

    MARMEX_VB(
        PinName SPI_mosi,
        PinName SPI_miso,
        PinName SPI_sck,
        PinName SPI_cs,
        PinName cam_reset,
        PinName I2C_sda,
        PinName I2C_scl
    );

    /** Initialization
     *
     *  Performs MARMEX_VB reset and initializations
     *  This function is called from MARMEX_VB constoructor. So user don't have to call in the user code.
     *
     *  This function takes about 2 seconds because there is 99 times I2C access with 20ms interval.
     *
     *  @param res   select camera resolution : QCIF(default), VGA, QVGA or QQVGA
     */
    int init( CameraResolution res = QCIF );

    /** Color bar ON/OFF
     *
     *  Set colorbar ON/OFF
     *
     *  @param sw   turn-ON or -OFF colorbar : ON or OFF
     */
    void colorbar( SwitchState sw );

    /** Get holizontal size
     *
     *  Returns horizontal image size (pixels)
     *
     *  @return holizontal size
     */
    int get_horizontal_size( void );
    
    /** Get vertical size
     *
     *  Returns vertical image size (pixels)
     *
     *  @return vertical size
     */
    int get_vertical_size( void );

    /** Check camera availability
     *
     *  Returns last I2C access result
     *  This returns non-zero value if the camera initialization failed
     *
     *  @return error code in init function (I2C API return value)
     */
    int ready( void );

    /** Open transfer
     *
     *  Let the MARMEX_VB get ready to transfer the data.
     *  When this function is called, the camera will stop updating buffer (on camera board) at end of frame.
     *
     *  @return error code in init function (I2C API return value)
     */
    void open_transfer( void );

    /** Close transfer
     *
     *  Letting the MARMEX_VB to know the data transfer done.
     *  This function should be called when the data transfer done to resume the buffer update by camera
     */
    void close_transfer( void );

    /** Read one line data
     *
     *  Reads 1 line data from MARMEX_VB
     *  This function should be called when the data transfer done to resume the buffer update by camera
     *
     * @param *p            pointer to array of short
     * @param line_number   to select which line want to read
     * @param x_offset      holizontal offset (from left) to start the read
     * @param n_of_pixels   pixels to be read
     */
    void read_a_line( short *p, int line_number, int x_offset, int n_of_pixels );
    
    /** Read order change
     *
     *  Toggles flag for read order of 2 byte data. 
     *
     *  @return new status of the read order flag (false = normal / true = swapped)
     */
    int read_order_change( void );
    
private:
    int     send_spi( char data );
    void    write_register( char reg, char value );
    int     read_register( char reg );
    void    set_address( int address );

    SPI         _spi;
    DigitalOut  _cs;
    DigitalOut  _reset;
    I2C         _i2c;
    int         _error_state;
    int         _horizontal_size;
    int         _vertical_size;
    int         _read_order_change;

};

#endif  //  MBED_MARMEX_VB