#include "mbed.h"
#include "ADXL362.h"
 
// Interface pulled from ADXL362.cpp
// ADXL362::ADXL362(PinName CS, PinName MOSI, PinName MISO, PinName SCK) :
ADXL362 adxl362(PA_0,PA_7,PA_6,PA_1);
Serial pc(USBTX, USBRX);
DigitalOut myled(LED1);

int adxl362_reg_print(int start, int length);
void adxl362_get_average(int16_t avg[]);

int main() {
    pc.printf("Starting program\r\n");
    adxl362.reset();
    wait_ms(600); // we need to wait at least 500ms after ADXL362 reset
    adxl362.set_mode(ADXL362::MEASUREMENT);
    int16_t x,y,z; 
    int16_t avg[3]; // Avg values of x,y,z. x = avg[0], y = avg[1] z = avg[2]
    int countOverThreshHold = 0;
    adxl362_reg_print(1,0); // Test of axdl_reg_print
    wait_ms(1000); // wait so that values can be seen
    int threshHoldRange = 30; // threshhold difference between avg and read values
    adxl362_get_average(avg);// Generate an average of what x,y,z are.
    //pc.printf("xAvg = %d yAvg = %d zAvg = %d\r\n",avg[0],avg[1],avg[2]);
    while(1) {
        x=adxl362.scanx();
        y=adxl362.scany(); 
        z=adxl362.scanz();
        
        if (x > (avg[0] + threshHoldRange) || x < (avg[0] - threshHoldRange)
            || y > (avg[1] + threshHoldRange) || y < (avg[1] - threshHoldRange) 
            || z > (avg[2] + threshHoldRange) || z < (avg[2] - threshHoldRange)) {
            // commented out z as it gives weird values
            // ) {
            /* print cases used to test threshhold range
            pc.printf("x = %d y = %d z = %d\r\n",x,y,z);
            pc.printf("xAvg = %d yAvg = %d zAvg = %d\r\n",avg[0],avg[1],avg[2]);
            pc.printf("Threshold range is = %d\r\n", threshHoldRange);
            */
            //pc.printf("outside of threshold range\n\r");
            countOverThreshHold++;
            pc.printf("Gone over threshhold %d times\n\r", countOverThreshHold);
            myled = 1; // LED is ON
            wait_ms(1900); // Wait 1.9s.  Last .1 sec is done while getting average
            adxl362_get_average(avg); // Generate new average
            myled = 0; // LED is OFF
        }
        
        pc.printf("x = %d y = %d z = %d\r\n",x,y,z);
        wait_ms(100);
    }
}

void adxl362_get_average(int16_t avg[]) {
     // Generate an average of what x,y,z are.
    int numberOfPolls = 2;
    // reset average back to zero for each
    int tmp[3]; // required because avg was having integer overflow
    tmp[0] = 0;
    tmp[1] = 0;
    tmp[2] = 0;
    avg[0] = 0;
    avg[1] = 0;
    avg[2] = 0;
    int16_t x,y,z; 
    // Poll each value 10 times
    for (int i = 0; i < numberOfPolls; i++) {
        //pc.printf("x = %d y = %d z = %d\r\n",x,y,z);
        x = adxl362.scanx();
        tmp[0]+= x;
        y = adxl362.scany();
        tmp[1]+= y;
        z=adxl362.scanz();
        tmp[2]+= z; 
        wait_ms(10);
        //pc.printf("xAvg = %d yAvg = %d zAvg = %d\r\n",tmp[0],tmp[1],tmp[2]);

    } 
    // Divide each value by the number of polls to get the average
    tmp[0] = tmp[0] / numberOfPolls;
    tmp[1] = tmp[1] / numberOfPolls;
    tmp[2] = tmp[2] / numberOfPolls;
    // store values from the tmp to the passed in array
    avg[0] = tmp[0];
    avg[1] = tmp[1];
    avg[2] = tmp[2];
    pc.printf("New Average: xAvg = %d yAvg = %d zAvg = %d\r\n",avg[0],avg[1],avg[2]);
    wait_ms(5000);
}

int adxl362_reg_print(int start, int length) {
    int end = 0x2E;
    if ((start + length) < end) {
        end = (start + length); // so it only goes to the length
    }
    if (length == 0) { 
        end = 0x2E; // so it prints till the end
    }
    
    
    // Check if start is within registry
    if (start < 0 || start > 0x2E) {
        pc.printf("Error: start value passed to adxl362_reg_print outside of range of registry\n\r");
        return -1;
    }
    // check if length is negative
    if (length < 0) {
        pc.printf("Error: length passed to adxl362_reg_print is negative\n\r");
        return -1;
    }
    
    // check if valid communication with device going
    if (adxl362.read_reg(adxl362.DEVID_AD) != 0xAD) {
        pc.printf("Error: Unable to read from DEVID_AD register\n\r");
        return -1;
    } 
    // String array with all of the names of the different registers in order
    char regNames [40][20]  = {
         "DEVID_AD", "DEVID_MST", "PARTID",
         "REVID", "XDATA", "YDATA",
         "ZDATA", "STATUS", "FIFO_ENTRIES_L", 
         "FIFO_ENTRIES_H", "XDATA_L", "XDATA_H",
         "YDATA_L", "YDATA_H", "ZDATA_L",
         "ZDATA_H", "TEMP_L", "TEMP_H",
         "RESERVED", "RESERVED", "SOFT_RESET",
         "THRESH_ACT_L", "THRESH_ACT_H", "TIME_INACT_L",
         "TIME_ACT", "THRESH_INACT_L", "THRESH_INACT_H",
         "TIME_INACT_L", "TIME_INACT_H", "ACT_INACT_CTL",
         "FIFO_CONTROL", "FIFO_SAMPLES", "INTMAP1",
         "INTMATP2", "FILTER_CTL", "POWER_CTL",
         "SELF_TEST"};
    
    
    for (int hexValue = 0; hexValue < end; hexValue++) { 
        uint8_t tmp = 0; // hex value location of the register
        uint8_t arrayAddress = 0; // Hex value of the register
        // May need to increment the registers inside of the if, since the values 
        // are not always sequential
        // You can find registers inside of ADXL362.h file in ADXL362 folder
        // they are the ADXL362_register_t
        if (hexValue == 0) { 
            tmp = adxl362.read_reg(adxl362.DEVID_AD);
            arrayAddress = 0;
        } else if (hexValue == 1) { 
            tmp = adxl362.read_reg(adxl362.DEVID_MST);
            arrayAddress = 1;
        } else if (hexValue == 2) { 
            tmp = adxl362.read_reg(adxl362.PARTID);
            arrayAddress = 2;
        } else if (hexValue == 3) { 
            tmp = adxl362.read_reg(adxl362.REVID);
            arrayAddress = 3;
        } else if (hexValue == 8) { 
            tmp = adxl362.read_reg(adxl362.XDATA);
            arrayAddress = 4;
        } else if (hexValue == 9) { 
            tmp = adxl362.read_reg(adxl362.YDATA);
            arrayAddress = 5;
        } else if (hexValue == 0x0A) { 
            tmp = adxl362.read_reg(adxl362.ZDATA);
            arrayAddress = 6;
        } else if (hexValue == 0x0B) { 
            tmp = adxl362.read_reg(adxl362.STATUS);
            arrayAddress = 7;
        } else if (hexValue == 0x0C) { 
            tmp = adxl362.read_reg(adxl362.FIFO_ENTRIES_L);
            arrayAddress = 8;
        } else if (hexValue == 0x0D) {
            tmp = adxl362.read_reg(adxl362.FIFO_ENTRIES_H);
            arrayAddress = 9;
        } else if (hexValue == 0x0E) { 
            tmp = adxl362.read_reg(adxl362.XDATA_L);
            arrayAddress = 9;
        } else if (hexValue == 0x0F) { 
            tmp = adxl362.read_reg(adxl362.XDATA_H);
            arrayAddress = 10;
        } else if (hexValue == 0x10) { 
            tmp = adxl362.read_reg(adxl362.YDATA_L);
            arrayAddress = 11;
        } else if (hexValue == 0x11) { 
            tmp = adxl362.read_reg(adxl362.YDATA_H);
            arrayAddress = 12;
        } else if (hexValue == 0x12) { 
            tmp = adxl362.read_reg(adxl362.YDATA_H);
            arrayAddress = 13;
        } else if (hexValue == 0x13) { 
            tmp = adxl362.read_reg(adxl362.ZDATA_L);
            arrayAddress = 14;
        } else if (hexValue == 0x14) { 
            tmp = adxl362.read_reg(adxl362.ZDATA_H);
            arrayAddress = 15;
        } else if (hexValue == 0x15) { 
            // RESERVED
        } else if (hexValue == 0x16) { 
            // RESERVED
        } else if (hexValue == 0x1F) { 
            tmp = adxl362.read_reg(adxl362.SOFT_RESET);
            arrayAddress = 18;
        } else if (hexValue == 0x20) { 
            tmp = adxl362.read_reg(adxl362.THRESH_ACT_L);
            arrayAddress = 19;
        } else if (hexValue == 0x21) { 
            tmp = adxl362.read_reg(adxl362.THRESH_ACT_H);
            arrayAddress = 20;
        } else if (hexValue == 0x22) { 
            tmp = adxl362.read_reg(adxl362.TIME_ACT);
            arrayAddress = 21;
        } else if (hexValue == 0x23) { 
            tmp = adxl362.read_reg(adxl362.THRESH_INACT_L);
            arrayAddress = 22;
        } else if (hexValue == 0x24) { 
            tmp = adxl362.read_reg(adxl362.THRESH_INACT_H);
            arrayAddress = 23;
        } else if (hexValue == 0x25) { 
            tmp = adxl362.read_reg(adxl362.TIME_INACT_L);
            arrayAddress = 24;
        } else if (hexValue == 0x26) { 
            tmp = adxl362.read_reg(adxl362.TIME_INACT_H);
            arrayAddress = 25;
        } else if (hexValue == 0x27) { 
            tmp = adxl362.read_reg(adxl362.ACT_INACT_CTL);
            arrayAddress = 26;
        } else if (hexValue == 0x28) { 
            tmp = adxl362.read_reg(adxl362.FIFO_CONTROL);
            arrayAddress = 27;
        } else if (hexValue == 0x29) { 
            tmp = adxl362.read_reg(adxl362.FIFO_SAMPLES);
            arrayAddress = 28;
        } else if (hexValue == 0x2A) { 
            tmp = adxl362.read_reg(adxl362.INTMAP1);
            arrayAddress = 29;
        } else if (hexValue == 0x2B) { 
            tmp = adxl362.read_reg(adxl362.INTMAP2);
            arrayAddress = 30;
        } else if (hexValue == 0x2C) { 
            tmp = adxl362.read_reg(adxl362.FILTER_CTL);
            arrayAddress = 31;
        } else if (hexValue == 0x2D) { 
            tmp = adxl362.read_reg(adxl362.POWER_CTL);
            arrayAddress = 32;
        } else if (hexValue == 0x2E) { 
            tmp = adxl362.read_reg(adxl362.SELF_TEST);
            arrayAddress = 33;
        }
        if (hexValue != 0 && tmp != 0) {
            pc.printf("%#04x: %s=%#04x\n\r", hexValue, regNames[arrayAddress], tmp); // Print register
            // set both to zero so they don't show again
            arrayAddress = 0; 
            tmp = 0;
        }
    }
    return 0;
    
// below is github with data for ADXL362 methods
 //https://github.com/analogdevicesinc/mbed-adi/blob/master/libraries/ADXL362/ADXL362.cpp   
}