/*
 * ble_interface.cpp - main interface to android application
 *
 * Author: Jake Greaves
 */
#include "ble_interface.h"

//adisense1000 config as defined elsewhere
extern ADI_SENSE_CONFIG smartcushion_adisense_config;
extern ADI_SENSE_DEVICE_HANDLE hDevice;

extern ADI_SENSE_1000_LUT_DESCRIPTOR *sample_lut_desc_list[];
extern ADI_SENSE_1000_LUT_TABLE_DATA *sample_lut_data_list[];
extern unsigned                       sample_lut_num_tables;

//ble uart device
extern Serial bleSerialDevice;
extern DigitalOut ble_Cts;

//a work around for multiple requests received from app
char lastFieldNameID = 0xFF;


static char *fieldNames[ADI_SENSE_1000_MAX_CHANNELS] = {
	"Cjc0",
	"Cjc1",
	"Sensor0",
	"Sensor1",
	"Sensor2",
	"Sensor3",
	"Voltage0",
	"Current0",
	"I2c0",
	"I2c1",
	"Spi0",
	"Spi1",
	"Spi2"
};

//send error response
static uint32_t Ble_HandleError( ErrorType error );


//command ID to be handled
enum {
	REQ_REG_PACKET_COUNT 	= 0x21, //'!'
	REQ_REG_PACKETS 		= 0x40, //'@'
	REQ_FIELD_NAMES 		= 0x23, //'#'
	START_STREAM 			= 0x24, //'$'
	STOP_STREAM 			= 0x25, //'%'
} MSG_TYPE;

/*
enum {
	REQ_REG_PACKET_COUNT 	= 0x10,
	REQ_REG_PACKETS 		= 0x11,
	REQ_FIELD_NAMES 		= 0x12,
	START_STREAM 		= 0x20,
	STOP_STREAM 		= 0x21,
} MSG_TYPE;
*/
//possible measurement types
enum {
	TEMPERATURE = 0x00,
	LOADCELL = 1,
	PRESSURE = 2,
	HUMIDITY = 3,
	ACCELEROMETER = 4,
	UNDEFINED
} MEASUREMENT_TYPES;

//data type. Might change to bytes per measurement in newer API versions
enum {
	FLOAT_32 = 0x01
} DATA_TYPE;

//structs to store an active channels details for the BLE App
typedef struct {
	uint8_t measurementId;
	uint8_t measurementTypeId;
	char *measurementName;
	uint8_t numberOfFields;
	uint8_t associatedChannel;
	uint8_t dataType;
} MEASUREMENT_INFO;


//store all active channel information
static MEASUREMENT_INFO *measurementInfo = NULL;

//counts amount of channels active
static uint8_t activeChannelCount = 0;

//flag for streaming data
volatile static bool bStreamFlag = 0;


/***************************************************************
* Function Name: Ble_HandleReqRegPacketCount
* Description  : handle the REQ_REG_PACKET_COUNT command
****************************************************************/
static uint32_t Ble_HandleReqRegPacketCount( void )
{
	//buffer for sending BLE data
	char bleTxBuffer[BLE_PACKET_SIZE] = {0x00};
	//stores position in bleTxBuffer
	uint8_t buffPos = 0;
	
	//clear active channel count
	activeChannelCount = 0;
	
	//clear global array for aactive channels, as we are recalculating this now
	free( measurementInfo );
	measurementInfo = NULL;
	
	//check all channels
	for( uint32_t i = 0; i < ADI_SENSE_1000_MAX_CHANNELS; i++ ) {
		
		//struct to contain data on measurement
		MEASUREMENT_INFO channel_info;
		
		//current channel
		ADI_SENSE_1000_CHANNEL_CONFIG channel = smartcushion_adisense_config.adisense1000.channels[i];
		
		//include as independant channel if enabled and not a compensation channel
		//for now, compensation cahnnels get their own graph if publishing is enabled
		if( channel.enableChannel && !channel.disablePublishing ) {
			
			//add a measurement
			channel_info.measurementId = activeChannelCount;
			activeChannelCount++;
			
			//set name and units
			//check channel type based on ranges and assign name and type
			if( channel.adcChannelConfig.sensor <= ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_4_ADV_L2 ) {
				channel_info.measurementTypeId = TEMPERATURE;
				channel_info.measurementName = "Temperature";
			}
			
			else if( channel.adcChannelConfig.sensor >= ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_1_DEF_L2 && channel.adcChannelConfig.sensor <= ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_4_DEF_L2 ) {
				channel_info.measurementTypeId = LOADCELL;
				channel_info.measurementName = "Load cell";
			}
			
			else if( channel.spiChannelConfig.sensor >= ADI_SENSE_1000_SPI_SENSOR_PRESSURE_A_DEF_L1 && channel.spiChannelConfig.sensor <= ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_B_ADV_L2  ) {
				//ADI_SENSE_1000_SPI_SENSOR_PRESSURE_A_DEF_L1
				channel_info.measurementTypeId = ACCELEROMETER;
				channel_info.measurementName = "Accelerometer";
			}
			
			else if( channel.adcChannelConfig.sensor >= ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_A_DEF_L1 && channel.adcChannelConfig.sensor <= ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_B_DEF_L1 ) {
				channel_info.measurementTypeId = HUMIDITY;
				channel_info.measurementName = "Humidity";
			}
			
			else if( channel.adcChannelConfig.sensor >= ADI_SENSE_1000_ADC_SENSOR_VOLTAGE || channel.adcChannelConfig.sensor == ADI_SENSE_1000_SPI_SENSOR_PRESSURE_A_DEF_L1 ) {
				channel_info.measurementTypeId = PRESSURE;
				channel_info.measurementName = "Pressure";
			}
			
			else {
				channel_info.measurementTypeId = UNDEFINED;
				channel_info.measurementName = "Undefined";
			}
			
			//default to 1 field
			channel_info.numberOfFields = 1;
			
			//add current channel
			channel_info.associatedChannel = i;
			
			//all values are floats for now
			channel_info.dataType = FLOAT_32;
			
			//add to overall array
			measurementInfo = ( MEASUREMENT_INFO * )realloc( measurementInfo, activeChannelCount * sizeof( MEASUREMENT_INFO ) );
			measurementInfo[activeChannelCount - 1] = channel_info;
		}
	}
	
	if( activeChannelCount == 0 ) {
		return Ble_HandleError( ERROR_UNDEFINED );
	}
	
	//command ID
	bleTxBuffer[buffPos] = REQ_REG_PACKET_COUNT;
	buffPos++;
	
	//amount of data samples
	bleTxBuffer[buffPos] = (48 + activeChannelCount); //adding 48 decimal to convert to ascii
	buffPos++;
	
	//respond to app
	Bl652_Write( bleTxBuffer, buffPos );
	
	return 0;
}

/***************************************************************
* Function Name: Ble_HandleReqRegPackets
* Description  : handle the REQ_REG_PACKETS command
****************************************************************/
static uint32_t Ble_HandleReqRegPackets( void )
{
	//buffer for sending BLE data
	char bleTxBuffer[BLE_PACKET_SIZE] = {0x00};
	//stores position in bleTxBuffer
	uint8_t buffPos = 0;
	
	//ReqRegPacketCount has not been called, or had an error
	if( activeChannelCount == 0 ) {
		return Ble_HandleError( ERROR_UNDEFINED );
	}
	
	//repeat for all measurements
	for( uint32_t i = 0; i < activeChannelCount; i++ ) {
		
		//reset buffPos
		buffPos = 0;
		
		//add command ID
		bleTxBuffer[buffPos] = REQ_REG_PACKETS;
		buffPos++;
		
		//add regPackets ID
		bleTxBuffer[buffPos] = (48 + measurementInfo[i].measurementId);
		buffPos++;
		
		//add type of measurement
		bleTxBuffer[buffPos] = (48 + measurementInfo[i].measurementTypeId);
		buffPos++;
		
		//adding measurements name to packet
		char *measurementName = ( char* )measurementInfo[i].measurementName;
		for( uint32_t j = 0; j < MEASUREMENT_NAME_MAX_LEN; j++ ) {
			//populate measurement name for app
			if( j < strlen( measurementName ) ) {
				bleTxBuffer[buffPos] = measurementName[j];
			}
			//pad unused chars with 0x00
			else {
				bleTxBuffer[buffPos] = 0x00;
			}
			buffPos++;
		}
		
		//add number of plots on graph. ( 1 for now )
		bleTxBuffer[buffPos] = (48 + measurementInfo[i].numberOfFields);
		buffPos++;
		//add data format ( always float32_t for now )
		bleTxBuffer[buffPos] = (48 + measurementInfo[i].dataType);
		buffPos++;
	
		//transmit packet
		Bl652_Write( bleTxBuffer, buffPos );
	}
	
	return 0;
}

/***************************************************************
* Function Name: Ble_HandleReqFieldNames
* Description  : handle the REQ_FIELD_NAMES command
****************************************************************/
static uint32_t Ble_HandleReqFieldNames( char *bleRxBuffer ) 
{
	//buffer for sending BLE data
	char bleTxBuffer[BLE_PACKET_SIZE] = {0x00};
	//stores position in bleTxBuffer
	uint8_t buffPos = 0;
	
	//flag for reading from BLE
	volatile bool bBleMessageReceived = 0;
	uint8_t numberOfFields;
	
	//read a further byte
	Bl652_Read( &bleRxBuffer[1], 1 );
	
	//if ReqRegPacketCount has not been called, or had an error
	if( activeChannelCount == 0 ) {
		return Ble_HandleError( ERROR_UNDEFINED );
	}
	
	//dirtiest of hacks. Issue here is app seems to send same packet multiple times when sent once. 
	//Unsure why this is for now, ignore duplicates
	if( bleRxBuffer[1] == lastFieldNameID ) {
		return 0;
	}
	lastFieldNameID = bleRxBuffer[1];
	
	//find units for channel
	char unitDescriptor[UNIT_NAME_MAX_LEN];
	switch( measurementInfo[bleRxBuffer[1]].measurementTypeId ) {
		
	  case TEMPERATURE:
		strcpy( unitDescriptor, "°C" );
		break;
	
	case LOADCELL:
		strcpy( unitDescriptor, "Kg" );
		break;
		
	  case PRESSURE:
		strcpy( unitDescriptor, "PSI" );
		break;
		
	  case HUMIDITY:
		strcpy( unitDescriptor, "%" );
		break;
		
	  case ACCELEROMETER:
		strcpy( unitDescriptor, "XYZ" );
		break;
		
	  default:
		strcpy( unitDescriptor, "NA" );
		break;
	}
	
	//add command ID to BLE packet
	bleTxBuffer[buffPos] = REQ_FIELD_NAMES;
	buffPos++;
	
	//add unit name to ble packet, padding with 0x00
	for( uint32_t i = 0; i < UNIT_NAME_MAX_LEN; i++ ) {
		//populate measurement name for app
		if( i < strlen( unitDescriptor ) ) {
			bleTxBuffer[buffPos] = unitDescriptor[i];
		}
		//pad with 0x00
		else {
			bleTxBuffer[buffPos] = 0x00;
		}
		buffPos++;
	}
	
	//name of sensor fields
	numberOfFields = measurementInfo[bleRxBuffer[1]].numberOfFields;
	
	//ReqRegPacketCount has not been called, or had an error
	if( numberOfFields == 0 ) {
		return Ble_HandleError( ERROR_UNDEFINED );
	}
	
	//calculate the available space for names of each plot.
	//This is support for multiple channels associated together.
	uint8_t eachFieldNameMaxSize = FIELD_NAMES_MAX_SIZE / numberOfFields;
	
	//add field names to packet
	for( uint32_t i = 0; i < numberOfFields; i++ ) {
		char *fieldName = fieldNames[measurementInfo[bleRxBuffer[1]].associatedChannel];
		
		for( uint32_t j = 0; j < eachFieldNameMaxSize; j++ ) {
			//populate measurement name for app
			if( j < strlen( fieldName ) ) {
				bleTxBuffer[buffPos] = fieldName[j];
			}
			//pad with 0x00
			else {
				bleTxBuffer[buffPos] = 0x00;
			}
			buffPos++;
		}
	}
	
	//return measurement count, field name
	Bl652_Write( bleTxBuffer, buffPos );
	
	return 0;
}

/***************************************************************
* Function Name: ble_data_ready_callback
* Description  : callback for when data ready state changes
****************************************************************/
static void ble_data_ready_callback( ADI_SENSE_GPIO_PIN ePinId, void *pArg )
{
	volatile bool_t *pbDataready = ( volatile bool_t * ) pArg;
   	*pbDataready = true;
}

/***************************************************************
* Function Name: Ble_HandleStartStream
* Description  : handle the START_STREAM command
****************************************************************/
static uint32_t Ble_HandleStartStream( void )
{
	//buffer for sending BLE data
	char bleTxBuffer[BLE_PACKET_SIZE] = {0x00};
	//stores poition in bleTxBuffer
	uint8_t buffPos = 0;

	//part of the adisense1000 sampling
	ADI_SENSE_RESULT res;
	ADI_SENSE_DATA_SAMPLE *pSampleBuffer;
	ADI_SENSE_1000_OPERATING_MODE eOperatingMode;
	ADI_SENSE_1000_DATAREADY_MODE eDataPublishMode;
	ADI_SENSE_MEASUREMENT_MODE eMeasurementMode = ADI_SENSE_MEASUREMENT_MODE_NORMAL;
	uint32_t nSamplesPerDataready, nSamplesPerCycle;
	uint8_t nBytesPerSample;
	volatile bool_t bDataReady = false;
	
	//if ReqRegPacketCount has not been called, or had an error
	if( activeChannelCount == 0 ) {
		return Ble_HandleError( ERROR_UNDEFINED );
	}
	
	//upload new configuration and apply. 
	//This has been used to calculate all previous steps, so ensure that this is the active configuration
	adi_sense_SetConfig( hDevice, &smartcushion_adisense_config );
	
	unsigned lutBufferSize = ADI_SENSE_LUT_MAX_SIZE;
	ADI_SENSE_1000_LUT *pLutBuffer = (ADI_SENSE_1000_LUT *) ::operator new (lutBufferSize);
    if (pLutBuffer == NULL)
    {
        ADI_SENSE_LOG_ERROR("Failed to allocate memory for user-defined LUT data buffer");
        return ADI_SENSE_NO_MEM;
    }

    ADI_SENSE_LOG_INFO("Assembling LUT data");
    res = adi_sense_1000_AssembleLutData(pLutBuffer, lutBufferSize,
                                         sample_lut_num_tables,
                                         sample_lut_desc_list,
                                         sample_lut_data_list);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to assemble user-defined LUT data");
        return res;
    }
    
    /*
     * Write assembled user-defined Look-Up Table data structure to the device
     * User-defined LUT data is not applied until adi_sense_ApplyConfigUpdates() is called.
     */
    ADI_SENSE_LOG_INFO("Setting LUT data");
    res = adi_sense_1000_SetLutData(hDevice, pLutBuffer);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to set user-defined LUT data");
        return res;
    }

    delete pLutBuffer;
    
	
	adi_sense_ApplyConfigUpdates( hDevice );
	ADI_SENSE_LOG_INFO("ApplyConfig update applied");
	
	//get information about the adisense data ready mode
	adi_sense_1000_GetDataReadyModeInfo(hDevice,
							eMeasurementMode,
							&eOperatingMode,
							&eDataPublishMode,
							&nSamplesPerDataready,
							&nSamplesPerCycle,
							&nBytesPerSample);
	//allocate a buffer to store the samples retreived on each DATAREADY pulse
	pSampleBuffer = new ADI_SENSE_DATA_SAMPLE[nSamplesPerDataready];
	if( pSampleBuffer == NULL ) {
		return 1;
	}
	
	//interrupt-triggered DATAREADY notifications in this mode
	adi_sense_RegisterGpioCallback( hDevice, ADI_SENSE_GPIO_PIN_DATAREADY,
						ble_data_ready_callback,
						( void * )&bDataReady );
	
	//allow ble to send data for stream cancel command
	ble_Cts = 0;
	
	//handle the single cycle device mode
	if ( eOperatingMode == ADI_SENSE_1000_OPERATING_MODE_SINGLECYCLE ) {
		
		//loop single cycle until stream stop command is received
		do {
			//begin measurement cycles
			adi_sense_StartMeasurement( hDevice, eMeasurementMode );
			
			//read in each measurement cycle
			for ( uint32_t i = 0; i < nSamplesPerCycle / nSamplesPerDataready; i++ ) {
				uint32_t nReturned;
				
				//Interrupt method: wait for DATAREADY interrupt callback
				while ( !bDataReady );
				
				//reset bDataReady flag to detect the next callback
				bDataReady = false;
				
				//read data into structure
				adi_sense_GetData( hDevice, eMeasurementMode, pSampleBuffer, nBytesPerSample, nSamplesPerDataready, &nReturned );
				
				//for all returned samples
				for( uint32_t i = 0; i < nReturned; i++ ) {
					
					//to store adisense1000 channel index 
					//and the channel number according to the BLE device
					uint8_t channelIndex, activeChannelNumber;
					
					//obtain sample data in byte array form
					uint8_t *sample = ( uint8_t* )&pSampleBuffer[i].processedValue;
					
					//match sample ID to active channel
					for( uint32_t j = 0; j < activeChannelCount; j++ ) {
						
						//get the ID of the current active channel
						channelIndex = measurementInfo[j].associatedChannel;
						
						//if the sample matches the current active channel
						if( pSampleBuffer[i].channelId == channelIndex ) {
							activeChannelNumber = j;
							break;
						}
					}
					
					//clear buffPos
					buffPos = 0;
					
					//tag samples with command value
					bleTxBuffer[buffPos] = START_STREAM;
					buffPos++;
					
					//Tag with ID of measurement
					bleTxBuffer[buffPos] = measurementInfo[activeChannelNumber].measurementId;
					buffPos++;
						
					//transmute sample data into bytes for transmission
					for( uint32_t j = 0; j < sizeof( measurementInfo[activeChannelNumber].dataType ); j++ ) {
						
						//store bytes to buffer
						bleTxBuffer[buffPos] = sample[j];
						buffPos++;
					}
					
					//stream measurement
					Bl652_Write( bleTxBuffer, buffPos );
				}
			}
			
			if( bleSerialDevice.readable() ) {
				ble_Cts = 1;
				if( bleSerialDevice.getc() == STOP_STREAM ) {
					//stop stream measurement
					bleTxBuffer[0] = STOP_STREAM;
					Bl652_Write( bleTxBuffer, 1 );
					break;
				}
				else {
					return Ble_HandleError( ERROR_UNDEFINED );
				}
			}			
		} while ( 1 );
	}

	//handle continuous measurement mode
	else {

		//begin measurement cycles
		adi_sense_StartMeasurement( hDevice, eMeasurementMode );
	
		do {
			uint32_t nReturned;
		
			//Interrupt method: wait for DATAREADY interrupt callback
			while ( !bDataReady );
		
			//reset bDataReady flag to detect the next callback
			bDataReady = false;
		
			//read data into structure
			adi_sense_GetData( hDevice, eMeasurementMode, pSampleBuffer, nBytesPerSample, nSamplesPerDataready, &nReturned );
		
			//for all returned samples //TODO: Might need modified. Test with App
			for( uint32_t i = 0; i < nReturned; i++ ) {
				
				//to store adisense1000 channel index 
				//and the channel number according to the BLE device
				uint8_t channelIndex, activeChannelNumber;
				
				//obtain sample data in byte array form
				uint8_t *sample = ( uint8_t* )&pSampleBuffer[i].processedValue;
				
				//match sample ID to active channel
				for( uint32_t j = 0; j < activeChannelCount; j++ ) {
					
					//get the ID of the current active channel
					channelIndex = measurementInfo[j].associatedChannel;
					
					//if the sample matches the current active channel
					if( pSampleBuffer[i].channelId == channelIndex ) {
						activeChannelNumber = j;
						break;
					}
				}
				
				//clear buffPos
				buffPos = 0;
				
				//tag samples with command value
				bleTxBuffer[buffPos] = START_STREAM;
				buffPos++;
				
				//Tag with ID of measurement
				bleTxBuffer[buffPos] = (48 + measurementInfo[activeChannelNumber].measurementId);
				buffPos++;
					
				//transmute sample data into bytes for transmission
				for( uint32_t j = 0; j < sizeof( measurementInfo[activeChannelNumber].dataType ); j++ ) {
					
					//store bytes to buffer
					bleTxBuffer[buffPos] = (48 + sample[j]);
					buffPos++;
				}
				
				//stream measurement
				Bl652_Write( bleTxBuffer, buffPos );
			}
		
			if( bleSerialDevice.readable() ) {
				
				//signal ble to stop sending bytes
				ble_Cts = 1;
				
				//read byte buffer and determine if 
				//the STOP_STREAM byte has been received
				if( bleSerialDevice.getc() == STOP_STREAM ) {
					//stop stream measurement
					bleTxBuffer[0] = STOP_STREAM;
					Bl652_Write( bleTxBuffer, 1 );
					break;
				}
				else {
					return Ble_HandleError( ERROR_UNDEFINED );
				}
			}
		} while ( 1 );
	
		//stop measurement
		adi_sense_StopMeasurement( hDevice );
	}

	//free buffer
	delete [] pSampleBuffer;

	return 0;
}

/***************************************************************
* Function Name: Ble_HandleStartStream
* Description  : handle the START_STREAM command
****************************************************************/
static uint32_t Ble_HandleStartStream_v2( void )
{
	//buffer for sending BLE data
	char bleTxBuffer[BLE_PACKET_SIZE] = {0x00};
	//stores poition in bleTxBuffer
	uint8_t buffPos = 0;

	//part of the adisense1000 sampling
	ADI_SENSE_RESULT res;
	ADI_SENSE_DATA_SAMPLE *pSampleBuffer;
	ADI_SENSE_1000_OPERATING_MODE eOperatingMode;
	ADI_SENSE_1000_DATAREADY_MODE eDataPublishMode;
	ADI_SENSE_MEASUREMENT_MODE eMeasurementMode = ADI_SENSE_MEASUREMENT_MODE_NORMAL;
	uint32_t nSamplesPerDataready, nSamplesPerCycle;
	uint8_t nBytesPerSample;
	volatile bool_t bDataReady = false;
	
	//if ReqRegPacketCount has not been called, or had an error
	if( activeChannelCount == 0 ) {
		return Ble_HandleError( ERROR_UNDEFINED );
	}
	
	//upload new configuration and apply. 
	//This has been used to calculate all previous steps, so ensure that this is the active configuration
	adi_sense_SetConfig( hDevice, &smartcushion_adisense_config );
	
	unsigned lutBufferSize = ADI_SENSE_LUT_MAX_SIZE;
	ADI_SENSE_1000_LUT *pLutBuffer = (ADI_SENSE_1000_LUT *) ::operator new (lutBufferSize);
    if (pLutBuffer == NULL)
    {
        ADI_SENSE_LOG_ERROR("Failed to allocate memory for user-defined LUT data buffer");
        return ADI_SENSE_NO_MEM;
    }

    ADI_SENSE_LOG_INFO("Assembling LUT data");
    res = adi_sense_1000_AssembleLutData(pLutBuffer, lutBufferSize,
                                         sample_lut_num_tables,
                                         sample_lut_desc_list,
                                         sample_lut_data_list);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to assemble user-defined LUT data");
        return res;
    }
    
    /*
     * Write assembled user-defined Look-Up Table data structure to the device
     * User-defined LUT data is not applied until adi_sense_ApplyConfigUpdates() is called.
     */
    ADI_SENSE_LOG_INFO("Setting LUT data");
    res = adi_sense_1000_SetLutData(hDevice, pLutBuffer);
    if (res != ADI_SENSE_SUCCESS)
    {
        ADI_SENSE_LOG_ERROR("Failed to set user-defined LUT data");
        return res;
    }

    delete pLutBuffer;
	
	adi_sense_ApplyConfigUpdates( hDevice );
	ADI_SENSE_LOG_INFO("ApplyConfigUpdates");
	
	//get information about the adisense data ready mode
	adi_sense_1000_GetDataReadyModeInfo(hDevice,
							eMeasurementMode,
							&eOperatingMode,
							&eDataPublishMode,
							&nSamplesPerDataready,
							&nSamplesPerCycle,
							&nBytesPerSample );
	ADI_SENSE_LOG_INFO("GetDataReadyMode");
	//allocate a buffer to store the samples retreived on each DATAREADY pulse
	pSampleBuffer = new ADI_SENSE_DATA_SAMPLE[nSamplesPerDataready];
	if( pSampleBuffer == NULL ) {
		return 1;
	}
	
	//interrupt-triggered DATAREADY notifications in this mode
	adi_sense_RegisterGpioCallback( hDevice, ADI_SENSE_GPIO_PIN_DATAREADY,
						ble_data_ready_callback,
						( void * )&bDataReady );
	
	//allow ble to send data for stream cancel command
	ble_Cts = 0;
	
	//handle the single cycle device mode
	if ( eOperatingMode == ADI_SENSE_1000_OPERATING_MODE_SINGLECYCLE ) {
		
		//loop single cycle until stream stop command is received
		do {
			//begin measurement cycles
			adi_sense_StartMeasurement( hDevice, eMeasurementMode );
			
			//read in each measurement cycle
			for ( uint32_t i = 0; i < nSamplesPerCycle / nSamplesPerDataready; i++ ) {
				uint32_t nReturned;
				
				//Interrupt method: wait for DATAREADY interrupt callback
				while ( !bDataReady );
				
				//reset bDataReady flag to detect the next callback
				bDataReady = false;
				
				//read data into structure
				adi_sense_GetData( hDevice, eMeasurementMode, pSampleBuffer, nBytesPerSample, nSamplesPerDataready, &nReturned );
				
				//for all returned samples
				for( uint32_t i = 0; i < nReturned; i++ ) {
					
					//to store adisense1000 channel index 
					//and the channel number according to the BLE device
					uint8_t channelIndex, activeChannelNumber;
					
					//obtain sample data in byte array form
					uint8_t *sample = ( uint8_t* )&pSampleBuffer[i].processedValue;
					
					//match sample ID to active channel
					for( uint32_t j = 0; j < activeChannelCount; j++ ) {
						
						//get the ID of the current active channel
						channelIndex = measurementInfo[j].associatedChannel;
						
						//if the sample matches the current active channel
						if( pSampleBuffer[i].channelId == channelIndex ) {
							activeChannelNumber = j;
							break;
						}
					}
					
					//clear buffPos
					buffPos = 0;
					
					//tag samples with command value
					bleTxBuffer[buffPos] = START_STREAM;
					buffPos++;
					
					//Tag with ID of measurement
					bleTxBuffer[buffPos] = measurementInfo[activeChannelNumber].measurementId;
					buffPos++;
						
					//transmute sample data into bytes for transmission
					for( uint32_t j = 0; j < sizeof( measurementInfo[activeChannelNumber].dataType ); j++ ) {
						
						//store bytes to buffer
						bleTxBuffer[buffPos] = sample[j];
						buffPos++;
					}
					
					//stream measurement
					Bl652_Write( bleTxBuffer, buffPos );
				}
			}
			
			if( bleSerialDevice.readable() ) {
				ble_Cts = 1;
				if( bleSerialDevice.getc() == STOP_STREAM ) {
					//stop stream measurement
					bleTxBuffer[0] = STOP_STREAM;
					Bl652_Write( bleTxBuffer, 1 );
					break;
				}
				else {
					return Ble_HandleError( ERROR_UNDEFINED );
				}
			}			
		} while ( 1 );
	}

	//handle continuous measurement mode
	else {

		//begin measurement cycles
		adi_sense_StartMeasurement( hDevice, eMeasurementMode );
	
		do {
			uint32_t nReturned;
		
			//Interrupt method: wait for DATAREADY interrupt callback
			while ( !bDataReady );
		
			//reset bDataReady flag to detect the next callback
			bDataReady = false;
		
			//read data into structure
			adi_sense_GetData( hDevice, eMeasurementMode, pSampleBuffer, nBytesPerSample, nSamplesPerDataready, &nReturned );
		
			//for all returned samples //TODO: Might need modified. Test with App
			for( uint32_t i = 0; i < nReturned; i++ ) {
				
				Bl652_Printf("Sample # %2d Channel # %2d :: Raw %8d  :: Processed %.7f :: flags: %s %s",
                           i+1,
                           pSampleBuffer[i].channelId,
                           pSampleBuffer[i].rawValue,
                           pSampleBuffer[i].processedValue,
                           pSampleBuffer[i].status & ADI_SENSE_DEVICE_STATUS_ERROR ? "ERROR" : "",
                           pSampleBuffer[i].status & ADI_SENSE_DEVICE_STATUS_ALERT ? "ALERT" : "");
				
			}
		
			if( bleSerialDevice.readable() ) {
				
				//signal ble to stop sending bytes
				ble_Cts = 1;
				
				//read byte buffer and determine if 
				//the STOP_STREAM byte has been received
				if( bleSerialDevice.getc() == STOP_STREAM ) {
					//stop stream measurement
					bleTxBuffer[0] = STOP_STREAM;
					Bl652_Write( bleTxBuffer, 1 );
					break;
				}
				else {
					return Ble_HandleError( ERROR_UNDEFINED );
				}
			}
		} while ( 1 );
	
		//stop measurement
		adi_sense_StopMeasurement( hDevice );
	}

	//free buffer
	delete [] pSampleBuffer;

	return 0;
}

/***************************************************************
* Function Name: Ble_HandleStopStream
* Description  : handle the STOP_STREAM command
****************************************************************/
static uint32_t Ble_HandleStopStream( void )
{
	//buffer for sending BLE data
	char bleTxBuffer[BLE_PACKET_SIZE] = {0x00};
	//stores poition in bleTxBuffer
	uint8_t buffPos = 0;
		
	//stop stream measurement. Already stopped but let app know
	bleTxBuffer[buffPos] = STOP_STREAM;
	buffPos++;
	
	//transmit buffer
	Bl652_Write( bleTxBuffer, 1 );
	
	return 0;
}

/***************************************************************
* Function Name: handle_error
* Description  : handle an error
****************************************************************/
static uint32_t Ble_HandleError( ErrorType error )
{	
	char status[1] = {error};
	
	//send response error
	Bl652_Write( status, sizeof( status ) );

	return 1;
}

/***************************************************************
* Function Name: ble_parse_command
* Description  : parse the ble command
****************************************************************/
uint32_t Ble_ParseCommand(  char *bleRxBuffer  )
{
	uint32_t ret = 0;
	//here we read the next byte and use a case to decide the handler function
	//within the handler, the remaining bytes may be read and acted on appropriately with a response
	
	//confirm type of command ( 1st complete byte )
	switch( bleRxBuffer[0] ) {
		
	  case REQ_REG_PACKET_COUNT: {//app is requesting the amount of reg packets required
		//call handler function, passing buffer for further use
		ret = Ble_HandleReqRegPacketCount();
		char bleSend1[] = "PacketCount\r\n";
		Bl652_Write(bleSend1 , 15);
		ADI_SENSE_LOG_INFO("Result: ReqPacketCount");
		break;
		}
	  case REQ_REG_PACKETS: {//app is requesting for all reg packets
		//call handler function, passing buffer for further use
		ret = Ble_HandleReqRegPackets();
		char bleSend2[] = "ReqRegister\r\n";
		Bl652_Write(bleSend2 , 15);
		ADI_SENSE_LOG_INFO("Result: ReqRegister");
		break;
		}
	  case REQ_FIELD_NAMES: {//app is requesting field names
		//call handler function, passing buffer for further use
		ret = Ble_HandleReqFieldNames( bleRxBuffer );
		char bleSend3[] = "FieldNames\r\n";
		Bl652_Write(bleSend3 , 15);
		ADI_SENSE_LOG_INFO("Result: ReqFieldNames");
		break;
		}
	  case START_STREAM: {//app is requesting the start of the data stream
		//call handler function
		ret = Ble_HandleStartStream_v2();
		char bleSend4[] = "StartStream\r\n";
		Bl652_Write(bleSend4 , 15);
		ADI_SENSE_LOG_INFO("Result: StartStream");
		break;
		}
	  case STOP_STREAM: {//app is requesting to stop the data stream
		//call handler function
		ret = Ble_HandleStopStream();
		char bleSend5[] = "StopStream\r\n";
		Bl652_Write(bleSend5 , 15);
		ADI_SENSE_LOG_INFO("Result: StopStream");
		break;
		}
		//Template
		//case COMMAND:  return parse_COMMAND();
		//parse_COMMAND handles the action required
		//COMMAND must be defined within the header file

		//handle an unrecognised command/error
	  default: {//command not recognised
		ret = Ble_HandleError( ERROR_UNDEFINED );
		char bleSend6[] = "CmdUndifined\r\n";
		Bl652_Write(bleSend6 , 15);
		ADI_SENSE_LOG_INFO("Command not recognized");
		break;
		}
	}
	
	return ret;
}

