/*
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:
 * @brief:  Look-Up Table data-type definitions for ADSNS1000 API.
 *-----------------------------------------------------------------------------
 */

#ifndef __ADI_SENSE_1000_LUT_DATA_H__
#define __ADI_SENSE_1000_LUT_DATA_H__

#include "adi_sense_types.h"
#include "adi_sense_1000_sensor_types.h"

/*! @addtogroup ADSNS1000_Api
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! LUT data validation signature */
#define ADI_SENSE_LUT_SIGNATURE 0x4C555473

/*! LUT data CRC-16-CCITT seed value */
#define ADI_SENSE_LUT_CRC_SEED  0x4153

/*! LUT maximum allowed size  */
#define ADI_SENSE_LUT_MAX_SIZE 12288U

/*! Linearisation look-up table / co-efficient list geometry */
typedef enum {
    ADI_SENSE_1000_LUT_GEOMETRY_RESERVED = 0x00,
    /**< reserved - for internal use only */
	ADI_SENSE_1000_LUT_GEOMETRY_COEFFS = 0x01,
    /**< 1-dimensional equation coefficient list */
    ADI_SENSE_1000_LUT_GEOMETRY_NES_1D = 0x02,
    /**< 1-dimensional not-equally-spaced look-up table */
    ADI_SENSE_1000_LUT_GEOMETRY_NES_2D = 0x03,
    /**< 2-dimensional not-equally-spaced look-up table */
    ADI_SENSE_1000_LUT_GEOMETRY_ES_1D = 0x04,
    /**< 1-dimensional equally-spaced look-up table */
    ADI_SENSE_1000_LUT_GEOMETRY_ES_2D = 0x05,
    /**< 2-dimensional equally-spaced look-up table */
} ADI_SENSE_1000_LUT_GEOMETRY;

/*! Linearisation equation type */
typedef enum {
	ADI_SENSE_1000_LUT_EQUATION_POLYN,
    /**< Polynomial equation, typically used for Thermocouple and RTD
     * linearisation */
	ADI_SENSE_1000_LUT_EQUATION_POLYNEXP,
    /**< Polynomial + exponential equation, typically used for Thermocouple
     * inverse linearisation */
	ADI_SENSE_1000_LUT_EQUATION_QUADRATIC,
    /**< Quadratic linearisation equation, typically used for RTD
     * linearisation */
	ADI_SENSE_1000_LUT_EQUATION_STEINHART,
    /**< Steinhart-Hart equation, typically used for Thermistor
     * linearisation */
	ADI_SENSE_1000_LUT_EQUATION_LOGARITHMIC,
    /**< Beta-based logarithmic equation, typically used for Thermistor
     * linearisation */
	ADI_SENSE_1000_LUT_EQUATION_EXPONENTIAL,
    /**< Exponential equation */
	ADI_SENSE_1000_LUT_EQUATION_BIVARIATE_POLYN,
    /**< Bi-variate polynomial equation, typically used for bridge pressure
     * sensor linearisation
     * @note 2nd-degree is the maximum currently supported
     */
	ADI_SENSE_1000_LUT_EQUATION_COUNT,
    /**< Enum count value - for internal use only */
	ADI_SENSE_1000_LUT_EQUATION_LUT,
    /**< Hard-coded Look-Up Table - for internal use only */
} ADI_SENSE_1000_LUT_EQUATION;

typedef enum {
	ADI_SENSE_1000_LUT_TC_DIRECTION_FORWARD,
    /**< Thermocouple forward (mV to Celsius) linearisation
     *   Use this value by default for non-thermocouple sensors */
	ADI_SENSE_1000_LUT_TC_DIRECTION_BACKWARD,
    /**< Thermocouple inverse (Celsius to mV) linearisation */
	ADI_SENSE_1000_LUT_TC_DIRECTION_COUNT,
    /**< Enum count value - for internal use only */
} ADI_SENSE_1000_LUT_TC_DIRECTION;

/*! Linearisation data vector format */
typedef enum {
	ADI_SENSE_1000_LUT_DATA_TYPE_RESERVED = 0,
    /**< Reserved - for internal use only */
	ADI_SENSE_1000_LUT_DATA_TYPE_FLOAT32 = 1,
    /**< Single-precision 32-bit floating-point */
	ADI_SENSE_1000_LUT_DATA_TYPE_FLOAT64 = 2,
    /**< Double-precision 64-bit floating-point */
} ADI_SENSE_1000_LUT_DATA_TYPE;

/*! Struct for a list of coefficients to be used in an equation */
typedef struct __attribute__((packed, aligned(4))){
    uint32_t nCoeffs;
    /**< number of coefficients */
    float32_t rangeMin;
    /**< look-up table range - minimum */
    float32_t rangeMax;
    /**< look-up table range - maximum */
	float64_t coeffs[];
    /**< C99 flexible array: sorted by ascending exponent in polynomials */
} ADI_SENSE_1000_LUT_COEFF_LIST;

/*! Struct for a 1-dimensional equally-spaced look-up table */
typedef struct __attribute__((packed, aligned(4))){
    uint32_t nElements;
    /**< number of elements. */
	float32_t initInputValue;
    /**< initial input value, corresponding to first table element */
	float32_t inputValueIncrement;
    /**< interval between successive input values */
	float32_t lut[];
    /**< C99 flexible array */
} ADI_SENSE_1000_LUT_1D_ES;

/*! Struct for a 1-dimensional not-equally-spaced look-up table */
typedef struct __attribute__((packed, aligned(4))){
    uint32_t nElements;
    /**< number of elements of each array. */
    float32_t lut[];
    /**< C99 flexible array, first X's array then Y's array*/
} ADI_SENSE_1000_LUT_1D_NES;

/*! Struct for a 2-dimensional equally-spaced look-up table */
typedef struct __attribute__((packed, aligned(4))){
    uint16_t nElementsX;
    /**< number of elements for input X. */
    uint16_t nElementsY;
    /**< number of elements for input Y. */
    float32_t initInputValueX;
    /**< initial X input value */
    float32_t inputValueIncrementX;
    /**< interval between successive X input values */
    float32_t initInputValueY;
    /**< initial Y input value */
    float32_t inputValueIncrementY;
    /**< interval between successive Y input values */
    float32_t lut[];
    /**< C99 flexible array, Z matrix[y][x] */
} ADI_SENSE_1000_LUT_2D_ES;

/*! Struct for a 2-dimensional not-equally-spaced look-up table */
typedef struct __attribute__((packed, aligned(4))){
    uint16_t nElementsX;
    /**< number of elements in array X. */
    uint16_t nElementsY;
    /**< number of elements in array Y. */
    float32_t lut[];
    /**< C99 flexible array, Order: X's array, Y's array, Z matrix[y][x] */
} ADI_SENSE_1000_LUT_2D_NES;

/*! Struct for a 2-dimensional list of coefficients to be used in a
 * bi-variate polynomial equation */
typedef struct __attribute__((packed, aligned(4))){
    uint32_t  maxDegree;
    /**< number of coefficients */
    float32_t rangeMinX;
    /**< look-up table range - minimum X input value */
    float32_t rangeMaxX;
    /**< look-up table range - maximum X input value */
    float32_t rangeMinY;
    /**< look-up table range - minimum Y input value */
    float32_t rangeMaxY;
    /**< look-up table range - maximum Y input value */
    float64_t coeffs[];
    /**< C99 flexible array: sorted by ascending X degree then sorted by
     * ascending Y exponent */
} ADI_SENSE_1000_LUT_2D_POLYN_COEFF_LIST;

/*! Macro to calculate the number of elements in
 * a @ref ADI_SENSE_1000_LUT_COEFF_LIST table */
#define ADI_SENSE_1000_LUT_COEFF_LIST_NELEMENTS(_t)    \
    ((_t).nCoeffs)

/*! Macro to calculate the number of elements in
 * a @ref ADI_SENSE_1000_LUT_1D_ES table */
#define ADI_SENSE_1000_LUT_1D_ES_NELEMENTS(_t) \
    ((_t).nElements)

/*! Macro to calculate the number of elements in
 * a @ref ADI_SENSE_1000_LUT_1D_NES table */
#define ADI_SENSE_1000_LUT_1D_NES_NELEMENTS(_t) \
    ((_t).nElements * 2)

/*! Macro to calculate the number of elements in
 * a @ref ADI_SENSE_1000_LUT_2D_ES table */
#define ADI_SENSE_1000_LUT_2D_ES_NELEMENTS(_t)  \
    ((_t).nElementsX * (_t).nElementsX)

/*! Macro to calculate the number of elements in
 * a @ref ADI_SENSE_1000_LUT_2D_NES table */
#define ADI_SENSE_1000_LUT_2D_NES_NELEMENTS(_t)                         \
    ((_t).nElementsX + (_t).nElementsY + ((_t).nElementsX * (_t).nElementsY))

/*! Macro to calculate the number of elements in
 * a @ref ADI_SENSE_1000_LUT_2D_POLYN_COEFF_LIST table */
#define ADI_SENSE_1000_LUT_2D_POLYN_COEFF_LIST_NELEMENTS(_t)    \
    (((_t).maxDegree + 1) * ((_t).maxDegree + 2) / 2)

/*! Macro to calculate the storage size in bytes of
 * a @ref ADI_SENSE_1000_LUT_COEFF_LIST table */
#define ADI_SENSE_1000_LUT_COEFF_LIST_SIZE(_t)    \
    (sizeof(_t) + (sizeof(float64_t) * ADI_SENSE_1000_LUT_COEFF_LIST_NELEMENTS(_t)))

/*! Macro to calculate the storage size in bytes of
 * a @ref ADI_SENSE_1000_LUT_1D_ES table */
#define ADI_SENSE_1000_LUT_1D_ES_SIZE(_t)                               \
    (sizeof(_t) + (sizeof(float32_t) * ADI_SENSE_1000_LUT_1D_ES_NELEMENTS(_t)))

/*! Macro to calculate the storage size in bytes of
 * a @ref ADI_SENSE_1000_LUT_1D_NES table */
#define ADI_SENSE_1000_LUT_1D_NES_SIZE(_t)                              \
    (sizeof(_t) + (sizeof(float32_t) * ADI_SENSE_1000_LUT_1D_NES_NELEMENTS(_t)))

/*! Macro to calculate the storage size in bytes of
 * a @ref ADI_SENSE_1000_LUT_2D_ES table */
#define ADI_SENSE_1000_LUT_2D_ES_SIZE(_t)                               \
    (sizeof(_t) + (sizeof(float32_t) * ADI_SENSE_1000_LUT_2D_ES_NELEMENTS(_t)))

/*! Macro to calculate the storage size in bytes of
 * a @ref ADI_SENSE_1000_LUT_2D_NES table */
#define ADI_SENSE_1000_LUT_2D_NES_SIZE(_t)                              \
    (sizeof(_t) + (sizeof(float32_t) * ADI_SENSE_1000_LUT_2D_NES_NELEMENTS(_t)))

/*! Macro to calculate the storage size in bytes of
 * a @ref ADI_SENSE_1000_LUT_2D_POLYN_COEFF_LIST table */
#define ADI_SENSE_1000_LUT_2D_POLYN_COEFF_LIST_SIZE(_t)                 \
    (sizeof(_t) + (sizeof(float64_t) * ADI_SENSE_1000_LUT_2D_POLYN_COEFF_LIST_NELEMENTS(_t)))

/*! Look-Up Table descriptor */
typedef union __attribute__((packed, aligned(4))) {
    struct {
        ADI_SENSE_1000_LUT_GEOMETRY geometry  : 6;
        /**< Table geometry */
        ADI_SENSE_1000_LUT_EQUATION equation  : 6;
        /**< Equation type */
        ADI_SENSE_1000_LUT_TC_DIRECTION dir   : 4;
        /**< Thermocouple linearisation direction */
        ADI_SENSE_1000_ADC_SENSOR_TYPE sensor : 12;
        /**< Sensor Type ID */
        ADI_SENSE_1000_LUT_DATA_TYPE dataType : 4;
        /**< Table vector data type */
        uint16_t length;
        /**< Length in bytes of table data section
         *   (excluding this header) */
        uint16_t crc16;
        /**< CRC-16-CCITT of the data */
    };
    uint64_t value64;
} ADI_SENSE_1000_LUT_DESCRIPTOR;

/*! Look-Up Table geometry-specific data structures */
typedef union {
    ADI_SENSE_1000_LUT_COEFF_LIST          coeffList;
    /**< Data format for tables with ADI_SENSE_1000_LUT_GEOMETRY_COEFFS geometry
     *   except where equation is ADI_SENSE_1000_LUT_EQUATION_BIVARIATE_POLYN */
    ADI_SENSE_1000_LUT_1D_ES               lut1dEs;
    /**< Data format for tables with ADI_SENSE_1000_LUT_GEOMETRY_ES_1D geometry */
    ADI_SENSE_1000_LUT_1D_NES              lut1dNes;
    /**< Data format for tables with ADI_SENSE_1000_LUT_GEOMETRY_NES_1D geometry */
    ADI_SENSE_1000_LUT_2D_ES               lut2dEs;
    /**< Data format for tables with ADI_SENSE_1000_LUT_GEOMETRY_ES_2D geometry */
    ADI_SENSE_1000_LUT_2D_NES              lut2dNes;
    /**< Data format for tables with ADI_SENSE_1000_LUT_GEOMETRY_NES_2D geometry */
    ADI_SENSE_1000_LUT_2D_POLYN_COEFF_LIST coeffList2d;
    /**< Data format for tables with ADI_SENSE_1000_LUT_GEOMETRY_COEFFS geometry
     *   and ADI_SENSE_1000_LUT_EQUATION_BIVARIATE_POLYN equation */
} ADI_SENSE_1000_LUT_TABLE_DATA;

/*! Look-Up Table structure */
typedef struct __attribute__((packed, aligned(4))) {
    ADI_SENSE_1000_LUT_DESCRIPTOR descriptor;
    /**< Look-Up Table descriptor */
    ADI_SENSE_1000_LUT_TABLE_DATA data;
    /**< Look-Up Table data */
} ADI_SENSE_1000_LUT_TABLE;

/*! LUT data format versions */
typedef struct __attribute__((packed, aligned(4))) {
    uint8_t major; /*!< Major version number */
    uint8_t minor; /*!< Minor version number */
} ADI_SENSE_1000_LUT_VERSION;

/*! LUT data header structure */
typedef struct __attribute__((packed, aligned(4))) {
    uint32_t signature;
    /**< Hard-coded signature value (@ref ADI_SENSE_LUT_SIGNATURE) */
    ADI_SENSE_1000_LUT_VERSION version;
    /**< LUT data format version (@ref ADI_SENSE_LUT_VERSION) */
    uint16_t numTables;
    /**< Total number of tables */
    uint32_t totalLength;
    /**< Total length (in bytes) of all table descriptors and data
     *   (excluding this header)
     *   This, plus the header length, must not exceed ADI_SENSE_LUT_MAX_SIZE
     */
} ADI_SENSE_1000_LUT_HEADER;

/*! LUT data top-level structure */
typedef struct __attribute__((packed, aligned(4))) {
	ADI_SENSE_1000_LUT_HEADER header;
    /*!< LUT data top-level header structure */
    ADI_SENSE_1000_LUT_TABLE tables[];
        /*!< Variable-length array of one-or-more look-up table structures */
} ADI_SENSE_1000_LUT;

/*! Alternative top-level structure for raw LUT data representation
 *
 * @note This is intended to be used for encapsulating the storage of static
 *       LUT data declarations in C files.  The rawTableData can be cast
 *       to the ADI_SENSE_LUT type for further parsing/processing.
 */
typedef struct __attribute__((packed, aligned(4))) {
	ADI_SENSE_1000_LUT_HEADER header;
    /*!< LUT data top-level header structure */
    uint8_t rawTableData[];
    /*!< Variable-length byte array of look-up tables in raw binary format */
} ADI_SENSE_1000_LUT_RAW;

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADI_SENSE_1000_LUT_DATA_H__ */
