/*
 * bl652.cpp - BLE driver for serial port service
 *
 * Author: Jake Greaves
 */

#include "bl652.h"
#include <SerialBase.h>

#include "mbed.h"
#include "myproswift_periph.h"
#include "communications.h"


//used by callback
static char *cbSerialBuffer;
volatile bool *bBleMessageReceived;
static uint32_t cbCount = 0;
static uint32_t cbMaxSize;


//reset pin of BLE module
extern DigitalOut ble_rst;
//flow control
extern DigitalOut ble_Cts;
extern DigitalIn ble_Rts;
//Serial for ble
extern Serial bleSerialDevice;
//SPI
extern SPI bleSpiDevice;
extern DigitalOut cs_Ble;

extern DigitalOut bleMode0;
extern DigitalOut bleMode1;


//ble requires a minimum delay between packets
Timer timer;


/***************************************************************
* Function Name: Bl652_AtTransaction
* Description  : Perform an AT command transaction
***************************************************************/
uint32_t Bl652_AtTransaction( char *command, char *resp )
{
	//write command to device
	Bl652_Write( command, strlen(command) );
	
	//read response
	int8_t i = -1;
	do {
		i++;
		Bl652_Read( &resp[i], 1 );
	} while(resp[i] != AT_COMMAND_DELIMITER);
	
	if( !strncmp(resp, AT_COMMAND_ERROR, strlen(AT_COMMAND_ERROR)) ) {
		return 1;
	}
	   
	else {
		return 0;
	}
}

/***************************************************************
* Function Name: Bl652_Boot
* Description  : Main boot function
***************************************************************/
uint32_t Bl652_Boot( void )
{
	
	//setup device as vSP mode
	bleMode0 = 1;
	bleMode1 = 1;
	
	//reset device
	ble_rst = 0;
	wait_ms( RESET_LENGTH );
	ble_rst = 1;
	
	//set_flow_control(TYPE=RTSCTS, RTS pin, CTS pin)
	//bleSerialDevice.set_flow_control(Serial::RTSCTS, BLE_CTS_PIN, BLE_RTS_PIN );
	
	//start sending timer for minimum packet delay
	timer.start();
	
	return 0;
}

/***************************************************************
* Function Name: Bl652_Callback
* Description  : callback for uart communications
***************************************************************/
static void Bl652_Callback( void )
{	
	while( bleSerialDevice.readable() ) {
		//read next character
		char c = bleSerialDevice.getc();
		
		//store char in buffer
		if( cbSerialBuffer != NULL ) {
		    cbSerialBuffer[cbCount] = c;
		}
		
		//increment count
		cbCount++;
		
		//check if buffer is filled
		if( cbCount >= cbMaxSize ) {
			
		    //set flag that data has been received
		    *bBleMessageReceived = 1;
		    break;
		}
		//else continue sending
		else {    
			//indicate ble is clear to send
			ble_Cts = 0;
		}
	}
}

/***************************************************************
* Function Name: Bl652_WaitUntilAvailable
* Description  : wait until ble is awake
***************************************************************/
static uint32_t Bl652_WaitUntilAvailable( void )
{
    //wait until BLE is available
    while( ble_Rts );

    return 0;
}

/***************************************************************
* Function Name: Bl652_Write
* Description  : write string to ble over uart
***************************************************************/
uint32_t Bl652_Write( char *txBuffer, uint32_t byteCount )
{
	uint8_t res = 0;
	char padding[1] = {0x00};
	
	//check packet is not too large
	if( byteCount > BLE_PACKET_SIZE ) {
		return 1;
	}
	
	//wait the time limit for packets to be sent 
	while( timer.read_ms() < BLE_TX_PACKET_DELAY );
	timer.stop();
	
	for( uint32_t i = 0; i < byteCount; i++ ) {
		res = Bl652_WaitUntilAvailable();
		res = Uart_Write( bleSerialDevice, txBuffer + i, 1 );
	}
	
	//padding to 20 bytes
	for( uint32_t i = 0; i < BLE_PACKET_SIZE - byteCount; i++ ) {
		res = Bl652_WaitUntilAvailable();
		res = Uart_Write( bleSerialDevice, padding, 1 );
	}
	
	timer.reset();
	timer.start();
	
	return res;
}

/***************************************************************
* Function Name: ble_read
* Description  : blocking read over uart
***************************************************************/
uint32_t Bl652_Read( char *rxBuffer, uint32_t byteCount )
{
    ble_Cts = 0;
    
    for(uint32_t i = 0; i < byteCount; i++) {
	    rxBuffer[i] = bleSerialDevice.getc();
    }
	
    ble_Cts = 1;
    
    return 0;
}

/***************************************************************
* Function Name: ble_setup_read_cb
* Description  : register callback and flag for receiving ble data
***************************************************************/
uint32_t Bl652_SetupReadCb( void *rxBuffer, uint32_t maxSize, volatile bool *bMessageReadFlag )
{
	uint32_t ret = 0;
	
    //setup global vars for handler funtions
    cbCount = 0;
    cbMaxSize = maxSize;
    
    //assign serial callback buffer
    if( rxBuffer != NULL ) {
	  cbSerialBuffer = ( char* )rxBuffer;
    }
    else {
	    return 1;
    }
    
    //register global flag
    bBleMessageReceived = bMessageReadFlag;
    
    //assign callback buffer
    cbSerialBuffer = ( char* )rxBuffer;

    //call UART function
    ret = Uart_ReadCb( bleSerialDevice, Bl652_Callback );
  	
    //wait until BLE is awake
    //ble_wait_until_available();
    
    //indicate ble is clear to send
    ble_Cts = 0;
    
    return ret;
}

/***************************************************************
* Function Name: ble_clear_cb
* Description  : clear cb and top receiing ble data
***************************************************************/
uint32_t Bl652_ClearCb( void )
{
    //indicate ble to stop sending
    ble_Cts = 1;

    //clear callback to uart handle
    return Uart_ClearCb( bleSerialDevice );
}

uint32_t Bl652_Printf(const char *format, ...)
{
	uint32_t ret = 0;
	
	char buffer[256];
    va_list va_args;

    if (!bleSerialDevice){
        return 1;
	}
    va_start(va_args, format);
    vsnprintf(buffer, sizeof(buffer), format, va_args);
    va_end(va_args);

    bleSerialDevice.printf("%s\r\n", buffer);
    
    return ret;
}
