/*
 * pc_serial.cpp - serial communcations to pc as debug
 *                  future use for eval software
 *
 * Author: Jake Greaves
 */

#include "pc_interface/pc_serial.h"

//init for pc comms
extern Serial *gpUartDevice;

//used by callback handler
static char *cbSerialBuffer;
volatile static bool *bPcMessageReceived;
static uint32_t cbCount = 0;
static uint32_t cbMaxSize;

/***************************************************************
* Function Name: pc_uart_callback
* Description  : Callback for pc communications
****************************************************************/
static void Pc_Callback(void)
{
	while(gpUartDevice -> readable()) {
		//read char and effecively clear this callback flag
		char c = gpUartDevice -> getc();
		
		//check buffer is allocated
		if(cbSerialBuffer != NULL) {
			//store char to return buffer
			cbSerialBuffer[cbCount] = c;
				
			if(c == '\n') {
				//set flag that data has been received
				*bPcMessageReceived = 1;
				break;
			}
			else {
				//increment count, pos in buffer
				cbCount++;
			}
		}
	}
}

/***************************************************************
* Function Name: pc_setup_read_line_cb
* Description  : setup callback to handle pc comms
*		cb sets flag when full message is received
****************************************************************/
uint32_t Pc_SetupReadLineCb( char *rxBuffer, uint32_t maxSize, volatile bool *bMessageReadFlag ) 
{
  uint32_t ret = 0;
	
  //setup global vars for handler funtions
  cbCount = 0;
  cbMaxSize = maxSize;
  
  //assign serial callback buffer
  if( rxBuffer != NULL ) {
	cbSerialBuffer = rxBuffer;
  }
  else {
	  return 1;
  }
  
  //clear global flag
  bPcMessageReceived = bMessageReadFlag;
  
  //setup callback to uart handle
  ret = Uart_ReadCb( *gpUartDevice, Pc_Callback );
  
  return ret;
}

/***************************************************************
* Function Name: pc_setup_read_line_cb
* Description  : setup callback to handle pc comms
*		cb sets flag when full message is received
****************************************************************/
uint32_t Pc_ClearReadLineCb( void )
{
  //clear callback to uart handle
  return Uart_ClearCb( *gpUartDevice );
}

/***************************************************************
* Function Name: pc_uart_callback
* Description  : Callback for pc communications
****************************************************************/
uint32_t Pc_Read( char *rxBuffer, uint32_t maxSize )
{
    for(uint32_t i = 0; i < maxSize; i++)
	    rxBuffer[i] = gpUartDevice -> getc();
    
    return 0;
}

/***************************************************************
* Function Name: pc_write
* Description  : write formatted string to pc terminal
****************************************************************/
uint32_t Pc_Write( const char* txBuffer, uint32_t size )
{
    //call uart functions
    Uart_Write( *gpUartDevice, txBuffer, size );

    return 0;
}

/***************************************************************
* Function Name: pc_write_error
* Description  : write formatted string to pc terminal
*                   preceded by Error:
****************************************************************/
uint32_t Pc_WriteError( const char* txBuffer, ... )
{
	char buff[PC_BUFFER_SIZE];
	
	//precede string with error tag
	gpUartDevice -> printf( "Error: " );

	//format string to send
	va_list args;
	va_start( args, txBuffer );
	vsprintf( buff, txBuffer, args );

	//call uart functions
	Uart_Write( *gpUartDevice, buff, strlen( buff ) );

	//cleanup args
	va_end( args );

	return 0;
}

/***************************************************************
* Function Name: pc_write_debug
* Description  : write formatted string to pc terminal
                    preceded by Debug:
****************************************************************/
uint32_t Pc_WriteDebug(const char* txBuffer, ...)
{
	char buff[PC_BUFFER_SIZE];

	//precede string with debug tag
	gpUartDevice -> printf( "Debug: " );

	//format string to send
	va_list args;
	va_start( args, txBuffer );
	vsprintf( buff, txBuffer, args );

	//call uart functions
	Uart_Write( *gpUartDevice, buff, strlen( buff ) );
	gpUartDevice -> printf( "\n\r" );

	//cleanup args
	va_end(args);

	return 0;
}

//EOF


