/*
 ******************************************************************************
 * file:   main.cpp
 *-----------------------------------------------------------------------------
 *****************************************************************************/
#include "mbed.h"
#include "inc/adi_sense_api.h"
#include "inc/adi_sense_log.h"
#include "common/utils.h"

//additional libraries
#include "adisense1000_boot.h"
#include "ble_interface/ble_interface.h"
#include "myproswift_eval.h"
#include "eeprom_virtual/eeprom.h"
#include "myproswift_periph.h"
#include "led.h"
#include "bootloader.h"
#include "rcc_backup_registers.h"

#ifdef BL652
#include "ble_interface/bl652.h"
#endif

/** BLE COMMAND **
    REQ_REG_PACKET_COUNT    = 0x21, //'!'
    REQ_REG_PACKETS         = 0x40, //'@'
    REQ_FIELD_NAMES         = 0x23, //'#'
    START_STREAM            = 0x24, //'$'
    STOP_STREAM             = 0x25, //'%'
**/

int main( void )
{
    int ret = 1;
    bool bleActive;
    
    // Indicate device is booting
    Led_Boot();
     
    // Check if device needs to enter bootloader function, this is set via a PC command
    // Flag is stored in RTC registers that persist while vbat is powered
    bool_t bBootLoader = Rcc_ReadBackupReg( BOOTLOADER_FLAG_BACKUP_REG ) == 1 ? true : false;
    if( bBootLoader ) {
        
        // Let user know device is entering bootloader mode
        Pc_ResetSuccess( bBootLoader );
        
        // Jump to bootloader
        Bootloader_Init();
    }
    
    // Otherwise, boot main program
    
    // Boot the adisense1000 to a known state
    ret = Adisense1000_Boot();
    bool adiSense1000Active = !ret;
    
#ifdef BL652
    //boot BLE device
    ret = Bl652_Boot();
    if (ret == 0) {
        ADI_SENSE_LOG_INFO("BLE Boot success..");
    } else {
        ADI_SENSE_LOG_INFO("BLE Boot unsuccessful..");
    }    
#endif

    // Set if ble is present or not
    // If not, the PC interface can still be initialised
    bleActive = !ret;
    
    // Check adisense has booted correctly
    if( adiSense1000Active ) {
        Pc_ResetSuccess( bBootLoader );
        ADI_SENSE_LOG_INFO("ADIsense boot successful 03/09/2018");
        ADI_SENSE_LOG_INFO("Available BLE COMMAND");
        ADI_SENSE_LOG_INFO("REQ_REG_PACKET_COUNT = 0x21, //'!'");
        ADI_SENSE_LOG_INFO("REQ_REG_PACKETS      = 0x40, //'@'");
        ADI_SENSE_LOG_INFO("REQ_FIELD_NAMES      = 0x23, //'#'");
        ADI_SENSE_LOG_INFO("START_STREAM         = 0x24, //'$'");
        ADI_SENSE_LOG_INFO("STOP_STREAM          = 0x25, //'%'");
    }
    else {
        // Cannot continue without adi_sense module
        exit( 1 );
    }
    
    // Set led on to indicate boot has completed
    
    ADI_SENSE_LOG_INFO("Set LED Idle...");
    Led_Idle();
    
    // Begin main program
    ADI_SENSE_LOG_INFO("Begin main program...");
    while(1) {
        // Read in a command and handle appropriately
        ret = MyProSwift_Command( bleActive );
        if( ret != 0 ) {
            // Errors should be handled at lower levels, using reponse codes 
            // to the appropriate active interface
            // exit(1);
        }
    }
}

