/*
 * pc_interface.cpp - main interface to pc application
 *
 * Author: Jake Greaves
 */

#include "pc_interface/pc_interface.h"

#include "bootloader.h"

extern DigitalOut status_led;
	 
extern ADI_SENSE_CONFIG adi_sense_config;
extern ADI_SENSE_DEVICE_HANDLE hDevice;
extern bool_t bDeviceReady;
extern ADI_SENSE_PRODUCT_ID productId;

char respBuffer[RESPONSE_BUFFER_SIZE];
char seperatorBuf[] = {PC_SEPERATOR};
char delimBuf[1] = {PC_DELIMITER};


/***************************************************************
* Function Name: pc_reset_success
* Description  : notify that device is ready
****************************************************************/
int Pc_ResetSuccess(bool_t bBootLoader)
{
	char status[3];
	//char hello_pc[] = "hello pc!\r\n";
	
	if(bBootLoader) {
		//format error code into chars
		//return error code indicating that we are entering bootloader mode
		snprintf(status, sizeof(status), "%02X", NO_ERROR_BOOTLOADER);
	}
	
	else {
		//format error code into chars
		snprintf(status, sizeof(status), "%02X", NO_ERROR);
	}
	
	//print to serial
	Pc_Write(status, strlen(status));
	Pc_Write(delimBuf, sizeof(delimBuf));
	//Pc_Write(hello_pc, 11);
	//Pc_Write(hello_pc, strlen(hello_pc));
	
	return 0;
}

/***************************************************************
* Function Name: pc_read_next_arg
* Description  : read next arg, setting a flag if the delimiter is reached
****************************************************************/
static int pc_read_next_arg(char *command, char *arg, int size, bool *flag)
{
	int readPos = 0;
	char c;
	
	//clear flag
	*flag = 0;
	
	//read until break or buffer is used
	for(int i = 0; i < size + 1; i++) {
		
		//read single char from pc
  		c = command[0];
		//remove c from command
		memmove(command, command + 1, strlen(command  + 1));
		
		//if seperator is found, argument is in buffer, return
		if(c == PC_SEPERATOR) {
			arg[readPos] = '\0';
			return 0;
		}
		
		//if delimiter is found, set flag and return
		else if(c == PC_DELIMITER) {
			arg[readPos] = '\0';
			*flag = 1;
			return 0;
		}
		
		//else fill buffer with char
		else {
			if(i < size)
				arg[readPos] = c;
			readPos++;
		}
	}
	
	//buffer filled, return error
	return 1;
}

/***************************************************************
* Function Name: pc_parse_device_info
* Description  : get device info and return to pc
****************************************************************/
static int pc_parse_device_info(void) 
{
	//need to actually monitor this
	char status[3];
	char batteryVoltage[] = "3.3";
	
	//need to adjust hardware for this
	//batteryV.read() * VCC;
	
	//format error code into chars
	snprintf(status, sizeof(status), "%02X", NO_ERROR);
	
	//just a test, need to implement proper battery monitoring and firmware version control
	Pc_Write(status, strlen(status));
	Pc_Write(seperatorBuf, 1);
	Pc_Write(DEVICE_NAME, strlen(DEVICE_NAME));
	Pc_Write(seperatorBuf, 1);
	Pc_Write(batteryVoltage, strlen(batteryVoltage));
	Pc_Write(seperatorBuf, 1);
	Pc_Write(FIRMWARE_VERSION, strlen(FIRMWARE_VERSION));
	Pc_Write(delimBuf, 1);
	
	return 0;
}

/***************************************************************
* Function Name: pc_parse_flash_led
* Description  : flash led to identify device
****************************************************************/
static int pc_parse_flash_led(void) 
{
	char status[3];
	//format error code into chars
	snprintf(status, sizeof(status), "%02X", NO_ERROR);
	
	//flash led
	for(int i = 0; i < 10; i++) {
		status_led = !status_led;
		wait(0.05);
	}
	
	//return status ok
	Pc_Write(status, strlen(status));
	Pc_Write(delimBuf, sizeof(delimBuf));
	
	return 0;
}

/***************************************************************
* Function Name: parse_reset
* Description  : perform a soft reset of the device
****************************************************************/
static int pc_parse_reset(void)
{
	//call CMSIS reset function
	NVIC_SystemReset();
	
	//will never return as reset is called
	return 0;
}

/***************************************************************
* Function Name: pc_parse_configure
* Description  : parse the configure json command
****************************************************************/
static int pc_parse_configure(char *commandString)
{
	int ret = 0;
	bool delimiterFlag = 0;
	char channel[32];
	int channelIndex = (ADI_SENSE_1000_CHANNEL_ID)-2;
	char keyword[64];
	int keywordIndex = -1;
	
	//read in channel
	pc_read_next_arg(commandString, channel, sizeof(channel), &delimiterFlag);
	if(delimiterFlag) //if delimiter, action is complete
		return pc_handle_error(ERROR_EXPECTED_CHANNEL);
	
	//read in first keyword
	pc_read_next_arg(commandString, keyword, sizeof(keyword), &delimiterFlag);
	if(delimiterFlag)
		return pc_handle_error(ERROR_EXPECTED_KEYWORD);
	
	//find index of channel to configure
	for(int i = -1; i < ADI_SENSE_1000_MAX_CHANNELS; i++) {
		//cycle through conversion array, strncmp until found
		if(!strncmp(channel, channelIdVals[i+1].stringVal, strlen(channelIdVals[i+1].stringVal))) {
			channelIndex = i;
			break;
		}
	}

	//handle error, channel not found
	if(channelIndex < -1)
		return pc_handle_error(ERROR_EXPECTED_CHANNEL);

	for(int i = 0; i < ENUM_ATTRIBUTE_SIZE; i++) {
	//cycle through conversion array, strncmp until found
		if(!strncmp(keyword, keywordConvert[i].stringEquiv, strlen(keywordConvert[i].stringEquiv))) {
			keywordIndex = i;
			break;
		}
	}
	
	//handle error, channel not found
	if(keywordIndex < 0)
		return pc_handle_error(ERROR_EXPECTED_KEYWORD);
	
	//pass channel and keyword index to configure command which parses the value and applies it to the struct
	if(channelIndex > ADI_SENSE_1000_CHANNEL_ID_NONE)
		ret = pc_parse_configure_channel((ADI_SENSE_1000_CHANNEL_ID)channelIndex, (CONFIG_ATTRIBUTE)keywordIndex, commandString);
	else
		ret = pc_parse_configure_device((CONFIG_ATTRIBUTE)keywordIndex, commandString);
	
	
	if(ret != 0)
		return pc_handle_error((ErrorType)ret);
	
	char status[3];
	//format error code into chars
	snprintf(status, sizeof(status), "%02X", NO_ERROR);
	Pc_Write(status, strlen(status));
	
	//send response
	Pc_Write(delimBuf, sizeof(delimBuf));

	return ret;
}

/***************************************************************
* Function Name: parse_configure_device
* Description  : parse the configure_device json command
****************************************************************/
static int pc_parse_configure_device(CONFIG_ATTRIBUTE attribute, char *commandString)
{
	int valueIndex = -1;
	char value[64];
	bool delimiterFlag = 0;
	char *p;
	float32_t newRegValFloat32;
	
	//read in first value
	pc_read_next_arg(commandString, value, sizeof(value), &delimiterFlag);
	
	for(int i = 0; i < keywordConvert[attribute].valCount; i++) {
		if(!strncmp(value, keywordConvert[attribute].valsConvert[i].stringVal, 
			strlen(keywordConvert[attribute].valsConvert[i].stringVal))) {
			//value matched, store index and break
			valueIndex = i;
			break;
		}
	}
	
	switch(attribute) {
	//configure global struct to reflect new configuration
	//this struct can then be commited to write the ADISense1000 registers
	  case PRODUCT_ID:
		if(valueIndex == -1)
			return (ERROR_CONVERSION_FAILED);
		adi_sense_config.productId = (ADI_SENSE_PRODUCT_ID)keywordConvert[attribute].valsConvert[valueIndex].structVal;
		break;
	  case POWER_MODE:
		if(valueIndex == -1)
			return (ERROR_CONVERSION_FAILED);
		adi_sense_config.adisense1000.power.powerMode = (ADI_SENSE_1000_POWER_MODE)keywordConvert[attribute].valsConvert[valueIndex].structVal;
		break;
	  case OPERATIONAL_MODE:
		if(valueIndex == -1)
			return (ERROR_CONVERSION_FAILED);
		adi_sense_config.adisense1000.measurement.operatingMode = (ADI_SENSE_1000_OPERATING_MODE)keywordConvert[attribute].valsConvert[valueIndex].structVal;
		break;
	  case DATA_READY_MODE:
		if(valueIndex == -1)
			return (ERROR_CONVERSION_FAILED);
		adi_sense_config.adisense1000.measurement.dataReadyMode = (ADI_SENSE_1000_DATAREADY_MODE)keywordConvert[attribute].valsConvert[valueIndex].structVal;
		break;
	  case CYCLE_TIME:
		if(valueIndex == -1) {
			newRegValFloat32 = strtof(value, &p);
			if(*p != '\0')
				return (ERROR_EXPECTED_VALUE);
		}
		else {
			newRegValFloat32 = (float32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
		}
		adi_sense_config.adisense1000.measurement.cycleInterval = newRegValFloat32;
		break;
	  case GLOBAL_DIAGNOSTICS:
		if(valueIndex == -1)
			return (ERROR_CONVERSION_FAILED);
		adi_sense_config.adisense1000.diagnostics.disableGlobalDiag = (bool_t)keywordConvert[attribute].valsConvert[valueIndex].structVal;
		break;
	  case MEASUREMENT_SPECIFIC_DIAGNOSTICS:
		if(valueIndex == -1)
			return (ERROR_CONVERSION_FAILED);
		adi_sense_config.adisense1000.diagnostics.disableMeasurementDiag = (bool_t)keywordConvert[attribute].valsConvert[valueIndex].structVal;
		break;
	  case DIAGNOSTIC_MEASUREMENT_FREQUENCY:
		if(valueIndex == -1)
			return (ERROR_CONVERSION_FAILED);
		adi_sense_config.adisense1000.diagnostics.osdFrequency = (ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS)keywordConvert[attribute].valsConvert[valueIndex].structVal;
		break;
		
	  default:
		return (ERROR_CONVERSION_FAILED);
	}
	
	//expect command string to be fully used
	if(delimiterFlag)
		return 0;
	
	else
		return (ERROR_EXPECTED_DELIMITER);
}

/***************************************************************
* Function Name: parse_configure_channel
* Description  : parse the configure_channel command
****************************************************************/
static int pc_parse_configure_channel(ADI_SENSE_1000_CHANNEL_ID channelIndex, CONFIG_ATTRIBUTE attribute, char *commandString)
{
	int valueIndex = -1;
	char value[64];
	bool delimiterFlag = 0;
	char *p;
	uint32_t newRegValUint32;
	float32_t newRegValFloat32;
	uint32_t sensorTimeNewVal;
	uint32_t frontEndTimeNewVal;
	uint32_t extraSettlingTime;
	
	//read in first value
	pc_read_next_arg(commandString, value, sizeof(value), &delimiterFlag);
	
	//find index of value, where possible
	for(int i = 0; i < keywordConvert[attribute].valCount; i++) {
		if(!strncmp(value, keywordConvert[attribute].valsConvert[i].stringVal, 
				strlen(keywordConvert[attribute].valsConvert[i].stringVal))) {
			//value matched, store index and break
			valueIndex = i;
			break;
		}
	}
	
	ADI_SENSE_1000_CHANNEL_CONFIG *channel = &adi_sense_config.adisense1000.channels[channelIndex];
	
	//check for only analog attributes on analog channels
	if(channelIndex < ADI_SENSE_1000_CHANNEL_ID_I2C_0) {
	
		switch(attribute) {
		//configure global struct to reflect new configuration
		//this struct can then be commited to write the ADISense1000 registers
		  case MEASUREMENT_ENABLE:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> enableChannel = (bool_t)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case PUBLISH_MEASUREMENT:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> disablePublishing = (bool_t)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case ASSIGNED_COMPENSATION_MEASUREMENT_CHANNEL:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> compensationChannel = (ADI_SENSE_1000_CHANNEL_ID)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case DISPLAY_UNIT:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> measurementUnit = (ADI_SENSE_1000_MEASUREMENT_UNIT)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case MEASUREMENT_MIN_VALUE:
			if(valueIndex == -1) {
				newRegValFloat32 = strtof(value, &p);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
			}
			else {
				newRegValFloat32 = (float32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			channel -> lowThreshold = newRegValFloat32;
			break;
		  case MEASUREMENT_MAX_VALUE:
			if(valueIndex == -1) {
				newRegValFloat32 = strtof(value, &p);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
			}
			else {
				newRegValFloat32 = (float32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			channel -> highThreshold = newRegValFloat32;
			break;
		  case MEASUREMENTS_PER_CYCLE:
			if(valueIndex == -1) {
				newRegValUint32 = strtol(value, &p, 10);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
			}
			else {
				newRegValFloat32 = (uint32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			channel -> measurementsPerCycle = newRegValUint32;
			break;
		  case SETTLING_TIME:
			if(valueIndex == -1) {
				//check command string hasn't ended as we expect more parameters
				if(delimiterFlag)
					return (ERROR_EXPECTED_VALUE);
				//parse last parameter expected to calculate extra settling time
				char frontEndSettling[32];
				pc_read_next_arg(commandString, frontEndSettling, sizeof(frontEndSettling), &delimiterFlag);
				//extra settling time is equal to sensor settling - frontend. This is a check value for the ADISense1000
				sensorTimeNewVal = strtol(value, &p, 10);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
				frontEndTimeNewVal = strtol(frontEndSettling, &p, 10);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
				
				extraSettlingTime = sensorTimeNewVal - frontEndTimeNewVal;
			}
			else {
				extraSettlingTime = (uint32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			channel -> extraSettlingTime = extraSettlingTime;
			break;
		  case SENSOR_TYPE:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> adcChannelConfig.sensor = (ADI_SENSE_1000_ADC_SENSOR_TYPE)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case GAIN:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> adcChannelConfig.gain = (ADI_SENSE_1000_ADC_GAIN)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case EXCITATION_CURRENT:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> adcChannelConfig.current.outputLevel = (ADI_SENSE_1000_ADC_EXC_CURRENT)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case FILTER_TYPE:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> adcChannelConfig.filter.type = (ADI_SENSE_1000_ADC_FILTER_TYPE)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case FILTER_FS:
			if(valueIndex == -1) {
				newRegValUint32 = strtol(value, &p, 10);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
			}
			else {
				newRegValUint32 = (uint32_t)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			channel -> adcChannelConfig.filter.fs = newRegValUint32;
			break;
		  case REFERENCE:
			//check command string hasn't ended as we expected more parameters
			if(delimiterFlag)
				return (ERROR_EXPECTED_VALUE);
			
			//need to read in 4 args to parse this. ReferenceSelect (Value), ReferenceResistor, ReferenceResistorSelect, ReferenceVoltage.
			char referenceResistor[32];
			pc_read_next_arg(commandString, referenceResistor, sizeof(referenceResistor), &delimiterFlag);
			if(delimiterFlag)
				return (ERROR_EXPECTED_VALUE);
			
			char referenceResistorSelect[32];
			pc_read_next_arg(commandString, referenceResistorSelect, sizeof(referenceResistorSelect), &delimiterFlag);
			if(delimiterFlag)
				return (ERROR_EXPECTED_VALUE);
			
			char referenceVoltage[32];
			pc_read_next_arg(commandString, referenceVoltage, sizeof(referenceVoltage), &delimiterFlag);
			
			if(!strncmp(referenceResistorSelect, REF_RES_SELECT_INTERNAL, strlen(REF_RES_SELECT_INTERNAL))) {
				if(!strncmp(value, REF_SELECT_REFIN1, strlen(REF_SELECT_REFIN1))) {
					channel -> adcChannelConfig.reference.type = ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_INTERNAL_1;
				}
				else if(!strncmp(value, REF_SELECT_REFIN2, strlen(REF_SELECT_REFIN2))) {
					channel -> adcChannelConfig.reference.type = ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_INTERNAL_2;
				}
				else
					return (ERROR_CONVERSION_FAILED);
			}
			else if(!strncmp(referenceResistorSelect, REF_RES_SELECT_EXTERNAL, strlen(REF_RES_SELECT_EXTERNAL))) {
				if(!strncmp(value, REF_SELECT_REFIN1, strlen(REF_SELECT_REFIN1))) {
					channel -> adcChannelConfig.reference.type = ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_EXTERNAL_1;
					//strtol reference resistor to config. Not currently available
				}
				else if(!strncmp(value, REF_SELECT_REFIN2, strlen(REF_SELECT_REFIN2))) {
					channel -> adcChannelConfig.reference.type = ADI_SENSE_1000_ADC_REFERENCE_RESISTOR_EXTERNAL_2;
					//strtol reference resistor to config. Not currently available
				}
				else
					return (ERROR_CONVERSION_FAILED);
			}
			else if(!strncmp(referenceResistorSelect, REF_RES_SELECT_NA, strlen(REF_RES_SELECT_NA))) {
				if(!strncmp(value, REF_SELECT_REFIN1, strlen(REF_SELECT_REFIN1))) {
					channel -> adcChannelConfig.reference.type = ADI_SENSE_1000_ADC_REFERENCE_VOLTAGE_EXTERNAL_1;
					//strtol reference voltage to config. Not currently available
				}
				else if(!strncmp(value, REF_SELECT_REFIN2, strlen(REF_SELECT_REFIN2))) {
					channel -> adcChannelConfig.reference.type = ADI_SENSE_1000_ADC_REFERENCE_VOLTAGE_EXTERNAL_2;
					//strtol reference volatge to config. Not currently available
				}
				else if(!strncmp(value, REF_SELECT_INTERNAL, strlen(REF_SELECT_INTERNAL))) {
					channel -> adcChannelConfig.reference.type = ADI_SENSE_1000_ADC_REFERENCE_VOLTAGE_INTERNAL;
				}
				else
					channel -> adcChannelConfig.reference.type = ADI_SENSE_1000_ADC_REFERENCE_NONE;
			}
			else
				return (ERROR_CONVERSION_FAILED);
				
			break;
		  case VBIAS_ENABLE:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> adcChannelConfig.enableVbias = (bool_t)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
			
		  default:
			return (ERROR_ATTR_NOT_SUPPORTED_ON_CHANNEL);
		}
	}
	
	//check for only digital attributes on digital channels
	else if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0) {
		
		switch(attribute) {
		//configure global struct to reflect new configuration
		//this struct can then be commited to write the ADISense1000 registers
		  case MEASUREMENT_ENABLE:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> enableChannel = (bool_t)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case PUBLISH_MEASUREMENT:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> disablePublishing = (bool_t)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case ASSIGNED_COMPENSATION_MEASUREMENT_CHANNEL:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> compensationChannel = (ADI_SENSE_1000_CHANNEL_ID)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case DISPLAY_UNIT:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			channel -> measurementUnit = (ADI_SENSE_1000_MEASUREMENT_UNIT)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;
		  case MEASUREMENT_MIN_VALUE:
			if(valueIndex == -1) {
				newRegValFloat32 = strtof(value, &p);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
			}
			else {
				newRegValFloat32 = (float32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			channel -> lowThreshold = newRegValFloat32;
			break;
		  case MEASUREMENT_MAX_VALUE:
			if(valueIndex == -1) {
				newRegValFloat32 = strtof(value, &p);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
			}
			else {
				newRegValFloat32 = (float32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			channel -> highThreshold = newRegValFloat32;
			break;
		  case MEASUREMENTS_PER_CYCLE:
			if(valueIndex == -1) {
				newRegValUint32 = strtol(value, &p, 10);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
			}
			else {
				newRegValFloat32 = (uint32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			channel -> measurementsPerCycle = newRegValUint32;
			break;
		  case SETTLING_TIME:
			if(valueIndex == -1) {
				//check command string hasn't ended as we expect more parameters
				if(delimiterFlag)
					return (ERROR_EXPECTED_VALUE);
				//parse last parameter expected to calculate extra settling time
				char frontEndSettling[32];
				pc_read_next_arg(commandString, frontEndSettling, sizeof(frontEndSettling), &delimiterFlag);
				//extra settling time is equal to sensor settling - frontend. This is a check value for the ADISense1000
				sensorTimeNewVal = strtol(value, &p, 10);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
				frontEndTimeNewVal = strtol(frontEndSettling, &p, 10);
				if(*p != '\0')
					return (ERROR_EXPECTED_VALUE);
				
				extraSettlingTime = sensorTimeNewVal - frontEndTimeNewVal;
			}
			else {
				extraSettlingTime = (uint32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			channel -> extraSettlingTime = extraSettlingTime;
			break;
		  case SENSOR_TYPE:
			if(valueIndex == -1)
				return (ERROR_CONVERSION_FAILED);
			
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0)
				channel -> i2cChannelConfig.sensor = (ADI_SENSE_1000_I2C_SENSOR_TYPE)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			else if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_SPI_0 && channelIndex < ADI_SENSE_1000_MAX_CHANNELS)
				channel -> spiChannelConfig.sensor = (ADI_SENSE_1000_SPI_SENSOR_TYPE)keywordConvert[attribute].valsConvert[valueIndex].structVal;
			break;

		  case DEVICE_ADDRESS:
			if(valueIndex == -1) {
				if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
					newRegValUint32 = strtol(value, &p, 0);
					if(*p != '\0')
						return (ERROR_EXPECTED_VALUE);
					channel -> i2cChannelConfig.deviceAddress = newRegValUint32;
				}
				else
					return (ERROR_CONVERSION_FAILED);
			}
			else {
				channel -> i2cChannelConfig.deviceAddress = (uint32_t) keywordConvert[attribute].valsConvert[valueIndex].structVal;
			}
			break;
		
			//NOT NEEDED FOR FIRST SAMPLING
		  /*case READ_COMMAND:
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
				commandBuf = strtol(value, NULL, 10);
				memcpy(, channel -> i2cChannelConfig.dataRequestCommand.command);
				channel -> i2cChannelConfig.dataRequestCommand.commandLength = ;
			}
			break;
		  case CONFIG_COMMAND:
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
				channel -> i2cChannelConfig.configurationCommand.command = ;
				channel -> i2cChannelConfig.configurationCommand.commandLength = ;
			}
			break;
		  case NUMBER_OF_BITS:
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
				channel -> i2cChannelConfig.dataFormat.numDataBits = ;
			}
			break;
		  case FRAME_WIDTH:
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
				channel -> i2cChannelConfig.dataFormat.frameLength = ;
			}
			break;
		  case LEFT_ALIGNMENT:
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
				channel -> i2cChannelConfig.dataFormat.leftJustified = ;
			}
			break;
		  case DATA_ENDIANNESS:
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
				channel -> i2cChannelConfig.dataFormat.littleEndian = ;
			}
			break;
		  case OFFSET:
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
				channel -> i2cChannelConfig.dataFormat.bitOffset = ;
			}
			break;
		  case CODING:
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
				channel -> i2cChannelConfig.dataFormat.coding = ;
			}
			break;
		  case CPOL_CPHA:
			if(channelIndex >= ADI_SENSE_1000_CHANNEL_ID_I2C_0 && channelIndex < ADI_SENSE_1000_CHANNEL_ID_SPI_0) {
				channel -> i2cChannelConfig.dataFormat = ;
			}
			break;
			*/
		  default:
			return (ERROR_ATTR_NOT_SUPPORTED_ON_CHANNEL);
		}
	}
	
	//expect that the whole command string has been used
	if(delimiterFlag)
		return 0;
	
	else
		return (ERROR_EXPECTED_DELIMITER);
}

/***************************************************************
* Function Name: pc_parse_apply
* Description  : apply the config to the device
****************************************************************/
static int pc_parse_apply(void)
{
	//upload new configuration and apply
	if(adi_sense_SetConfig(hDevice, &adi_sense_config) != ADI_SENSE_SUCCESS)
		return pc_handle_error(ERROR_SET_CONFIG);
	if(adi_sense_ApplyConfigUpdates(hDevice) != ADI_SENSE_SUCCESS)
		return pc_handle_error(ERROR_UPDATE_CONFIG);
	
#ifdef VIRT_EEPROM
	//Unlock the Flash Program Erase controller
	HAL_FLASH_Unlock();
	
	//write adi_sense_config to eeprom
	uint16_t *p = (uint16_t*)&adi_sense_config;
	uint16_t temp;
	
	for(int i = 0; i < sizeof(adi_sense_config)/sizeof(uint16_t); i++, p++) {
		
		if((EE_WriteVariable(VirtAddVarTab[1] + i, *p)) != HAL_OK)
		{
			exit(1);
		}
		if((EE_WriteVariable(VirtAddVarTab[1] + i, *p)) != HAL_OK)
		{
			exit(1);
		}
		if((EE_ReadVariable(VirtAddVarTab[1] + i, &temp)) != HAL_OK)
		{
			exit(1);
		}
		if (temp != *p)
		{
			exit(1);
		}
	}
	
	//flag for valid configuration
	uint16_t eepromConfigValid = 0xAF;
	
	//write configuration valid flag
	if((EE_WriteVariable(VirtAddVarTab[0], eepromConfigValid)) != HAL_OK)
	{
		exit(1);
	}
	
	//Lock the Flash Program Erase controller
	HAL_FLASH_Lock();
#endif
	
	char status[3];
	//format error code into chars
	snprintf(status, sizeof(status), "%02X", NO_ERROR);
	Pc_Write(status, strlen(status));
	
	//send response
	Pc_Write(delimBuf, sizeof(delimBuf));

	return 0;
}

/***************************************************************
* Function Name: data_ready_callback
* Description  : callback for when data ready state changes
****************************************************************/
static void pc_data_ready_callback(
    	ADI_SENSE_GPIO_PIN ePinId,
    	void *pArg)
{
    	volatile bool_t *pbFlag = (volatile bool_t *) pArg;

    	*pbFlag = true;
}

/***************************************************************
* Function Name: pc_register_callbacks
* Description  : callback for when data ready state changes
****************************************************************/
static int pc_register_callbacks(ADI_SENSE_DEVICE_HANDLE hDevice,
    volatile bool_t *pbDataReady,
    volatile bool_t *pbError,
    volatile bool_t *pbAlert)
{
    	int res;

	res = adi_sense_RegisterGpioCallback(hDevice, ADI_SENSE_GPIO_PIN_DATAREADY,
	                                     pc_data_ready_callback, (void *)pbDataReady);
	if (res != ADI_SENSE_SUCCESS)
	{
	    return res;
	}
	
	res = adi_sense_RegisterGpioCallback(hDevice, ADI_SENSE_GPIO_PIN_ERROR,
	                                     pc_data_ready_callback, (void *)pbError);
	if (res != ADI_SENSE_SUCCESS)
	{
	    return res;
	}
	
	res = adi_sense_RegisterGpioCallback(hDevice, ADI_SENSE_GPIO_PIN_ALERT,
	                                     pc_data_ready_callback, (void *)pbAlert);
	if (res != ADI_SENSE_SUCCESS)
	{
	    return res;
	}
	
	return ADI_SENSE_SUCCESS;
}

/***************************************************************
* Function Name: pc_deregister_callbacks
* Description  : callback for when data ready state changes
****************************************************************/
ADI_SENSE_RESULT pc_deregister_callbacks(
    ADI_SENSE_DEVICE_HANDLE hDevice)
{
	ADI_SENSE_RESULT res;
	
	res = adi_sense_RegisterGpioCallback(hDevice, ADI_SENSE_GPIO_PIN_DATAREADY,
	                                     NULL, NULL);
	if (res != ADI_SENSE_SUCCESS)
	{
	    return res;
	}
	
	res = adi_sense_RegisterGpioCallback(hDevice, ADI_SENSE_GPIO_PIN_ERROR,
	                                     NULL, NULL);
	if (res != ADI_SENSE_SUCCESS)
	{
	    return res;
	}
	
	res = adi_sense_RegisterGpioCallback(hDevice, ADI_SENSE_GPIO_PIN_ALERT,
	                                     NULL, NULL);
	if (res != ADI_SENSE_SUCCESS)
	{
	    return res;
	}
	
	return ADI_SENSE_SUCCESS;
}

/***************************************************************
* Function Name: pc_parse_sample
* Description  : parse the sample command
****************************************************************/
static int pc_parse_sample(char *commandString)
{
	ADI_SENSE_STATUS status;
	ADI_SENSE_DATA_SAMPLE *pSampleBuffer;
	ADI_SENSE_1000_OPERATING_MODE eOperatingMode;
	ADI_SENSE_1000_DATAREADY_MODE eDataPublishMode;
	uint32_t nSamplesPerDataready;
	uint32_t nSamplesPerCycle;
	uint32_t maxMeasurementCycles;
	ADI_SENSE_MEASUREMENT_MODE eMeasurementMode = ADI_SENSE_MEASUREMENT_MODE_NORMAL;
	bool firstSample = true; //flag for if this is the first round of sampling, include status code
	volatile bool_t bDataReady = false;
	volatile bool_t bError = false;
	volatile bool_t bAlert = false;

	char *p; //check strtol has parsed correctly

	char maxMeasurementCyclesString[20] = "";
	bool delimiterFlag = 0;
	
	//register callbacks for status pins
	if (pc_register_callbacks(hDevice, &bDataReady, &bError, &bAlert) != ADI_SENSE_SUCCESS)
		return pc_handle_error(ERROR_GPIO_CALLBACK_REG);

	pc_read_next_arg(commandString, maxMeasurementCyclesString, sizeof(maxMeasurementCyclesString), &delimiterFlag);
	if(!strcmp(maxMeasurementCyclesString, "") || !strncmp(maxMeasurementCyclesString, "0", strlen("0")))
		return pc_handle_error(ERROR_EXPECTED_VALUE);
	if(!delimiterFlag)
		return pc_handle_error(ERROR_EXPECTED_DELIMITER);

	//convert string to integer
	maxMeasurementCycles = strtol(maxMeasurementCyclesString, &p, 10);
	if(*p != '\0')
		return pc_handle_error(ERROR_EXPECTED_VALUE);
	
	if(maxMeasurementCycles < 1)
		return pc_handle_error(ERROR_EXPECTED_VALUE);

	//Allocate a buffer to store the samples retreived on each DATAREADY pulse
	if(adi_sense_1000_GetDataReadyModeInfo(hDevice,
								eMeasurementMode,
								&eOperatingMode,
								&eDataPublishMode,
								&nSamplesPerDataready,
								&nSamplesPerCycle) != ADI_SENSE_SUCCESS)
		return pc_handle_error(ERROR_GET_DRDY_INFO);

	//allocate a buffer for samples using the ADISense1000 API
	pSampleBuffer = new ADI_SENSE_DATA_SAMPLE[nSamplesPerDataready];
	if(pSampleBuffer == NULL)
		return pc_handle_error(ERROR_ALLOCATE_BUFFER);

	char seperatorBuf[] = {PC_SEPERATOR};

	//handle the single cycle device mode
	if (eOperatingMode == ADI_SENSE_1000_OPERATING_MODE_SINGLECYCLE) {
		uint32_t nTotalCycles = 0;
		
		do {
			//begin measurement cycles
			if(adi_sense_StartMeasurement(hDevice, eMeasurementMode) != ADI_SENSE_SUCCESS)
				return pc_handle_error(ERROR_START_MEASUREMENT);
		
			//read in each measurement cycle
			for (unsigned i = 0; i < nSamplesPerCycle / nSamplesPerDataready; i++) {
				uint32_t nReturned;
			
				//Interrupt method: wait for DATAREADY interrupt callback
				while(!(bDataReady || bError));
				
				if(!bError) {
					//reset bDataReady flag to detect the next callback
					bDataReady = false;
				
					//read data into structure
					if(adi_sense_GetData(hDevice, eMeasurementMode, pSampleBuffer, nSamplesPerDataready, &nReturned) != ADI_SENSE_SUCCESS)
						return pc_handle_error(ERROR_GET_DATA);
					if(nReturned != nSamplesPerDataready)
						return pc_handle_error(ERROR_SAMPLE_COUNT_MISMATCH);
				
					if(firstSample) {
						firstSample = false;
						//begin with status byte
						char status[3];
						//format error code into chars
						snprintf(status, sizeof(status), "%02X", NO_ERROR);
						//send response
						Pc_Write(status, sizeof(status) - 1);
					}
					
					//build a response string
					for(int i = 0; i < nReturned; i++) {
						float32_t *sample = &pSampleBuffer[i].processedValue;
						memcpy(respBuffer, (uint8_t*)sample, sizeof(float32_t));
						Pc_Write(seperatorBuf, 1);
						
						for(int j = 0; j < sizeof(float32_t); j++) {
							char buf[3];
							snprintf(buf, sizeof(buf), "%02X", respBuffer[j]);
							Pc_Write(buf, 2);
						}
					}
				}
				
				else {
					break;
				}
		
			}
			nTotalCycles++;
			
		} while (!bError && (nTotalCycles < maxMeasurementCycles));
	}

	//handle continuous measurement mode
	else {
		uint32_t nTotalSamples = 0;
	
		//begin measurement cycles
		if(adi_sense_StartMeasurement(hDevice, eMeasurementMode) != ADI_SENSE_SUCCESS)
			return pc_handle_error(ERROR_START_MEASUREMENT);
	
		do {
			uint32_t nReturned;
		
			//Interrupt method: wait for DATAREADY interrupt callback
			while(!(bDataReady || bError));
			
			if(!bError) {
				//reset bDataReady flag to detect the next callback
				bDataReady = false;
			
				//read data into structure
				if(adi_sense_GetData(hDevice, eMeasurementMode, pSampleBuffer, nSamplesPerDataready, &nReturned) != ADI_SENSE_SUCCESS)
					return pc_handle_error(ERROR_GET_DATA);
				if(nReturned != nSamplesPerDataready)
					return pc_handle_error(ERROR_SAMPLE_COUNT_MISMATCH);
			
				if(firstSample) {
					firstSample = false;
					//begin with status byte
					char status[3];
					//format error code into chars
					snprintf(status, sizeof(status), "%02X", NO_ERROR);
					//send response
					Pc_Write(status, sizeof(status) - 1);
				}
				
				//build a response string
				for(int i = 0; i < nReturned; i++) {
					float32_t *sample = &pSampleBuffer[i].processedValue;
					memcpy(respBuffer, (uint8_t*)sample, sizeof(float32_t));
					Pc_Write(seperatorBuf, 1);
					for(int j = 0; j < sizeof(float32_t); j++) {
						char buf[3];
						snprintf(buf, sizeof(buf), "%02X", respBuffer[j]);
						Pc_Write(buf, 2);
					}
				}
				
				//count samples
				nTotalSamples += nReturned;
			}
		
		} while (!bError && ((nTotalSamples / nSamplesPerCycle) < maxMeasurementCycles));
	
		if(adi_sense_StopMeasurement(hDevice) != ADI_SENSE_SUCCESS)
			return pc_handle_error(ERROR_STOP_MEASUREMENT);
	}
	
	//currently only error case is handled
	if (bError || bAlert)
	{
		if(adi_sense_GetStatus(hDevice, &status) != ADI_SENSE_SUCCESS)
			return pc_handle_error(ERROR_GET_DATA);

		if (status.deviceStatus &
		(ADI_SENSE_DEVICE_STATUS_ERROR | ADI_SENSE_DEVICE_STATUS_ALERT))
		{
			if (bError)
		    		return pc_handle_error(ERROR_GET_DATA);
		}
	}

	delete [] pSampleBuffer;
	
	if(pc_deregister_callbacks(hDevice) != ADI_SENSE_SUCCESS)
	    return pc_handle_error(ERROR_GPIO_CALLBACK_REG);

	Pc_Write(delimBuf, sizeof(delimBuf));

	return 0;
	
	/*ADI_SENSE_RESULT res;
	
	volatile bool_t bDataReady = false;
	volatile bool_t bError = false;
	volatile bool_t bAlert = false;
	res = utils_registerCallbacks(hDevice, &bDataReady, &bError, &bAlert);
	if (res != ADI_SENSE_SUCCESS)
	    return res;
	
	
	//Retrieve the number of samples per cycle, per DATAREADY pulse, etc. for this configuration.
	
	ADI_SENSE_1000_OPERATING_MODE eOperatingMode;
	ADI_SENSE_1000_DATAREADY_MODE eDataReadyMode;
	uint32_t nSamplesPerDataready;
	uint32_t nSamplesPerCycle;
	res = adi_sense_1000_GetDataReadyModeInfo(hDevice,
	                                          eMeasurementMode,
	                                          &eOperatingMode,
	                                          &eDataReadyMode,
	                                          &nSamplesPerDataready,
	                                          &nSamplesPerCycle);
	if (res != ADI_SENSE_SUCCESS)
	    return res;

//	 Allocate a buffer to store the samples retrieved on each DATAREADY pulse
//	 However, if the DATAREADY pulse is per-conversion, allocate a bigger buffer
//	 to accumulate a full cycle of samples before printing them
//	
	ADI_SENSE_DATA_SAMPLE *pSampleBuffer;
	if (eDataReadyMode == ADI_SENSE_1000_DATAREADY_PER_CONVERSION)
	    pSampleBuffer = malloc(sizeof(ADI_SENSE_DATA_SAMPLE) * nSamplesPerCycle);
	else
	    pSampleBuffer = malloc(sizeof(ADI_SENSE_DATA_SAMPLE) * nSamplesPerDataready);
	if (pSampleBuffer == NULL)
	{
	    ADI_SENSE_LOG_ERROR("Failed to allocate sample buffer");
	    return ADI_SENSE_NO_MEM;
	}
	
//	
//	Kick off the measurement cycle(s) here
//	
	res = adi_sense_StartMeasurement(hDevice, eMeasurementMode);
	if (res != ADI_SENSE_SUCCESS)
	{
	    ADI_SENSE_LOG_ERROR("Failed to start measurement");
	    return res;
	}
	
//	
//	Loop continuously unless operating mode is single-cycle
//	
	do {
	    ADI_SENSE_STATUS status;
	    uint32_t nCurrentSamples;
	    uint32_t nReturned;
	    nCurrentSamples = 0;
	
//	    
//	    Accumulate the samples from a cycle and print them
//	    NOTE: requires a sufficient idle time between cycles to allow printing to occur
//	    
	    do {
//	        
//	        Wait for the cycle to complete, continuously checking DATAREADY until it is asserted
//	        
	        while (! (bDataReady || bError))
	            ;
	
	        if (! bError)
	        {
//	            
//	            Retrieve the data samples from the measurement cycle, if no error has occurred
//	            
	            bDataReady = false;
	            res = adi_sense_GetData(hDevice, eMeasurementMode, &pSampleBuffer[nCurrentSamples], nSamplesPerDataready, &nReturned);
	            nCurrentSamples += nReturned;
	            if (res != ADI_SENSE_SUCCESS)
	            {
	                if (res == ADI_SENSE_INCOMPLETE)
	                {
//	                     For this case, let's get the device status and print
//	                     any samples we did get
	                    ADI_SENSE_LOG_WARN("Failed to retrieve all requested data samples");
	                    break;
	                }
	                else
	                {
	                    ADI_SENSE_LOG_WARN("Failed to retrieve data samples from device");
	                    return res;
	                }
	            }
	        }
	    } while (!bError && (nCurrentSamples < nSamplesPerCycle));
	
//	    
//	    Display the data samples
//	    
	    utils_printSamples(pSampleBuffer, nCurrentSamples);
	
//	    
//	    Check and print device status if errors/alerts have been triggered
//	    
	    if (bError || bAlert)
	    {
	        res = adi_sense_GetStatus(hDevice, &status);
	        if (res != ADI_SENSE_SUCCESS)
	        {
	            ADI_SENSE_LOG_ERROR("Failed to retrieve device status");
	            return res;
	        }
	
	        if (status.deviceStatus &
	            (ADI_SENSE_DEVICE_STATUS_ERROR | ADI_SENSE_DEVICE_STATUS_ALERT))
	        {
	            utils_printStatus(&status);
	
//	             Break out of the loop if any errors are raised
	            if (bError)
	                break;
	        }
	    }
	} while (eOperatingMode != ADI_SENSE_1000_OPERATING_MODE_SINGLECYCLE);
	
	res = adi_sense_StopMeasurement(hDevice);
	if (res != ADI_SENSE_SUCCESS)
	{
	    ADI_SENSE_LOG_ERROR("Failed to send stop measurement");
	    return res;
	}
	
	free(pSampleBuffer);
	
	res = utils_deregisterCallbacks(hDevice);
	if (res != ADI_SENSE_SUCCESS)
	    return res;
	
	return ADI_SENSE_SUCCESS;*/
}

/***************************************************************
* Function Name: pc_parse_start_stream
* Description  : parse the stream command
****************************************************************/
static int pc_parse_start_stream(void) 
{
	ADI_SENSE_DATA_SAMPLE *pSampleBuffer;
	ADI_SENSE_1000_OPERATING_MODE eOperatingMode;
	ADI_SENSE_1000_DATAREADY_MODE eDataPublishMode;
	uint32_t nSamplesPerDataready;
	uint32_t nSamplesPerCycle;
	ADI_SENSE_MEASUREMENT_MODE eMeasurementMode = ADI_SENSE_MEASUREMENT_MODE_NORMAL;
	
	char status[3];
	
	volatile bool_t bDataReady = false;
	
	//upload new configuration and apply
	if(adi_sense_SetConfig(hDevice, &adi_sense_config) != ADI_SENSE_SUCCESS)
	    return pc_handle_error(ERROR_SET_CONFIG);

	if(adi_sense_ApplyConfigUpdates(hDevice) != ADI_SENSE_SUCCESS)
	    return pc_handle_error(ERROR_UPDATE_CONFIG);

	//Allocate a buffer to store the samples retreived on each DATAREADY pulse
	if(adi_sense_1000_GetDataReadyModeInfo(hDevice,
							  eMeasurementMode,
							  &eOperatingMode,
							  &eDataPublishMode,
							  &nSamplesPerDataready,
							  &nSamplesPerCycle) != ADI_SENSE_SUCCESS)
	    return pc_handle_error(ERROR_GET_DRDY_INFO);
	
    	//allocate a buffer for samples using the ADISense1000 API
    	pSampleBuffer = new ADI_SENSE_DATA_SAMPLE[nSamplesPerDataready];
    	if(pSampleBuffer == NULL)
		    return pc_handle_error(ERROR_ALLOCATE_BUFFER);
	
	//begin with status byte
	//format error code into chars
	snprintf(status, sizeof(status), "%02X", NO_ERROR);
	//send response
	Pc_Write(status, sizeof(status) - 1);
	
	//setup callback routine for pc
  	bool bPcMessageFlag = 0;
	char msgBuffer[1];
  	Pc_SetupReadLineCb(msgBuffer, 1, &bPcMessageFlag);
	  
    	//handle the single cycle device mode
    	if (eOperatingMode == ADI_SENSE_1000_OPERATING_MODE_SINGLECYCLE) {
        uint32_t nTotalCycles = 0;

      //interrupt-triggered DATAREADY notifications in this mode
      if(adi_sense_RegisterGpioCallback(hDevice, ADI_SENSE_GPIO_PIN_DATAREADY,
                                              pc_data_ready_callback,
                                              (void *)&bDataReady) != ADI_SENSE_SUCCESS)
		  return pc_handle_error(ERROR_GPIO_CALLBACK_REG);
	  
		do {
			//begin measurement cycles
			if(adi_sense_StartMeasurement(hDevice, eMeasurementMode) != ADI_SENSE_SUCCESS)
				return pc_handle_error(ERROR_START_MEASUREMENT);

			//read in each measurement cycle
			for (unsigned i = 0; i < nSamplesPerCycle / nSamplesPerDataready; i++) {
				uint32_t nReturned;

				//Interrupt method: wait for DATAREADY interrupt callback
				while (!bDataReady);

				//reset bDataReady flag to detect the next callback
				bDataReady = false;

				//read data into structure
				if(adi_sense_GetData(hDevice, eMeasurementMode, pSampleBuffer, nSamplesPerDataready, &nReturned) != ADI_SENSE_SUCCESS)
				    return pc_handle_error(ERROR_GET_DATA);
				if(nReturned != nSamplesPerDataready)
				    return pc_handle_error(ERROR_SAMPLE_COUNT_MISMATCH);

			    	//build a response string
				for(int i = 0; i < nReturned; i++) {
				    float32_t *sample = &pSampleBuffer[i].processedValue;
				    memcpy(respBuffer, (uint8_t*)sample, sizeof(float32_t));
				    Pc_Write(seperatorBuf, 1);
				    for(int j = 0; j < sizeof(float32_t); j++) {
					    char buf[3];
					    snprintf(buf, sizeof(buf), "%02X", respBuffer[j]);
					    Pc_Write(buf, 2);
				    }
				}
			    
			}
			nTotalCycles++;

			//check read byte to see if streaming should be stopped
			if(bPcMessageFlag) {
				if(msgBuffer[0] == COMMAND_STOP_STREAM) {
					Pc_Write(delimBuf, sizeof(delimBuf));
					break;
				}
				else
					Pc_SetupReadLineCb(msgBuffer, 1, &bPcMessageFlag);
			 }
		} while (1);
	}

	//handle continuous measurement mode
	else {
		uint32_t nTotalSamples = 0;

		//interrupt-triggered DATAREADY notifications in this mode
		if(adi_sense_RegisterGpioCallback(hDevice, ADI_SENSE_GPIO_PIN_DATAREADY,
							    pc_data_ready_callback,
							    (void *)&bDataReady) != ADI_SENSE_SUCCESS)
		  return pc_handle_error(ERROR_GPIO_CALLBACK_REG);

		//begin measurement cycles
		if(adi_sense_StartMeasurement(hDevice, eMeasurementMode) != ADI_SENSE_SUCCESS)
		  return pc_handle_error(ERROR_START_MEASUREMENT);

		do {
			uint32_t nReturned;

			//Interrupt method: wait for DATAREADY interrupt callback
			while (!bDataReady);

			//reset bDataReady flag to detect the next callback
			bDataReady = false;
			    
			//read data into structure
			if(adi_sense_GetData(hDevice, eMeasurementMode, pSampleBuffer, nSamplesPerDataready, &nReturned) != ADI_SENSE_SUCCESS)
				return pc_handle_error(ERROR_GET_DATA);
			if(nReturned != nSamplesPerDataready)
				return pc_handle_error(ERROR_SAMPLE_COUNT_MISMATCH);

			//build a response string
			for(int i = 0; i < nReturned; i++) {
			    float32_t *sample = &pSampleBuffer[i].processedValue;
			    memcpy(respBuffer, (uint8_t*)sample, sizeof(float32_t));
			    Pc_Write(seperatorBuf, 1);
			    for(int j = 0; j < sizeof(float32_t); j++) {
				    char buf[3];
				    snprintf(buf, sizeof(buf), "%02X", respBuffer[j]);
				    Pc_Write(buf, 2);
			    }
			}

			//count samples
			nTotalSamples += nReturned;

			//check read byte to see if streaming should be stopped
			if(bPcMessageFlag) {
				if(msgBuffer[0] == COMMAND_STOP_STREAM) {
					Pc_Write(delimBuf, sizeof(delimBuf));
					break;
				}
				else
					Pc_SetupReadLineCb(msgBuffer, 1, &bPcMessageFlag);
			}
		} while (1);
			
		if(adi_sense_StopMeasurement(hDevice) != ADI_SENSE_SUCCESS)
		  return pc_handle_error(ERROR_STOP_MEASUREMENT);
	}
    
	Pc_ClearReadLineCb();

	delete [] pSampleBuffer;
    
    	//begin with status byte
	//format error code into chars
	snprintf(status, sizeof(status), "%02X", NO_ERROR);
	//send response
	Pc_Write(status, sizeof(status) - 1);
	
   	Pc_Write(delimBuf, sizeof(delimBuf));
    
    	return 0;
}

/***************************************************************
* Function Name: pc_parse_update_fw_version
* Description  : set bootloader flag and reset device into 
* 			mode expecting programming over serial.
****************************************************************/
int pc_parse_update_fw_version()
{	
	//set backup register flag to enter bootloader
	Rcc_WriteBackupReg(BOOTLOADER_FLAG_BACKUP_REG, 1);
	
	//perform soft reset
	return pc_parse_reset();
}

/***************************************************************
* Function Name: handle_error
* Description  : handle an error
****************************************************************/
int pc_handle_error(ErrorType error)
{	
	char status[3];
	//begin with status byte
	//format error code into chars
	snprintf(status, sizeof(status), "%02X", error);
	//send response
	Pc_Write(status, sizeof(status) - 1);
	
	Pc_Write(delimBuf, sizeof(delimBuf));

	return 1;
}

/***************************************************************
* Function Name: Pc_ParseCommand
* Description  : parse the configure json command
****************************************************************/
int Pc_ParseCommand( char *commandString )
{
	char c;
	bool flag;
	
	pc_read_next_arg(commandString, &c, 1, &flag);
	
	//if no command was found, enumIndex will equal -1 and therefore execute the default case. This handles the error
	switch(c) {	
	
		//pc is requesting the information packet about this device
	  case COMMAND_DEVICE_INFO:
	  	return pc_parse_device_info();

	  	//pc is requesting this device to flash it's led to identify itself
	  case COMMAND_FLASH_LED:
	  	return pc_parse_flash_led();
		
		//soft reset the system
	  case COMMAND_RESET:
	  	return pc_parse_reset();
		
		//configure the local reg map
	  case COMMAND_CONFIGURE:
	  	return pc_parse_configure(commandString);
		
		//apply configuration to device
	  case COMMAND_APPLY:
	  	return pc_parse_apply();
		
		//return a single instance of sampling to the pc
	  case COMMAND_SAMPLE:
	  	return pc_parse_sample(commandString);

	  	//begin streaming data to the pc, this locks out until the stopstream command is received
	  case COMMAND_START_STREAM:
	  	return pc_parse_start_stream();

	  case COMMAND_STOP_STREAM:
	  	return pc_handle_error(ERROR_STREAM_NOT_STARTED);
		
	  case COMMAND_UPDATE_FW_VERSION:
		return pc_parse_update_fw_version();

		  //Template
		  //case COMMAND:  return parse_COMMAND();
		  //parse_COMMAND handles the action required
		  //COMMAND must be defined within the header file

		//handle an unrecognised command/error
		  
	  default:
		  return pc_handle_error(ERROR_NO_COMMAND_MATCHED);
	}
}

