

/**
 * @file pc_interface.h
 * @author Jake Greaves
 * @date 21 Dec 2017
 * @brief The main interface to the pc application.
 *
 * This file handles all commands to and from the PC
 * application.
 */

#ifndef __PC_INTERFACE_H_
#define __PC_INTERFACE_H_

#include "string.h"
#include "adi_sense_api.h"
#include "adi_sense_1000/adi_sense_1000_api.h"
#include "pc_interface/pc_serial.h"
#include "pc_interface/pc_conversions.h"
#include "myproswift_periph.h"
#include "core_cm4.h"
#include "myproswift_error_codes.h"
#include "rcc_backup_registers/rcc_backup_registers.h"
#include "eeprom_virtual/eeprom.h"

/*
    Definitions
*/

#define RESPONSE_BUFFER_SIZE 	256
#define PC_DELIMITER		'\n'
#define PC_SEPERATOR		','

#undef DEBUG
#define BITP_VALID_DATA (7)

/*
    Command List
*/


#define COMMAND_DEVICE_INFO 		'i'
#define COMMAND_FLASH_LED 		'f'
#define COMMAND_RESET 			'r'
#define COMMAND_CONFIGURE 		'c'
#define COMMAND_APPLY			'u'
#define COMMAND_SAMPLE 			's'
#define COMMAND_START_STREAM 		'd'
#define COMMAND_STOP_STREAM 		'h'
#define COMMAND_UPDATE_FW_VERSION	'v'

//not sure why this is needed by the eeprom library
extern uint16_t VirtAddVarTab[NB_OF_VAR];

/*
    Function Prototypes
*/

/**
 * @brief Tell any connected device that device is ready to use
 *
 * @param bBootLoader If device is entering bootloader, a different success code is given.
 */
int Pc_ResetSuccess(bool_t bBootLoader);

/**
 * @brief Read the next arg from a string.
 *
 * This function scans through the given command string
 * searching for either the seperator value or the delimiter.
 * On finding either of these chars, all text before is returned
 * as the argument. If the size limit is reached before a delimiter
 * this functions returns an error. The flag variable is set when
 * the delimiter is found, useful for debugging and error handling
 * when more arguments might be expected.
 * 
 * @param command Command string to be analysed.
 * @param arg A found arg will be returned to this address.
 * @param size The max size of the arg buffer.
 * @param flag Set when delimiter is found.
 *
 * @return Returns 0 on success, 1 on error.
 *
 * @see PC_DELIMITER
 * @see PC_SEPERATOR
 */
static int pc_read_next_arg(char *command, char *arg, int size, bool *flag);

/**
 * @brief Parse the device info command.
 *
 * This function handles the device info command, responding to the PC
 * with the device name, battery voltage and firmware version.
 *
 * @return Returns 0 on success, 1 on error.
 */
static int pc_parse_device_info(void);

/**
 * @brief Parse the flash led command.
 *
 * This function handles the flash led command, pulsing an led
 * to identify this device. This function also responds to the PC
 * with the status code.
 *
 * @return Returns 0 on success, 1 on error.
 */
static int pc_parse_flash_led(void);

/**
 * @brief Reset the device.
 *
 * This function handles the reset command, performing a soft
 * reset on the device. The reponse to the pc is handled in the boot
 * sequence for this device.
 * 
 * @return Returns 0 on success, 1 on error.
 */
static int pc_parse_reset(void);

/**
 * @brief Parse the configure command.
 *
 * This function handles the configure command. Identifiers such as channel,
 * keyword and value are parsed and converted to struct values. The 
 * configure struct is updated and applied to the device.
 *
 * @param commandString Args are parsed from this string.
 * 
 * @return Returns 0 on success, 1 on error.
 */
static int pc_parse_configure(char *commandString);

/**
 * @brief Support for the parse configure command.
 *
 * If the configure command receives no channel, the configuration is for the 
 * entire device. This function parses and applies the new value to the 
 * configuration struct.
 *
 * @param attribute The attribute enum identifying which part of the struct 
 * is to be changed.
 *
 * @param commandString The command string incase arguments are expected.
 * 
 * @return Returns 0 on success, 1 on error.
 */
static int pc_parse_configure_device(CONFIG_ATTRIBUTE attribute, char *commandString);

/**
 * @brief Support for the parse configure command.
 *
 * If the configure command receives a channel, the configuration is channel 
 * specific. This function parses and applies the new value to the 
 * configuration struct.
 *
 * @param channelIndex Which channel is to be configured.
 * 
 * @param attribute The attribute enum identifying which part of the struct 
 * is to be changed.
 * 
 * @param commandString The command string incase arguments are expected.
 *
 * @return Returns 0 on success, 1 on error.
 */
static int pc_parse_configure_channel(ADI_SENSE_1000_CHANNEL_ID channelIndex, CONFIG_ATTRIBUTE attribute, char *commandString);

/**
 * @brief Callback for sampling. Drdy is tied to this.
 */
static void pc_data_ready_callback(
    	ADI_SENSE_GPIO_PIN ePinId,
    	void *pArg);

/**
 * @brief Parse the sample command.
 *
 * Obtain as many samples as required for the given channels and return the raw
 * 32-bit floats to the PC.
 *
 * @param commandString The amount of measurement cycles are obtained from this.
 * 
 * @return Returns 0 on success, 1 on error.
 */
static int pc_parse_sample(char *commandString);

/**
 * @brief Parse the start stream command.
 *
 * Begin the sample stream, pushing out samples as fast as possible to the PC.
 * 
 * @return Returns 0 on success, 1 on error.
 */
static int pc_parse_start_stream(void);

/**
 * @brief Handle an error.
 *
 * Send an error code to the PC.
 * 
 * @return Returns 0 on success, 1 on error.
 */
int pc_handle_error(ErrorType error);

/**
 * @brief Find which command to be executed.
 *
 * parse the command character and execute the appropriate command.
 * 
 * @param commandString the received command from the PC
 * 
 * @return Returns 0 on success, 1 on error.
 */
int Pc_ParseCommand( char *commandString );

#endif
